<?php
/**
 * Plugin Name: WP Static HTML Pages
 * Plugin URI: https://plugins.assertivlogix.com/plugin/wp-static-html-pages
 * Description: Generates static HTML for pages and serves them on request.
 * Version: 1.0.0
 * Author: Assertiv Logix
 * Author URI: https://assertivlogix.com
 * License: GPLv2 or later
 * License URI: http://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: wp-static-html-pages
 */

if (!defined('ABSPATH')) exit;

/**
 * Check if license is active. Plugin functionality runs only when this returns true.
 * Uses wpsh_license option (same pattern as Assertiv Logix 2FA plugin).
 */
function wpsh_is_license_active() {
    $opt = get_option('wpsh_license', []);
    if (empty($opt) || (isset($opt['status']) ? $opt['status'] : '') !== 'active') {
        return false;
    }
    $exp = isset($opt['expires_at']) ? (int) $opt['expires_at'] : 0;
    return $exp <= 0 || $exp > time();
}

/**
 * High-Performance Early Serving
 * This part runs as early as possible to bypass the database and core WP when possible.
 */
function wpsh_early_serve_static() {
    if (!wpsh_is_license_active()) return;
    if (get_option('wpsh_enabled', '0') !== '1') return;

    // Basic exclusions
    if (isset($_GET['s']) || isset($_GET['preview']) || isset($_GET['wpsh_msg']) || isset($_GET['wc-ajax']) || isset($_GET['add-to-cart'])) return;
    if (strpos($_SERVER['REQUEST_URI'], '/wp-admin/') !== false || strpos($_SERVER['REQUEST_URI'], '/wp-login.php') !== false) return;
    if ($_SERVER['REQUEST_METHOD'] !== 'GET') return;

    // Cookie check
    if (!empty($_COOKIE)) {
        foreach ($_COOKIE as $k => $v) {
            if (strpos($k, 'wordpress_logged_in_') === 0) return;
        }
    }

    $static_dir = WP_CONTENT_DIR . '/static-html';
    $raw_uri = strtok($_SERVER['REQUEST_URI'], '?');
    $uri = trim($raw_uri, '/');
    
    // Subdirectory install detection
    $script_dir = trim(dirname($_SERVER['SCRIPT_NAME']), '/\\');
    if (!empty($script_dir) && strpos($uri, $script_dir) === 0) {
        $uri = substr($uri, strlen($script_dir));
    }
    $clean_uri = trim($uri, '/');

    $filename = empty($clean_uri) ? 'index.html' : $clean_uri . '/index.html';
    $path = $static_dir . '/' . $filename;

    if (file_exists($path)) {
        $start = microtime(true);
        $mtime = filemtime($path);
        $etag = md5($mtime . filesize($path));
        
        header('Last-Modified: ' . gmdate('D, d M Y H:i:s', $mtime) . ' GMT');
        header('Etag: "' . $etag . '"');
        header('Cache-Control: public, max-age=3600');
        
        if ((isset($_SERVER['HTTP_IF_MODIFIED_SINCE']) && strtotime($_SERVER['HTTP_IF_MODIFIED_SINCE']) >= $mtime) || 
            (isset($_SERVER['HTTP_IF_NONE_MATCH']) && trim($_SERVER['HTTP_IF_NONE_MATCH'], '"') === $etag)) {
            header('HTTP/1.1 304 Not Modified');
            exit;
        }

        header('Content-Type: text/html; charset=UTF-8');
        header('X-WPSH-Served: Early (Static)');
        header('X-WPSH-Speed: ' . round((microtime(true) - $start) * 1000, 3) . 'ms');
        readfile($path);
        exit;
    }
}
add_action('plugins_loaded', 'wpsh_early_serve_static', 0);

/**
 * Plugin Bootstrap
 */
final class WPSH_Plugin {

    private static $instance = null;

    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct() {
        $this->define_constants();
        $this->includes();
        $this->init_hooks();
        $this->init_components();
    }

    private function define_constants() {
        define('WPSH_VERSION', '1.0.0');
        define('WPSH_PATH', plugin_dir_path(__FILE__));
        define('WPSH_URL', plugin_dir_url(__FILE__));
        define('WPSH_DIR', WP_CONTENT_DIR . '/static-html');
    }

    private function includes() {
        require_once WPSH_PATH . 'includes/class-wpsh-core.php';
        require_once WPSH_PATH . 'includes/class-wpsh-license.php';
        require_once WPSH_PATH . 'includes/class-wpsh-admin.php';
        require_once WPSH_PATH . 'includes/class-wpsh-ajax.php';
    }

    private function init_hooks() {
        add_action('init', [$this, 'load_textdomain']);
        add_action('template_redirect', [$this, 'on_visit_capture']);
        add_action('save_post', [$this, 'on_save_regenerate'], 20);
        register_deactivation_hook(__FILE__, [$this, 'deactivate']);
    }

    private function init_components() {
        // Initialize license manager (runs daily checks)
        WPSH_License::get_instance();
        
        if (is_admin()) {
            new WPSH_Admin();
        }
        if (defined('DOING_AJAX') && DOING_AJAX) {
            new WPSH_AJAX();
        }
    }

    public function load_textdomain() {
        load_plugin_textdomain('wp-static-html-pages', false, dirname(plugin_basename(__FILE__)) . '/languages');
    }

    /**
     * On-visit capture logic
     */
    public function on_visit_capture() {
        if (!wpsh_is_license_active()) return;
        if (get_option('wpsh_enabled', '0') !== '1') return;
        if (is_admin() || is_user_logged_in() || is_feed() || is_search() || is_404()) return;
        if (isset($_GET['add-to-cart']) || isset($_GET['wc-ajax']) || isset($_GET['wc-api'])) return;
        if (function_exists('is_cart') && (is_cart() || is_checkout() || is_account_page())) return;

        $core = WPSH_Core::get_instance();

        // Taxonomy Archive Check
        if (is_archive()) {
            $term = get_queried_object();
            if ($term && isset($term->taxonomy)) {
                // Check both global tax exclusion and individual term exclusion
                if ($core->is_excluded($term->term_id, 'term')) return;
            }
        }
        
        // Single Post Check
        if (is_singular()) {
            if ($core->is_excluded(get_the_ID(), 'post')) return;
        }

        // Global URL pattern exclusions (handles home etc)
        if ($core->is_excluded()) return;

        // Fetch Path
        global $wp;
        $current_url = home_url(add_query_arg([], $wp->request));
        if (substr($current_url, -1) !== '/') $current_url = trailingslashit($current_url);
        
        $path = $core->get_static_path_from_url($current_url);

        ob_start(function($buffer) use ($path) {
            if (empty($buffer) || stripos($buffer, '<html') === false) return $buffer;
            
            // Ensure dir exists
            wp_mkdir_p(dirname($path));
            file_put_contents($path, $buffer);
            return $buffer . "\n<!-- Served from dynamic source and cached by WP Static HTML -->";
        });
        header('X-WPSH-Served: Dynamic (Caching New)');
    }

    /**
     * Post-save regeneration logic
     */
    public function on_save_regenerate($post_id) {
        if (!wpsh_is_license_active()) return;
        if (get_option('wpsh_enabled', '0') !== '1') return;
        if (wp_is_post_revision($post_id) || wp_is_post_autosave($post_id)) return;
        if (get_option('wpsh_auto_regen', '1') !== '1') return;

        $core = WPSH_Core::get_instance();
        if ($core->is_excluded($post_id)) {
            $url = get_permalink($post_id);
            $path = $core->get_static_path_from_url($url);
            if ($path && file_exists($path)) unlink($path);
            return;
        }

        if (get_post_status($post_id) !== 'publish') {
            $url = get_permalink($post_id);
            $path = $core->get_static_path_from_url($url);
            if ($path && file_exists($path)) unlink($path);
            return;
        }

        $url = get_permalink($post_id);
        $core->generate_html_by_url($url);
        
        // Also regenerate home/blog
        if (get_post_type($post_id) === 'post') {
            $core->generate_html_by_url(home_url('/'));
            $posts_page_id = get_option('page_for_posts');
            if ($posts_page_id) $core->generate_html_by_url(get_permalink($posts_page_id));
        }
    }

    public function deactivate() {
        update_option('wpsh_enabled', '0');
        WPSH_Core::get_instance()->update_htaccess();
        
        // Clear scheduled license check
        $timestamp = wp_next_scheduled('wpsh_daily_license_check');
        if ($timestamp) {
            wp_unschedule_event($timestamp, 'wpsh_daily_license_check');
        }
    }
}

// Initialize the plugin
function wpsh_plugin_init() {
    return WPSH_Plugin::get_instance();
}
wpsh_plugin_init();
