<?php
/**
 * License Manager Class
 * Handles license activation, deactivation, and validation.
 * Uses Assertiv Logix API: https://plugins.assertivlogix.com/api/v1/validate
 */

if (!defined('ABSPATH')) exit;

class WPSH_License {

    private static $instance = null;
    private $api_url = 'https://plugins.assertivlogix.com/api/v1/validate';
    private $option_name = 'wpsh_license';
    private $product_id = 12; // WP Static HTML Pages

    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct() {
        if (!wp_next_scheduled('wpsh_daily_license_check')) {
            wp_schedule_event(time(), 'daily', 'wpsh_daily_license_check');
        }
        add_action('wpsh_daily_license_check', [$this, 'check_license_status']);
    }

    /**
     * Activate license via validate API (same pattern as Assertiv Logix 2FA plugin).
     *
     * @param string $license_key
     * @return array{success: bool, message: string, data?: array}|WP_Error
     */
    public function activate_license($license_key) {
        $response = wp_remote_post($this->api_url, [
            'body' => [
                'license_key' => $license_key,
                'site_url'    => site_url(),
                'product_id'  => $this->product_id,
            ],
            'timeout' => 15,
        ]);

        if (is_wp_error($response)) {
            return $response;
        }

        $response_code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);

        if ($response_code !== 200) {
            $message = isset($data['message']) ? $data['message'] : __('Error connecting to license server.', 'wp-static-html-pages');
            return new WP_Error('api_error', $message);
        }

        if ((isset($data['valid']) && $data['valid'] === true) || (isset($data['success']) && $data['success'] === true)) {
            $license_info = isset($data['data']) ? $data['data'] : $data;

            $expires_at = time() + (365 * 24 * 60 * 60);
            if (isset($license_info['expires_at'])) {
                $expires_at = is_numeric($license_info['expires_at']) ? (int) $license_info['expires_at'] : strtotime($license_info['expires_at']);
            } elseif (isset($license_info['expiry_date'])) {
                $expires_at = is_numeric($license_info['expiry_date']) ? (int) $license_info['expiry_date'] : strtotime($license_info['expiry_date']);
            }

            $subscription_data = [
                'status'       => isset($license_info['status']) ? $license_info['status'] : 'active',
                'type'         => 'paid',
                'license_key'  => sanitize_text_field($license_key),
                'product_id'   => isset($license_info['product_id']) ? (int) $license_info['product_id'] : $this->product_id,
                'activated_at' => time(),
                'expires_at'   => $expires_at,
            ];
            if (isset($license_info['customer_name'])) {
                $subscription_data['customer_name'] = $license_info['customer_name'];
            }
            if (isset($license_info['customer_email'])) {
                $subscription_data['customer_email'] = $license_info['customer_email'];
            }

            update_option($this->option_name, $subscription_data);
            return [
                'success' => true,
                'message' => __('License activated successfully!', 'wp-static-html-pages'),
                'data'    => $subscription_data,
            ];
        }

        $message = isset($data['message']) ? $data['message'] : __('Invalid license key.', 'wp-static-html-pages');
        return new WP_Error('invalid_key', $message);
    }

    /**
     * Deactivate license (clear local data only; same pattern as 2FA plugin).
     *
     * @return array{success: bool, message: string}
     */
    public function deactivate_license() {
        $stored = get_option($this->option_name, []);
        if (empty($stored) || empty($stored['license_key'])) {
            return ['success' => false, 'message' => __('No license key found.', 'wp-static-html-pages')];
        }

        delete_option($this->option_name);
        return [
            'success' => true,
            'message' => __('License deactivated successfully.', 'wp-static-html-pages'),
        ];
    }

    /**
     * Check license status via validate API (e.g. daily cron).
     */
    public function check_license_status() {
        $stored = get_option($this->option_name, []);
        $license_key = isset($stored['license_key']) ? $stored['license_key'] : '';
        if (empty($license_key)) {
            return;
        }

        $result = $this->activate_license($license_key);
        if (is_wp_error($result)) {
            delete_option($this->option_name);
        }
        // On success, activate_license already updates the option
    }

    /**
     * Get license information for UI.
     *
     * @return array{key: string, status: string, data: array, expires: string, customer_name: string, customer_email: string, activations_left: string}
     */
    public function get_license_info() {
        $stored = get_option($this->option_name, []);
        $license_key = isset($stored['license_key']) ? $stored['license_key'] : '';
        $status = isset($stored['status']) ? $stored['status'] : 'inactive';
        $expires_at = isset($stored['expires_at']) ? (int) $stored['expires_at'] : 0;

        if ($status === 'active' && $expires_at > 0 && $expires_at < time()) {
            $status = 'expired';
        }

        $expires = '';
        if ($expires_at > 0) {
            $ten_years = time() + (10 * 365 * 24 * 60 * 60);
            $expires = $expires_at >= $ten_years ? 'lifetime' : date('Y-m-d', $expires_at);
        }

        return [
            'key'              => $license_key,
            'status'           => $status,
            'data'             => $stored,
            'expires'          => $expires,
            'customer_name'    => isset($stored['customer_name']) ? $stored['customer_name'] : '',
            'customer_email'   => isset($stored['customer_email']) ? $stored['customer_email'] : '',
            'activations_left' => isset($stored['activations_left']) ? $stored['activations_left'] : '',
        ];
    }
}
