<?php
/**
 * Core Logic Class
 * Handles generation, early serving, and path resolution.
 */

if (!defined('ABSPATH')) exit;

class WPSH_Core {

    /**
     * Singleton instance
     */
    private static $instance = null;

    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct() {
        // Initialization
    }

    /**
     * Get the static storage directory path
     */
    public function get_storage_dir() {
        return WP_CONTENT_DIR . '/static-html';
    }

    /**
     * Get the static storage directory URL
     */
    public function get_storage_url() {
        return content_url('/static-html');
    }

    /**
     * Resolve a URL to its expected static file path
     */
    public function get_static_path_from_url($url) {
        if (empty($url)) return false;
        
        $path = parse_url($url, PHP_URL_PATH);
        
        // Normalize path relative to WP install
        $wp_path = parse_url(home_url('/'), PHP_URL_PATH);
        if ($wp_path && $wp_path !== '/') {
            $path = preg_replace('#^' . preg_quote($wp_path, '#') . '#', '', $path);
        }
        
        $path = trim($path, '/');
        $base_dir = $this->get_storage_dir();
        
        if (empty($path)) {
            return $base_dir . '/index.html';
        } else {
            return $base_dir . '/' . $path . '/index.html';
        }
    }

    /**
     * Generate static HTML by URL
     */
    public function generate_html_by_url($url) {
        if (!$url) return __('Empty URL', 'wp-static-html-pages');

        $response = wp_remote_get($url, [
            'timeout' => 30,
            'sslverify' => false,
            'cookies' => []
        ]);

        if (is_wp_error($response)) return $response->get_error_message();
        $code = wp_remote_retrieve_response_code($response);
        if ($code !== 200) return sprintf(__('HTTP Error %d', 'wp-static-html-pages'), $code);

        $html = wp_remote_retrieve_body($response);
        if (empty($html)) return __('Empty Response Body', 'wp-static-html-pages');
        
        $path = $this->get_static_path_from_url($url);
        if (!$path) return __('Invalid Path', 'wp-static-html-pages');
        
        // Ensure parent directory exists
        $dir = dirname($path);
        if (!file_exists($dir)) {
            if (!wp_mkdir_p($dir)) return sprintf(__('Could not create directory: %s', 'wp-static-html-pages'), $dir);
        }
        
        if (file_put_contents($path, $html) === false) {
            return sprintf(__('Could not write file: %s', 'wp-static-html-pages'), $path);
        }
        
        return true;
    }

    /**
     * Get effective excluded post types: post/page never excluded, attachment always excluded.
     * When saved option is empty, defaults to excluding all public types except post and page.
     */
    public function get_effective_excluded_post_types() {
        $raw = get_option('wpsh_excluded_post_types', []);
        $all = get_post_types(['public' => true]);
        $never = ['post', 'page'];
        $always = ['attachment'];
        if (empty($raw)) {
            return array_values(array_diff($all, $never));
        }
        return array_values(array_diff(array_unique(array_merge($raw, $always)), $never));
    }

    /**
     * Check if a page/post/taxonomy is excluded
     */
    public function is_excluded($id = false, $type = 'post') {
        // 0. Check Global URL Exclusions
        $url = false;
        if ($id) {
            $url = ($type === 'post') ? get_permalink($id) : get_term_link($id);
        }
        
        // If no URL yet and not admin, check current request
        if (!$url && !is_admin()) {
            global $wp;
            $url = home_url(add_query_arg([], $wp->request));
        }

        if ($url && !is_wp_error($url)) {
            $excluded_patterns = get_option('wpsh_url_exclusions', '');
            if (!empty($excluded_patterns)) {
                $patterns = array_filter(array_map('trim', explode("\n", $excluded_patterns)));
                foreach ($patterns as $pattern) {
                    if (strpos($url, $pattern) !== false) return true;
                }
            }
        }

        if (!$id) return false;
        
        if ($type === 'post') {
            // 1. Check Global Post Type Exclusion (post/page never excluded, attachment always excluded)
            $post_type = get_post_type($id);
            $excluded_pts = $this->get_effective_excluded_post_types();
            if (in_array($post_type, $excluded_pts)) return true;
            
            // 2. Check Individual Post Exclusion
            return get_post_meta($id, '_wpsh_exclude', true) === '1';
        } else {
            // 1. Check Global Taxonomy Exclusion
            $term = get_term($id);
            if ($term && !is_wp_error($term)) {
                if ($this->is_tax_excluded($term->taxonomy)) return true;
                
                // 2. Check Individual Term Exclusion
                return get_term_meta($id, '_wpsh_exclude', true) === '1';
            }
        }

        return false;
    }

    /**
     * Get effective excluded taxonomies. When saved option is empty, all public taxonomies are excluded by default.
     */
    public function get_effective_excluded_taxonomies() {
        $raw = get_option('wpsh_excluded_taxonomies', []);
        if (empty($raw)) {
            return get_taxonomies(['public' => true]);
        }
        return $raw;
    }

    /**
     * Check if a taxonomy is excluded
     */
    public function is_tax_excluded($taxonomy) {
        $excluded_taxes = $this->get_effective_excluded_taxonomies();
        return in_array($taxonomy, $excluded_taxes);
    }

    /**
     * Build the .htaccess rules block (for display or writing). Returns the full block including markers.
     */
    public function get_htaccess_rules() {
        $marker_start = '# BEGIN WP Static HTML Pages';
        $marker_end = '# END WP Static HTML Pages';

        $base_raw = parse_url(home_url('/'), PHP_URL_PATH);
        $base = '/' . trim($base_raw, '/') . '/';
        if ($base === '//') $base = '/';

        $content_dir = str_replace(ABSPATH, '', WP_CONTENT_DIR);
        $static_rel_path = trim($content_dir, '/') . '/static-html';

        $rules = $marker_start . "\n";
        if (get_option('wpsh_enabled', '0') === '1') {
            $rules .= "<IfModule mod_rewrite.c>\n";
            $rules .= "RewriteEngine On\n";
            $rules .= "RewriteBase $base\n";
            $rules .= "RewriteCond %{REQUEST_METHOD} !GET [OR]\n";
            $rules .= "RewriteCond %{QUERY_STRING} (?:^|&)(s|wc-ajax|add-to-cart|wc-api)= [NC,OR]\n";
            $rules .= "RewriteCond %{HTTP:Cookie} wordpress_logged_in_ [NC]\n";
            $rules .= "RewriteRule .* - [S=2]\n";

            $excluded_patterns = get_option('wpsh_url_exclusions', '');
            if (!empty($excluded_patterns)) {
                $patterns = array_filter(array_map('trim', explode("\n", $excluded_patterns)));
                foreach ($patterns as $pattern) {
                    $p = str_replace(' ', '\ ', $pattern);
                    $rules .= "RewriteCond %{REQUEST_URI} !" . $p . "\n";
                }
            }

            $rules .= "RewriteCond %{REQUEST_URI} ^$base" . "?$\n";
            $rules .= "RewriteCond %{DOCUMENT_ROOT}$base$static_rel_path/index.html -f\n";
            $rules .= "RewriteRule ^$ $static_rel_path/index.html [L]\n";
            $rules .= "RewriteCond %{REQUEST_URI} ^$base" . "(.*)/?$\n";
            $rules .= "RewriteCond %{DOCUMENT_ROOT}$base$static_rel_path/%1/index.html -f\n";
            $rules .= "RewriteRule ^(.*)$ $static_rel_path/$1/index.html [L]\n";
            $rules .= "</IfModule>\n";
        }
        $rules .= $marker_end;
        return $rules;
    }

    /**
     * Update .htaccess rules
     */
    public function update_htaccess() {
        $htaccess_path = ABSPATH . '.htaccess';
        if (!file_exists($htaccess_path) || !is_writable($htaccess_path)) return false;

        $content = file_get_contents($htaccess_path);
        $marker_start = '# BEGIN WP Static HTML Pages';
        $marker_end = '# END WP Static HTML Pages';
        $rules = "\n" . $this->get_htaccess_rules() . "\n";

        if (strpos($content, $marker_start) !== false) {
            $content = preg_replace('/' . preg_quote($marker_start, '/') . '.*?' . preg_quote($marker_end, '/') . '/s', '', $content);
        }
        $content = $rules . "\n" . trim($content);
        return file_put_contents($htaccess_path, $content) !== false;
    }
}
