<?php
/**
 * AJAX Handlers Class
 * Encapsulates all backend AJAX functionality.
 */

if (!defined('ABSPATH')) exit;

class WPSH_AJAX {

    private $core;

    public function __construct() {
        $this->core = WPSH_Core::get_instance();

        // Register AJAX actions
        add_action('wp_ajax_wpsh_generate_all', [$this, 'handle_batch_generation']);
        add_action('wp_ajax_wpsh_clear_cache', [$this, 'handle_clear_cache']);
        add_action('wp_ajax_wpsh_toggle_plugin', [$this, 'handle_plugin_toggle']);
        add_action('wp_ajax_wpsh_save_pt_exclusion', [$this, 'handle_pt_exclusion']);
        add_action('wp_ajax_wpsh_save_tax_exclusion', [$this, 'handle_tax_exclusion']);
        add_action('wp_ajax_wpsh_save_global_setting', [$this, 'handle_global_setting']);
        add_action('wp_ajax_wpsh_update_htaccess', [$this, 'handle_htaccess_update']);
        add_action('wp_ajax_wpsh_generate_single', [$this, 'handle_single_generation']);
        add_action('wp_ajax_wpsh_activate_license', [$this, 'handle_activate_license']);
        add_action('wp_ajax_wpsh_deactivate_license', [$this, 'handle_deactivate_license']);
        add_action('wp_ajax_wpsh_check_license', [$this, 'handle_check_license']);
    }

    public function handle_single_generation() {
        $post_id = isset($_GET['post_id']) ? intval($_GET['post_id']) : 0;
        $term_id = isset($_GET['term_id']) ? intval($_GET['term_id']) : 0;
        $tax     = isset($_GET['taxonomy']) ? sanitize_key($_GET['taxonomy']) : '';

        if ($post_id) {
            check_ajax_referer('wpsh_single_' . $post_id, 'nonce');
            $url = get_permalink($post_id);
        } elseif ($term_id && $tax) {
            check_ajax_referer('wpsh_term_' . $term_id, 'nonce');
            $url = get_term_link($term_id, $tax);
        } else {
            wp_die(__('Invalid request', 'wp-static-html-pages'));
        }
        
        if (!current_user_can('manage_options')) wp_die(__('Permission denied', 'wp-static-html-pages'));
        if (!wpsh_is_license_active()) {
            $redirect_url = add_query_arg(['wpsh_msg' => 'license_required'], wp_get_referer());
            wp_safe_redirect($redirect_url);
            exit;
        }
        if (is_wp_error($url)) wp_die($url->get_error_message());

        $res = $this->core->generate_html_by_url($url);
        
        $redirect_url = add_query_arg(['wpsh_msg' => 'regenerated'], wp_get_referer());
        wp_safe_redirect($redirect_url);
        exit;
    }

    public function handle_batch_generation() {
        check_ajax_referer('wpsh_generate_all', 'nonce');
        if (!wpsh_is_license_active()) wp_send_json_error(__('Please activate your license first.', 'wp-static-html-pages'));
        if (get_option('wpsh_enabled', '0') !== '1') wp_send_json_error(__('Plugin disabled', 'wp-static-html-pages'));
        if (!current_user_can('manage_options')) wp_send_json_error(__('Permission denied', 'wp-static-html-pages'));
        
        $offset = isset($_POST['offset']) ? intval($_POST['offset']) : 0;
        $limit = 10;
        $processed = [];

        // 1. Get Setup Data (post/page always included, attachment always excluded)
        $excluded_pts = $this->core->get_effective_excluded_post_types();
        $target_pts = array_diff(get_post_types(['public' => true]), $excluded_pts);
        
        $excluded_taxes = $this->core->get_effective_excluded_taxonomies();
        $target_taxes = array_diff(get_taxonomies(['public' => true]), $excluded_taxes);

        $special_urls = [home_url('/')];
        $posts_page_id = get_option('page_for_posts');
        if ($posts_page_id) $special_urls[] = get_permalink($posts_page_id);
        $special_urls = array_unique($special_urls);
        $num_special = count($special_urls);

        $total_posts = 0;
        foreach ($target_pts as $pt) $total_posts += intval(wp_count_posts($pt)->publish);

        $total_terms = 0;
        foreach ($target_taxes as $tax) $total_terms += intval(wp_count_terms(['taxonomy' => $tax, 'hide_empty' => false]));

        if ($offset === 0) {
            $total_to_process = $num_special + $total_posts + $total_terms;
        }

        $batch_processed_count = 0;

        // Phase 1: Special URLs
        if ($offset < $num_special) {
            $remaining = array_slice($special_urls, $offset, $limit);
            foreach ($remaining as $url) {
                $res = $this->core->generate_html_by_url($url);
                $processed[] = ($res === true ? 'Special: ' : 'FAILED Special: ') . $url;
                $batch_processed_count++;
                if ($batch_processed_count >= $limit) break;
            }
        }

        // Phase 2: Posts
        if ($batch_processed_count < $limit && ($offset + $batch_processed_count) < ($num_special + $total_posts)) {
            $post_offset = max(0, $offset + $batch_processed_count - $num_special);
            $post_limit = $limit - $batch_processed_count;
            
            $posts = get_posts([
                'post_type'      => $target_pts,
                'post_status'    => 'publish',
                'posts_per_page' => $post_limit,
                'offset'         => $post_offset,
                'fields'         => 'ids',
                'orderby'        => 'ID',
                'order'          => 'ASC'
            ]);

            foreach ($posts as $pid) {
                if ($this->core->is_excluded($pid)) {
                    $url = get_permalink($pid);
                    $path = $this->core->get_static_path_from_url($url);
                    if ($path && file_exists($path)) unlink($path);
                    $processed[] = 'Skipped (Excluded): ' . get_the_title($pid);
                } else {
                    $url = get_permalink($pid);
                    $res = $this->core->generate_html_by_url($url);
                    $processed[] = ($res === true ? '' : 'FAILED ') . get_the_title($pid) . ' (' . $url . ')';
                }
                $batch_processed_count++;
                if ($batch_processed_count >= $limit) break;
            }
        }

        // Phase 3: Taxonomies
        if ($batch_processed_count < $limit && ($offset + $batch_processed_count) < ($num_special + $total_posts + $total_terms)) {
            $term_offset = max(0, $offset + $batch_processed_count - $num_special - $total_posts);
            $term_limit = $limit - $batch_processed_count;

            $terms = get_terms([
                'taxonomy'   => $target_taxes,
                'hide_empty' => false,
                'number'     => $term_limit,
                'offset'     => $term_offset
            ]);

            if (!is_wp_error($terms) && !empty($terms)) {
                foreach ($terms as $term) {
                    $url = get_term_link($term);
                    if (is_wp_error($url)) continue;
                    $res = $this->core->generate_html_by_url($url);
                    $processed[] = ($res === true ? 'Archive: ' : 'FAILED Archive: ') . $term->name . ' (' . $url . ')';
                    $batch_processed_count++;
                    if ($batch_processed_count >= $limit) break;
                }
            }
        }

        $is_complete = ($offset + $batch_processed_count) >= ($num_special + $total_posts + $total_terms);

        wp_send_json_success([
            'complete'           => $is_complete,
            'next_offset'        => $offset + $batch_processed_count,
            'batch_count'        => $batch_processed_count,
            'processed'          => $processed,
            'total_to_process'   => ($offset === 0) ? $total_to_process : 0
        ]);
    }

    public function handle_clear_cache() {
        check_ajax_referer('wpsh_clear_cache', 'nonce');
        if (!wpsh_is_license_active()) wp_send_json_error(__('Please activate your license first.', 'wp-static-html-pages'));
        if (!current_user_can('manage_options')) wp_send_json_error(__('Permission denied', 'wp-static-html-pages'));

        $static_dir = $this->core->get_storage_dir();
        if (file_exists($static_dir)) {
            $this->rmdir_recursive($static_dir);
            wp_mkdir_p($static_dir);
        }
        wp_send_json_success();
    }

    private function rmdir_recursive($dir) {
        $files = array_diff(scandir($dir), array('.','..'));
        foreach ($files as $file) {
            (is_dir("$dir/$file")) ? $this->rmdir_recursive("$dir/$file") : unlink("$dir/$file");
        }
        return rmdir($dir);
    }

    public function handle_plugin_toggle() {
        check_ajax_referer('wpsh_toggle_plugin', 'nonce');
        if (!current_user_can('manage_options')) wp_send_json_error(__('Permission denied', 'wp-static-html-pages'));
        if (!wpsh_is_license_active()) wp_send_json_error(__('Please activate your license first.', 'wp-static-html-pages'));
        
        $enabled = isset($_POST['enabled']) ? ($_POST['enabled'] === 'true' ? '1' : '0') : '1';
        update_option('wpsh_enabled', $enabled);
        $this->core->update_htaccess();
        
        wp_send_json_success(['enabled' => $enabled]);
    }

    public function handle_pt_exclusion() {
        check_ajax_referer('wpsh_pt_exclusion', 'nonce');
        if (!current_user_can('manage_options')) wp_send_json_error(__('Permission denied', 'wp-static-html-pages'));
        if (!wpsh_is_license_active()) wp_send_json_error(__('Please activate your license first.', 'wp-static-html-pages'));
        
        $pt = isset($_POST['post_type']) ? sanitize_key($_POST['post_type']) : '';
        $excluded = isset($_POST['excluded']) && ($_POST['excluded'] === 'true' || $_POST['excluded'] === true);
        
        if (!$pt) wp_send_json_error(__('Invalid post type', 'wp-static-html-pages'));
        if (in_array($pt, ['post', 'page', 'attachment'], true)) {
            wp_send_json_error(__('This post type cannot be toggled.', 'wp-static-html-pages'));
        }
        
        $raw = get_option('wpsh_excluded_post_types', []);
        $all = get_post_types(['public' => true]);
        $never = ['post', 'page'];
        $always = ['attachment'];
        $effective = empty($raw)
            ? array_values(array_diff($all, $never))
            : array_values(array_diff(array_unique(array_merge($raw, $always)), $never));
        
        if ($excluded) {
            if (!in_array($pt, $effective)) $effective[] = $pt;
        } else {
            $effective = array_values(array_diff($effective, [$pt]));
        }
        $to_save = array_values(array_diff($effective, ['attachment']));
        update_option('wpsh_excluded_post_types', $to_save);
        wp_send_json_success();
    }

    public function handle_tax_exclusion() {
        check_ajax_referer('wpsh_tax_exclusion', 'nonce');
        if (!current_user_can('manage_options')) wp_send_json_error(__('Permission denied', 'wp-static-html-pages'));
        if (!wpsh_is_license_active()) wp_send_json_error(__('Please activate your license first.', 'wp-static-html-pages'));
        
        $tax = isset($_POST['taxonomy']) ? sanitize_key($_POST['taxonomy']) : '';
        $excluded = isset($_POST['excluded']) && ($_POST['excluded'] === 'true' || $_POST['excluded'] === true);
        
        if (!$tax) wp_send_json_error(__('Invalid taxonomy', 'wp-static-html-pages'));
        
        $raw = get_option('wpsh_excluded_taxonomies', []);
        $all = get_taxonomies(['public' => true]);
        $effective = empty($raw) ? $all : $raw;
        
        if ($excluded) {
            if (!in_array($tax, $effective)) $effective[] = $tax;
        } else {
            $effective = array_diff($effective, [$tax]);
        }
        update_option('wpsh_excluded_taxonomies', array_values($effective));
        wp_send_json_success();
    }

    public function handle_global_setting() {
        check_ajax_referer('wpsh_settings_nonce', 'nonce');
        if (!current_user_can('manage_options')) wp_send_json_error(__('Permission denied', 'wp-static-html-pages'));
        if (!wpsh_is_license_active()) wp_send_json_error(__('Please activate your license first.', 'wp-static-html-pages'));
        
        $setting = isset($_POST['setting']) ? sanitize_key($_POST['setting']) : '';
        $value = isset($_POST['value']) ? $_POST['value'] : '';
        
        if (in_array($setting, ['wpsh_auto_regen', 'wpsh_url_exclusions'])) {
            update_option($setting, $value);
            if ($setting === 'wpsh_url_exclusions') {
                $this->core->update_htaccess();
            }
            wp_send_json_success();
        }
        
        wp_send_json_error(__('Invalid setting', 'wp-static-html-pages'));
    }

    public function handle_htaccess_update() {
        check_ajax_referer('wpsh_update_htaccess', 'nonce');
        if (!current_user_can('manage_options')) wp_send_json_error(__('Permission denied', 'wp-static-html-pages'));
        if (!wpsh_is_license_active()) wp_send_json_error(__('Please activate your license first.', 'wp-static-html-pages'));
        
        if ($this->core->update_htaccess()) {
            wp_send_json_success(__('.htaccess updated', 'wp-static-html-pages'));
        } else {
            wp_send_json_error(__('Failed to update .htaccess. Make sure it is writable.', 'wp-static-html-pages'));
        }
    }

    public function handle_activate_license() {
        check_ajax_referer('wpsh_license', 'nonce');
        if (!current_user_can('manage_options')) wp_send_json_error(__('Permission denied', 'wp-static-html-pages'));
        
        $license_key = isset($_POST['license_key']) ? sanitize_text_field(trim($_POST['license_key'])) : '';
        
        if (empty($license_key)) {
            wp_send_json_error(__('Please enter a license key.', 'wp-static-html-pages'));
        }
        
        $license = WPSH_License::get_instance();
        $result = $license->activate_license($license_key);
        
        if (is_wp_error($result)) {
            wp_send_json_error($result->get_error_message());
        }
        if (!empty($result['success'])) {
            wp_send_json_success([
                'message' => $result['message'],
                'status'  => 'active',
                'data'    => isset($result['data']) ? $result['data'] : [],
            ]);
        }
        wp_send_json_error($result['message'] ?? __('License activation failed.', 'wp-static-html-pages'));
    }

    public function handle_deactivate_license() {
        check_ajax_referer('wpsh_license', 'nonce');
        if (!current_user_can('manage_options')) wp_send_json_error(__('Permission denied', 'wp-static-html-pages'));
        
        $license = WPSH_License::get_instance();
        $result = $license->deactivate_license();
        
        if ($result['success']) {
            wp_send_json_success([
                'message' => $result['message'],
                'status' => 'inactive'
            ]);
        } else {
            wp_send_json_error($result['message']);
        }
    }

    public function handle_check_license() {
        check_ajax_referer('wpsh_license', 'nonce');
        if (!current_user_can('manage_options')) wp_send_json_error(__('Permission denied', 'wp-static-html-pages'));
        
        $license = WPSH_License::get_instance();
        $license->check_license_status();
        $license_info = $license->get_license_info();
        
        if ($license_info['status'] === 'active') {
            wp_send_json_success([
                'message' => __('License is active and valid.', 'wp-static-html-pages'),
                'status'  => 'active',
                'data'    => $license_info,
            ]);
        }
        wp_send_json_error(__('License is not active.', 'wp-static-html-pages'));
    }
}
