<?php
/**
 * Admin Interface Class
 * Handles menus, settings rendering, and meta boxes.
 */

if (!defined('ABSPATH')) exit;

class WPSH_Admin {

    private $core;

    public function __construct() {
        $this->core = WPSH_Core::get_instance();
        
        add_action('admin_menu', [$this, 'add_admin_menu']);
        add_action('add_meta_boxes', [$this, 'add_post_meta_boxes']);
        add_action('save_post', [$this, 'save_post_meta'], 10);
        add_action('admin_notices', [$this, 'display_notices']);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_assets']);
        
        // UI Integrations
        add_action('init', [$this, 'init_ui_integrations'], 20);
        add_action('admin_bar_menu', [$this, 'add_admin_bar_menu'], 100);
    }

    public function init_ui_integrations() {
        $public_types = get_post_types(['public' => true]);
        foreach ($public_types as $type) {
            add_filter("manage_{$type}_posts_columns", [$this, 'add_static_column']);
            add_action("manage_{$type}_posts_custom_column", [$this, 'render_static_column'], 10, 2);
            add_filter("post_row_actions", [$this, 'add_row_actions'], 10, 2);
            add_filter("page_row_actions", [$this, 'add_row_actions'], 10, 2);
        }

        $public_taxes = get_taxonomies(['public' => true]);
        foreach ($public_taxes as $tax) {
            add_filter("manage_edit-{$tax}_columns", [$this, 'add_static_column']);
            add_filter("manage_{$tax}_custom_column", [$this, 'render_taxonomy_column'], 10, 3);
            add_filter("{$tax}_row_actions", [$this, 'add_taxonomy_row_actions'], 10, 2);
            
            // Term Metadata Fields (Edit screens)
            add_action("{$tax}_edit_form_fields", [$this, 'render_term_meta_fields'], 10, 2);
            add_action("edited_{$tax}", [$this, 'save_term_meta'], 10, 2);

            // Term Metadata Fields (Add screens)
            add_action("{$tax}_add_form_fields", [$this, 'render_new_term_meta_fields'], 10);
            add_action("created_{$tax}", [$this, 'save_term_meta'], 10, 2);
        }
    }

    public function add_static_column($columns) {
        $new_columns = [];
        foreach ($columns as $key => $value) {
            if ($key === 'date') {
                $new_columns['wpsh_status'] = __('Static HTML', 'wp-static-html-pages');
            }
            $new_columns[$key] = $value;
        }
        return $new_columns;
    }

    public function render_static_column($column, $post_id) {
        if ($column !== 'wpsh_status') return;

        $url = get_permalink($post_id);
        $path = $this->core->get_static_path_from_url($url);

        if ($path && file_exists($path)) {
            $mtime = filemtime($path);
            echo sprintf(
                '<span class="wpsh-badge wpsh-badge-success" title="%s">%s<br><small>%s</small></span>',
                esc_attr__('Static file exists', 'wp-static-html-pages'),
                __('Generated', 'wp-static-html-pages'),
                date_i18n(get_option('date_format') . ' ' . get_option('time_format'), $mtime)
            );
        } else {
            echo '<span class="wpsh-badge wpsh-badge-none">' . __('Not Cached', 'wp-static-html-pages') . '</span>';
        }
    }

    public function add_row_actions($actions, $post) {
        if (!current_user_can('manage_options')) return $actions;
        
        $url = admin_url('admin-ajax.php?action=wpsh_generate_single&post_id=' . $post->ID . '&nonce=' . wp_create_nonce('wpsh_single_' . $post->ID));
        $actions['wpsh_regen'] = sprintf(
            '<a href="%s" style="color: #3498db; font-weight: bold;">%s</a>',
            $url,
            __('Regenerate Static', 'wp-static-html-pages')
        );
        return $actions;
    }

    public function add_admin_bar_menu($wp_admin_bar) {
        if (!current_user_can('manage_options')) return;
        $wp_admin_bar->add_node([
            'id'    => 'wpsh-root',
            'title' => '<span class="ab-icon dashicons-html"></span> WP Static',
            'href'  => admin_url('admin.php?page=wp-static-html-pages'),
        ]);
        $wp_admin_bar->add_node([
            'parent' => 'wpsh-root',
            'id'     => 'wpsh-dashboard',
            'title'  => __('Dashboard', 'wp-static-html-pages'),
            'href'   => admin_url('admin.php?page=wp-static-html-pages#dashboard'),
        ]);
    }

    public function enqueue_assets($hook) {
        if ('toplevel_page_wp-static-html-pages' !== $hook) return;

        wp_enqueue_style('wpsh-admin-css', plugins_url('../assets/css/admin.css', __FILE__), [], '1.0.1');
        wp_enqueue_script('wpsh-admin-js', plugins_url('../assets/js/admin.js', __FILE__), ['jquery'], '1.0.1', true);

        wp_localize_script('wpsh-admin-js', 'wpshData', [
            'nonces' => [
                'toggle'        => wp_create_nonce('wpsh_toggle_plugin'),
                'generate'      => wp_create_nonce('wpsh_generate_all'),
                'clear'         => wp_create_nonce('wpsh_clear_cache'),
                'pt_exclusion'  => wp_create_nonce('wpsh_pt_exclusion'),
                'tax_exclusion' => wp_create_nonce('wpsh_tax_exclusion'),
                'settings'      => wp_create_nonce('wpsh_settings_nonce'),
                'htaccess'      => wp_create_nonce('wpsh_update_htaccess'),
                'license'       => wp_create_nonce('wpsh_license'),
            ],
            'strings' => [
                'enabled'           => __('Plugin Enabled', 'wp-static-html-pages'),
                'disabled'          => __('Plugin Disabled', 'wp-static-html-pages'),
                'confirmClear'      => __('Are you sure you want to delete all static files?', 'wp-static-html-pages'),
                'cacheCleared'      => __('Cache cleared successfully!', 'wp-static-html-pages'),
                'enableToAccess'       => __('Enable the plugin to access Tools and Settings.', 'wp-static-html-pages'),
                'activateLicenseToAccess' => __('Activate your license in the Licence tab to access Tools and Settings.', 'wp-static-html-pages'),
                'copiedToClipboard'    => __('Copied to clipboard.', 'wp-static-html-pages'),
            ]
        ]);
    }

    public function add_admin_menu() {
        add_menu_page(
            __('WP Static HTML', 'wp-static-html-pages'),
            __('WP Static', 'wp-static-html-pages'),
            'manage_options',
            'wp-static-html-pages',
            [$this, 'render_admin_page'],
            'dashicons-html',
            100
        );
    }

    public function add_post_meta_boxes() {
        $public_types = get_post_types(['public' => true]);
        foreach ($public_types as $type) {
            add_meta_box(
                'wpsh_settings',
                __('Static HTML Settings', 'wp-static-html-pages'),
                [$this, 'render_meta_box'],
                $type,
                'side'
            );
        }
    }

    public function render_meta_box($post) {
        wp_nonce_field('wpsh_meta_save', 'wpsh_meta_nonce');
        $val = get_post_meta($post->ID, '_wpsh_exclude', true);
        ?>
        <p>
            <label>
                <input type="checkbox" name="wpsh_exclude" value="1" <?php checked($val, '1'); ?>> 
                <?php _e('Exclude from Static HTML Cache', 'wp-static-html-pages'); ?>
            </label>
        </p>
        <p class="description">
            <?php _e('If checked, this page will not be cached as static HTML and will always be served dynamically.', 'wp-static-html-pages'); ?>
        </p>
        <?php
    }

    public function save_post_meta($post_id) {
        if (!isset($_POST['wpsh_meta_nonce']) || !wp_verify_nonce($_POST['wpsh_meta_nonce'], 'wpsh_meta_save')) return;
        if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) return;
        if (!current_user_can('edit_post', $post_id)) return;

        $exclude = isset($_POST['wpsh_exclude']) ? '1' : '0';
        update_post_meta($post_id, '_wpsh_exclude', $exclude);
        
        if ($exclude === '1') {
            $url = get_permalink($post_id);
            $path = $this->core->get_static_path_from_url($url);
            if ($path && file_exists($path)) unlink($path);
        }
    }

    public function display_notices() {
        if (isset($_GET['wpsh_msg']) && $_GET['wpsh_msg'] === 'regenerated') {
            echo '<div class="notice notice-success is-dismissible"><p>' . __('Static HTML has been regenerated successfully.', 'wp-static-html-pages') . '</p></div>';
        }
        if (isset($_GET['wpsh_msg']) && $_GET['wpsh_msg'] === 'license_required') {
            echo '<div class="notice notice-warning is-dismissible"><p>' . __('Please activate your license in the Licence tab to use this feature.', 'wp-static-html-pages') . '</p></div>';
        }
    }

    public function render_taxonomy_column($content, $column, $term_id) {
        if ($column !== 'wpsh_status') return $content;

        $term = get_term($term_id);
        if (!$term || is_wp_error($term)) return $content;

        $url = get_term_link($term);
        if (is_wp_error($url)) return $content;

        $path = $this->core->get_static_path_from_url($url);

        if ($path && file_exists($path)) {
            $mtime = filemtime($path);
            return sprintf(
                '<span class="wpsh-badge wpsh-badge-success" title="%s">%s<br><small>%s</small></span>',
                esc_attr__('Static file exists', 'wp-static-html-pages'),
                __('Generated', 'wp-static-html-pages'),
                date_i18n(get_option('date_format') . ' ' . get_option('time_format'), $mtime)
            );
        } else {
            return '<span class="wpsh-badge wpsh-badge-none">' . __('Not Cached', 'wp-static-html-pages') . '</span>';
        }
    }

    public function add_taxonomy_row_actions($actions, $term) {
        if (!current_user_can('manage_options')) return $actions;
        
        $url = admin_url('admin-ajax.php?action=wpsh_generate_single&term_id=' . $term->term_id . '&taxonomy=' . $term->taxonomy . '&nonce=' . wp_create_nonce('wpsh_term_' . $term->term_id));
        $actions['wpsh_regen'] = sprintf(
            '<a href="%s" style="color: #3498db; font-weight: bold;">%s</a>',
            $url,
            __('Regenerate Static', 'wp-static-html-pages')
        );
        return $actions;
    }

    public function render_term_meta_fields($term, $taxonomy) {
        $val = get_term_meta($term->term_id, '_wpsh_exclude', true);
        ?>
        <tr class="form-field">
            <th scope="row" valign="top"><label><?php _e('Static HTML Settings', 'wp-static-html-pages'); ?></label></th>
            <td>
                <label>
                    <input type="checkbox" name="wpsh_exclude" value="1" <?php checked($val, '1'); ?>> 
                    <?php _e('Exclude from Static HTML Cache', 'wp-static-html-pages'); ?>
                </label>
                <p class="description"><?php _e('If checked, this archive page will not be cached as static HTML.', 'wp-static-html-pages'); ?></p>
            </td>
        </tr>
        <?php
    }

    public function save_term_meta($term_id, $tt_id) {
        if (!current_user_can('manage_options')) return;

        $exclude = isset($_POST['wpsh_exclude']) ? '1' : '0';
        update_term_meta($term_id, '_wpsh_exclude', $exclude);
        
        if ($exclude === '1') {
            $term = get_term($term_id);
            if ($term && !is_wp_error($term)) {
                $url = get_term_link($term);
                if (!is_wp_error($url)) {
                    $path = $this->core->get_static_path_from_url($url);
                    if ($path && file_exists($path)) unlink($path);
                }
            }
        }
    }

    public function render_new_term_meta_fields($taxonomy) {
        ?>
        <div class="form-field">
            <label><?php _e('Static HTML Settings', 'wp-static-html-pages'); ?></label>
            <label style="display:flex; align-items:center; gap:8px;">
                <input type="checkbox" name="wpsh_exclude" value="1" style="width:auto; margin:0;"> 
                <?php _e('Exclude from Static HTML Cache', 'wp-static-html-pages'); ?>
            </label>
            <p class="description"><?php _e('If checked, this archive page will not be cached as static HTML.', 'wp-static-html-pages'); ?></p>
        </div>
        <?php
    }

    public function render_admin_page() {
        $template = plugin_dir_path(__FILE__) . '../templates/admin-dashboard.php';
        if (file_exists($template)) {
            include_once $template;
        }
    }
}
