jQuery(document).ready(function ($) {
    // Toast Notification Helper
    function showToast(message, type = 'success') {
        var icon = 'dashicons-info';
        if (type === 'success') icon = 'dashicons-yes';
        if (type === 'error') icon = 'dashicons-warning';

        var toastId = 'toast-' + Date.now();
        var html = '<div id="' + toastId + '" class="wpsh-toast ' + type + '">' +
            '<span class="wpsh-toast-icon dashicons ' + icon + '"></span>' +
            '<div class="wpsh-toast-message">' + message + '</div>' +
            '</div>';

        $('#wpsh-toaster').append(html);

        // Auto remove
        setTimeout(function () {
            $('#' + toastId).addClass('fade-out');
            setTimeout(function () {
                $('#' + toastId).remove();
            }, 500);
        }, 4000);
    }

    // Function to switch tab
    function switchTab(tab) {
        $('.wpsh-nav-item').removeClass('active');
        $('.wpsh-nav-item[data-tab="' + tab + '"]').addClass('active');
        $('.wpsh-tab-content').removeClass('active');
        $('#wpsh-tab-' + tab).addClass('active');
    }

    // Enable/disable Site Generation and Server Rules when plugin is off/on
    function setDashboardActionsState(isEnabled) {
        var $cards = $('.wpsh-requires-enabled');
        var $btns = $('#wpsh-generate-btn, #wpsh-clear-btn, #wpsh-update-htaccess-btn');
        if (isEnabled) {
            $cards.removeClass('wpsh-disabled');
            $btns.prop('disabled', false);
        } else {
            $cards.addClass('wpsh-disabled');
            $btns.prop('disabled', true);
        }
    }

    // Nav Tabs click
    $('.wpsh-nav-item').on('click', function (e) {
        e.preventDefault();
        if ($(this).hasClass('wpsh-nav-disabled')) {
            var msg = $('.wpsh-wrapper').attr('data-wpsh-license-active') === '1'
                ? (wpshData.strings.enableToAccess || 'Enable the plugin to access Tools and Settings.')
                : (wpshData.strings.activateLicenseToAccess || 'Activate your license in the Licence tab to access Tools and Settings.');
            showToast(msg, 'info');
            return;
        }
        var tab = $(this).data('tab');
        switchTab(tab);
        window.location.hash = tab;
    });

    // Go to Licence link (from dashboard notice)
    $('.wpsh-goto-licence').on('click', function (e) {
        e.preventDefault();
        switchTab('licence');
        window.location.hash = 'licence';
    });

    // Restore tab from hash on load (redirect to dashboard if Tools/Settings when plugin disabled)
    var isPluginEnabled = $('.wpsh-wrapper').attr('data-wpsh-enabled') === '1';
    setDashboardActionsState(isPluginEnabled);
    if (window.location.hash) {
        var tab = window.location.hash.substring(1);
        if (tab === 'tools' || tab === 'settings') {
            if (!isPluginEnabled) {
                switchTab('dashboard');
                window.location.hash = 'dashboard';
            } else if ($('#wpsh-tab-' + tab).length) {
                switchTab(tab);
            }
        } else if ($('#wpsh-tab-' + tab).length) {
            switchTab(tab);
        }
    }

    // Master Toggle
    $('#wpsh-master-toggle').on('change', function () {
        var isEnabled = $(this).is(':checked');
        $('.wpsh-master-status').toggleClass('enabled', isEnabled).toggleClass('disabled', !isEnabled);
        $('.status-label').text(isEnabled ? wpshData.strings.enabled : wpshData.strings.disabled);

        var $toolsNav = $('.wpsh-nav-item[data-tab="tools"]');
        var $settingsNav = $('.wpsh-nav-item[data-tab="settings"]');
        if (isEnabled) {
            $toolsNav.add($settingsNav).removeClass('wpsh-nav-disabled');
            $('.wpsh-wrapper').data('wpsh-enabled', '1');
            setDashboardActionsState(true);
        } else {
            $toolsNav.add($settingsNav).addClass('wpsh-nav-disabled');
            $('.wpsh-wrapper').data('wpsh-enabled', '0');
            setDashboardActionsState(false);
            var cur = $('.wpsh-nav-item.active').data('tab');
            if (cur === 'tools' || cur === 'settings') {
                switchTab('dashboard');
                window.location.hash = 'dashboard';
            }
        }

        $.ajax({
            url: ajaxurl,
            type: 'POST',
            data: {
                action: 'wpsh_toggle_plugin',
                enabled: isEnabled,
                nonce: wpshData.nonces.toggle
            },
            success: function () {
                showToast(isEnabled ? 'Plugin Enabled' : 'Plugin Disabled');
            }
        });
    });

    // Clear Cache
    $('#wpsh-clear-btn').on('click', function () {
        var $btn = $(this);
        if (!confirm(wpshData.strings.confirmClear)) return;

        $btn.prop('disabled', true);
        $.ajax({
            url: ajaxurl,
            type: 'POST',
            data: {
                action: 'wpsh_clear_cache',
                nonce: wpshData.nonces.clear
            },
            success: function (response) {
                if (response.success) {
                    showToast(wpshData.strings.cacheCleared);
                    $('#wpsh-cached-count').text('0');
                } else {
                    showToast('Error: ' + response.data, 'error');
                }
                $btn.prop('disabled', false);
            }
        });
    });

    // Bulk Generator
    var totalToProcess = 0;
    var currentProcessed = 0;

    $('#wpsh-generate-btn').on('click', function () {
        var $btn = $(this);
        $btn.prop('disabled', true);
        $('.wpsh-progress-wrap').fadeIn();
        $('.wpsh-log-wrap').fadeIn();
        $('#wpsh-log').empty();

        currentProcessed = 0;
        updateProgress(0, 'Starting...');
        runBatch(0);
    });

    function runBatch(offset) {
        $.ajax({
            url: ajaxurl,
            type: 'POST',
            data: {
                action: 'wpsh_generate_all',
                offset: offset,
                nonce: wpshData.nonces.generate
            },
            success: function (response) {
                if (response.success) {
                    if (offset === 0) totalToProcess = response.data.total_to_process;
                    currentProcessed = response.data.next_offset;

                    // Log entries
                    response.data.processed.forEach(function (msg) {
                        $('#wpsh-log').prepend('<div><span class="dashicons dashicons-arrow-right-alt2"></span> ' + msg + '</div>');
                    });

                    var percent = Math.min(100, Math.round((currentProcessed / totalToProcess) * 100));
                    updateProgress(percent, 'Processing ' + currentProcessed + ' of ' + totalToProcess + '...');

                    if (!response.data.complete) {
                        runBatch(currentProcessed);
                    } else {
                        updateProgress(100, 'Complete!');
                        showToast('Site generation complete!');
                        $('#wpsh-generate-btn').prop('disabled', false);
                    }
                } else {
                    showToast('Error: ' + response.data, 'error');
                    $('#wpsh-generate-btn').prop('disabled', false);
                }
            }
        });
    }

    function updateProgress(percent, label) {
        $('#wpsh-progress-fill').css('width', percent + '%');
        $('#wpsh-progress-percent').text(percent + '%');
        $('#wpsh-progress-label').text(label);
    }

    // Exclusions & Settings
    $('.wpsh-pt-toggle, .wpsh-tax-toggle').on('change', function () {
        var $toggle = $(this);
        var isPT = $toggle.hasClass('wpsh-pt-toggle');
        var val = isPT ? $toggle.data('pt') : $toggle.data('tax');
        var action = isPT ? 'wpsh_save_pt_exclusion' : 'wpsh_save_tax_exclusion';
        var nonce = isPT ? wpshData.nonces.pt_exclusion : wpshData.nonces.tax_exclusion;

        $.ajax({
            url: ajaxurl,
            type: 'POST',
            data: {
                action: action,
                value: val,
                post_type: val,
                taxonomy: val,
                excluded: $toggle.is(':checked'),
                nonce: nonce
            },
            success: function () {
                showToast('Exclusion settings updated');
            }
        });
    });

    $('#wpsh-save-exclusions-btn').on('click', function () {
        var $btn = $(this);
        var exclusions = $('#wpsh-url-exclusions').val();
        $btn.prop('disabled', true).text('Saving...');
        $.ajax({
            url: ajaxurl,
            type: 'POST',
            data: {
                action: 'wpsh_save_global_setting',
                setting: 'wpsh_url_exclusions',
                value: exclusions,
                nonce: wpshData.nonces.settings
            },
            success: function () {
                $btn.text('Saved!');
                showToast('Global exclusions saved');
                setTimeout(function () { $btn.prop('disabled', false).text('Save Exclusion List'); }, 2000);
            }
        });
    });

    // Copy .htaccess rules to clipboard
    $('#wpsh-copy-htaccess-btn').on('click', function () {
        var text = $('#wpsh-htaccess-rules').text();
        if (navigator.clipboard && navigator.clipboard.writeText) {
            navigator.clipboard.writeText(text).then(function () {
                showToast(wpshData.strings.copiedToClipboard || 'Copied to clipboard.');
            });
        } else {
            var $ta = $('<textarea>').val(text).css({ position: 'fixed', left: '-9999px' }).appendTo('body');
            $ta.select();
            document.execCommand('copy');
            $ta.remove();
            showToast(wpshData.strings.copiedToClipboard || 'Copied to clipboard.');
        }
    });

    // Update .htaccess
    $('#wpsh-update-htaccess-btn').on('click', function () {
        var $btn = $(this);
        $btn.prop('disabled', true).text('Updating...');
        $.ajax({
            url: ajaxurl,
            type: 'POST',
            data: {
                action: 'wpsh_update_htaccess',
                nonce: wpshData.nonces.htaccess
            },
            success: function (response) {
                if (response.success) {
                    showToast('Server rules updated successfully!');
                } else {
                    showToast('Error: ' + response.data, 'error');
                }
                $btn.prop('disabled', false).html('<span class="dashicons dashicons-admin-settings"></span> Update .htaccess Rules');
            }
        });
    });

    // Activate License
    $('#wpsh-activate-license-btn').on('click', function () {
        var $btn = $(this);
        var licenseKey = $('#wpsh-license-key').val().trim();
        
        if (!licenseKey) {
            showToast('Please enter a license key.', 'error');
            $('#wpsh-license-key').focus();
            return;
        }
        
        $btn.prop('disabled', true).html('<span class="dashicons dashicons-update"></span> Activating...');
        $.ajax({
            url: ajaxurl,
            type: 'POST',
            data: {
                action: 'wpsh_activate_license',
                license_key: licenseKey,
                nonce: wpshData.nonces.license
            },
            success: function (response) {
                if (response.success) {
                    showToast(response.data.message || 'License activated successfully!', 'success');
                    setTimeout(function () {
                        location.reload();
                    }, 1500);
                } else {
                    showToast(response.data || 'Failed to activate license. Please check your key.', 'error');
                    $btn.prop('disabled', false).html('<span class="dashicons dashicons-admin-network"></span> Activate License');
                }
            },
            error: function () {
                showToast('An error occurred. Please try again.', 'error');
                $btn.prop('disabled', false).html('<span class="dashicons dashicons-admin-network"></span> Activate License');
            }
        });
    });

    // Check License
    $('#wpsh-check-license-btn').on('click', function () {
        var $btn = $(this);
        $btn.prop('disabled', true).html('<span class="dashicons dashicons-update"></span> Checking...');
        $.ajax({
            url: ajaxurl,
            type: 'POST',
            data: {
                action: 'wpsh_check_license',
                nonce: wpshData.nonces.license
            },
            success: function (response) {
                if (response.success) {
                    showToast(response.data.message || 'License is active and valid.', 'success');
                    setTimeout(function () {
                        location.reload();
                    }, 1500);
                } else {
                    showToast(response.data || 'License is not active.', 'error');
                    $btn.prop('disabled', false).html('<span class="dashicons dashicons-update"></span> Check License');
                }
            },
            error: function () {
                showToast('An error occurred. Please try again.', 'error');
                $btn.prop('disabled', false).html('<span class="dashicons dashicons-update"></span> Check License');
            }
        });
    });

    // Deactivate License
    $('#wpsh-deactivate-license-btn').on('click', function () {
        var $btn = $(this);
        if (!confirm('Are you sure you want to deactivate your license?')) return;
        
        $btn.prop('disabled', true).html('<span class="dashicons dashicons-update"></span> Deactivating...');
        $.ajax({
            url: ajaxurl,
            type: 'POST',
            data: {
                action: 'wpsh_deactivate_license',
                nonce: wpshData.nonces.license
            },
            success: function (response) {
                if (response.success) {
                    showToast(response.data.message || 'License deactivated successfully.', 'success');
                    setTimeout(function () {
                        location.reload();
                    }, 1500);
                } else {
                    showToast(response.data || 'Failed to deactivate license.', 'error');
                    $btn.prop('disabled', false).html('<span class="dashicons dashicons-dismiss"></span> Deactivate License');
                }
            },
            error: function () {
                showToast('An error occurred. Please try again.', 'error');
                $btn.prop('disabled', false).html('<span class="dashicons dashicons-dismiss"></span> Deactivate License');
            }
        });
    });
});
