<?php
namespace AssertivlogixBackup;

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

if (!class_exists('WP_CLI_Command')) {
    return;
}

/**
 * WP-CLI commands for Assertivlogix Backup.
 */
class CLI extends \WP_CLI_Command {

    /**
     * Run a backup.
     * 
     * ## EXAMPLES
     * 
     *     wp assertivlogix-backup run
     * 
     * @when after_wp_load
     */
    public function run($args, $assoc_args) {
        \WP_CLI::line('Starting backup...');
        
        $backup = new Backup();
        $result = $backup->run();
        
        if (is_wp_error($result)) {
            \WP_CLI::error($result->get_error_message());
        } else {
            \WP_CLI::success("Backup created successfully: $result");
        }
    }

    /**
     * List backups.
     * 
     * ## EXAMPLES
     * 
     *     wp assertivlogix-backup list
     * 
     * @when after_wp_load
     */
    public function list($args, $assoc_args) {
        $backup = new Backup();
        $backups = $backup->get_backups();
        
        if (empty($backups)) {
            \WP_CLI::line('No backups found.');
            return;
        }

        \WP_CLI\Utils\format_items('table', $backups, ['date', 'filename', 'size']);
    }

    /**
     * Restore a backup.
     * 
     * ## OPTIONS
     * 
     * <filename>
     * : The filename of the backup to restore.
     * 
     * ## EXAMPLES
     * 
     *     wp assertivlogix-backup restore assertivlogix-backup-20230101_120000.zip
     * 
     * @when after_wp_load
     */
    public function restore($args, $assoc_args) {
        list($filename) = $args;
        
        \WP_CLI::confirm("Are you sure you want to restore $filename? This will overwrite current data.");
        
        $restore = new Restore();
        $result = $restore->run($filename);
        
        if (is_wp_error($result)) {
            \WP_CLI::error($result->get_error_message());
        } else {
            \WP_CLI::success("Restored successfully.");
        }
    }
}

\WP_CLI::add_command('assertivlogix-backup', 'AssertivlogixBackup\CLI');
?>
