<?php
namespace AssertivlogixBackup;

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

/**
 * Admin class handling the plugin settings page and UI.
 */
class Admin {
    /**
     * Initialize admin hooks.
     */
    public function init() {
        add_action('admin_menu', [$this, 'add_admin_menu']);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_assets']);
        
        // AJAX actions
        add_action('wp_ajax_assertivlogix_backup_run', [$this, 'ajax_run_backup']);
        add_action('wp_ajax_assertivlogix_backup_save_schedule', [$this, 'ajax_save_schedule']);
        add_action('wp_ajax_assertivlogix_backup_list', [$this, 'ajax_list_backups']);
        add_action('wp_ajax_assertivlogix_backup_restore', [$this, 'ajax_restore_backup']);
        add_action('wp_ajax_assertivlogix_backup_delete', [$this, 'ajax_delete_backup']);
        add_action('wp_ajax_assertivlogix_backup_save_settings', [$this, 'ajax_save_settings']);
        
        // Migration actions
        add_action('wp_ajax_assertivlogix_backup_upload_migrate', [$this, 'ajax_upload_migrate']);
        add_action('wp_ajax_assertivlogix_backup_search_replace', [$this, 'ajax_search_replace']);
        add_action('wp_ajax_assertivlogix_backup_cleanup', [$this, 'ajax_cleanup']);
        add_action('wp_ajax_assertivlogix_backup_selective', [$this, 'ajax_selective_backup']);
        add_action('wp_ajax_assertivlogix_backup_get_plugins', [$this, 'ajax_get_plugins']);
        add_action('wp_ajax_assertivlogix_backup_get_themes', [$this, 'ajax_get_themes']);
        add_action('wp_ajax_assertivlogix_backup_get_tables', [$this, 'ajax_get_tables']);
    }

    /**
     * Register the plugin menu page.
     */
    public function add_admin_menu() {
        add_menu_page(
            __('Assertivlogix Backup & Migration', 'assertivlogix-backup-and-migration'),
            __('Assertivlogix Backup', 'assertivlogix-backup-and-migration'),
            'manage_options',
            'assertivlogix-backup',
            [$this, 'render_settings_page'],
            'dashicons-backup',
            80
        );
        
        add_submenu_page(
            'assertivlogix-backup',
            __('How Assertivlogix Backup Works', 'assertivlogix-backup-and-migration'),
            __('How it works', 'assertivlogix-backup-and-migration'),
            'manage_options',
            'assertivlogix-backup-how-it-works',
            [$this, 'render_how_it_works']
        );
    }

    /**
     * Enqueue CSS/JS for the admin UI.
     */
    public function enqueue_assets($hook) {
        if (strpos($hook, 'assertivlogix-backup') === false) {
            return;
        }
        // Use a modern, premium UI – load a custom stylesheet.
        wp_enqueue_style(
            'assertivlogix-backup-admin',
            ASSERTIVLOGIX_BM_PLUGIN_URL . 'assets/css/admin.css',
            [],
            ASSERTIVLOGIX_BM_VERSION
        );
        wp_enqueue_script(
            'assertivlogix-backup-admin',
            ASSERTIVLOGIX_BM_PLUGIN_URL . 'assets/js/admin.js',
            ['jquery'],
            ASSERTIVLOGIX_BM_VERSION,
            true
        );
        // Localize script for AJAX actions.
        wp_localize_script('assertivlogix-backup-admin', 'assertivlogixBackup', [
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'nonce'   => wp_create_nonce('assertivlogix_backup_nonce'),
            'strings' => [
                'confirmRestore' => __('Are you sure you want to restore this backup? This will overwrite your current site content and database.', 'assertivlogix-backup-and-migration'),
                'backupSuccess' => __('Backup completed successfully!', 'assertivlogix-backup-and-migration'),
                'restoreSuccess' => __('Restore completed successfully!', 'assertivlogix-backup-and-migration'),
            ]
        ]);
    }

    /**
     * Render the plugin settings page.
     */
    public function render_settings_page() {
        $active_tab = isset($_GET['tab']) ? $_GET['tab'] : 'backup';
        ?>
        <div class="wrap assertivlogix-backup-wrapper">
            <h1 class="wp-heading-inline"><?php esc_html_e('Assertivlogix Backup & Migration', 'assertivlogix-backup-and-migration'); ?></h1>
            <hr class="wp-header-end"/>
            
            <nav class="nav-tab-wrapper">
                <a href="?page=assertivlogix-backup&tab=backup" class="nav-tab <?php echo $active_tab == 'backup' ? 'nav-tab-active' : ''; ?>"><?php _e('Backup / Restore', 'assertivlogix-backup-and-migration'); ?></a>
                <a href="?page=assertivlogix-backup&tab=migration" class="nav-tab <?php echo $active_tab == 'migration' ? 'nav-tab-active' : ''; ?>"><?php _e('Migration', 'assertivlogix-backup-and-migration'); ?></a>
                <a href="?page=assertivlogix-backup&tab=storage" class="nav-tab <?php echo $active_tab == 'storage' ? 'nav-tab-active' : ''; ?>"><?php _e('Storage', 'assertivlogix-backup-and-migration'); ?></a>
                <a href="?page=assertivlogix-backup&tab=settings" class="nav-tab <?php echo $active_tab == 'settings' ? 'nav-tab-active' : ''; ?>"><?php _e('Settings', 'assertivlogix-backup-and-migration'); ?></a>
            </nav>

            <div class="assertivlogix-backup-dashboard">
                <?php if ($active_tab == 'backup'): ?>
                    <section class="assertivlogix-backup-card" id="backup-now">
                        <h2><?php esc_html_e('Back Up Now', 'assertivlogix-backup-and-migration'); ?></h2>
                        <p><?php esc_html_e('Create an immediate backup of your site files and database.', 'assertivlogix-backup-and-migration'); ?></p>
                        <button class="button button-primary" id="btn-backup-now"><?php esc_html_e('Run Backup', 'assertivlogix-backup-and-migration'); ?></button>
                    </section>
                    <section class="assertivlogix-backup-card" id="schedule-backup">
                        <h2><?php esc_html_e('Schedule Backups', 'assertivlogix-backup-and-migration'); ?></h2>
                        <p><?php esc_html_e('Choose a schedule interval for automatic backups.', 'assertivlogix-backup-and-migration'); ?></p>
                        <select id="backup-interval" class="regular-text">
                            <option value="2" <?php selected(get_option('assertivlogix_backup_schedule'), '2'); ?>><?php esc_html_e('Every 2 hours', 'assertivlogix-backup-and-migration'); ?></option>
                            <option value="4" <?php selected(get_option('assertivlogix_backup_schedule'), '4'); ?>><?php esc_html_e('Every 4 hours', 'assertivlogix-backup-and-migration'); ?></option>
                            <option value="8" <?php selected(get_option('assertivlogix_backup_schedule'), '8'); ?>><?php esc_html_e('Every 8 hours', 'assertivlogix-backup-and-migration'); ?></option>
                            <option value="12" <?php selected(get_option('assertivlogix_backup_schedule'), '12'); ?>><?php esc_html_e('Every 12 hours', 'assertivlogix-backup-and-migration'); ?></option>
                            <option value="daily" <?php selected(get_option('assertivlogix_backup_schedule'), 'daily'); ?>><?php esc_html_e('Daily', 'assertivlogix-backup-and-migration'); ?></option>
                            <option value="weekly" <?php selected(get_option('assertivlogix_backup_schedule'), 'weekly'); ?>><?php esc_html_e('Weekly', 'assertivlogix-backup-and-migration'); ?></option>
                            <option value="fortnightly" <?php selected(get_option('assertivlogix_backup_schedule'), 'fortnightly'); ?>><?php esc_html_e('Fortnightly', 'assertivlogix-backup-and-migration'); ?></option>
                            <option value="monthly" <?php selected(get_option('assertivlogix_backup_schedule'), 'monthly'); ?>><?php esc_html_e('Monthly', 'assertivlogix-backup-and-migration'); ?></option>
                        </select>
                        <button class="button" id="btn-save-schedule"><?php esc_html_e('Save Schedule', 'assertivlogix-backup-and-migration'); ?></button>
                    </section>
                    <section class="assertivlogix-backup-card" id="restore-backup">
                        <h2><?php esc_html_e('Existing Backups', 'assertivlogix-backup-and-migration'); ?></h2>
                        <p><?php esc_html_e('Download or restore previous backups.', 'assertivlogix-backup-and-migration'); ?></p>
                        <div id="backup-list">
                            <!-- Dynamically populated via AJAX -->
                            <p class="description"><?php _e('Loading backups...', 'assertivlogix-backup-and-migration'); ?></p>
                        </div>
                    </section>
                
                <?php elseif ($active_tab == 'migration'): ?>
                    
                    <!-- Basic Migration -->
                    <section class="assertivlogix-backup-card">
                        <div class="moon-feature-header">
                            <div class="moon-feature-icon-inline"><span class="dashicons dashicons-update"></span></div>
                            <h2><?php esc_html_e('Basic Migration', 'assertivlogix-backup-and-migration'); ?></h2>
                        </div>
                        <p><?php esc_html_e('Upload a backup file from your source site and restore it here.', 'assertivlogix-backup-and-migration'); ?></p>
                        
                        <form id="moon-upload-backup-form" enctype="multipart/form-data">
                            <div class="moon-upload-area">
                                <input type="file" id="backup-file-upload" name="backup_file" accept=".zip" style="display: none;" />
                                <div class="moon-upload-box" id="upload-box">
                                    <span class="dashicons dashicons-upload"></span>
                                    <p><strong><?php _e('Click to upload', 'assertivlogix-backup-and-migration'); ?></strong> <?php _e('or drag and drop', 'assertivlogix-backup-and-migration'); ?></p>
                                    <p class="description"><?php _e('ZIP file only (Max size: 512MB)', 'assertivlogix-backup-and-migration'); ?></p>
                                </div>
                                <div id="file-info" class="moon-file-info" style="display: none;">
                                    <span class="dashicons dashicons-media-archive"></span>
                                    <span id="file-name"></span>
                                    <span id="file-size"></span>
                                    <button type="button" id="remove-file" class="button"><?php _e('Remove', 'assertivlogix-backup-and-migration'); ?></button>
                                </div>
                            </div>
                            
                            <div class="moon-migration-actions" style="margin-top: 20px;">
                                <button type="submit" class="button button-primary" id="btn-upload-migrate" disabled>
                                    <span class="dashicons dashicons-upload" style="margin-top: 3px;"></span>
                                    <?php esc_html_e('Upload & Migrate', 'assertivlogix-backup-and-migration'); ?>
                                </button>
                            </div>
                        </form>
                        
                        <div class="moon-migration-steps" style="margin-top: 30px; padding-top: 30px; border-top: 1px solid #e2e8f0;">
                            <h3><?php _e('Migration Steps:', 'assertivlogix-backup-and-migration'); ?></h3>
                            <ol class="moon-steps-list">
                                <li><?php _e('Create a backup on your source site (Backup/Restore tab)', 'assertivlogix-backup-and-migration'); ?></li>
                                <li><?php _e('Download the backup ZIP file', 'assertivlogix-backup-and-migration'); ?></li>
                                <li><?php _e('Upload the ZIP file here', 'assertivlogix-backup-and-migration'); ?></li>
                                <li><?php _e('Click "Upload & Migrate" to restore', 'assertivlogix-backup-and-migration'); ?></li>
                            </ol>
                        </div>
                    </section>

                    <!-- Search and Replace -->
                    <section class="assertivlogix-backup-card">
                        <div class="moon-feature-header">
                            <div class="moon-feature-icon-inline"><span class="dashicons dashicons-search"></span></div>
                            <h2><?php esc_html_e('Search and Replace', 'assertivlogix-backup-and-migration'); ?></h2>
                        </div>
                        <p><?php esc_html_e('Find and replace text in your database. Useful for changing URLs after migration.', 'assertivlogix-backup-and-migration'); ?></p>
                        
                        <div class="moon-form-row">
                            <div class="moon-form-col">
                                <label for="search-text"><?php _e('Search for:', 'assertivlogix-backup-and-migration'); ?></label>
                                <input type="text" id="search-text" class="regular-text" placeholder="https://oldsite.com" />
                            </div>
                            <div class="moon-form-col">
                                <label for="replace-text"><?php _e('Replace with:', 'assertivlogix-backup-and-migration'); ?></label>
                                <input type="text" id="replace-text" class="regular-text" placeholder="https://newsite.com" />
                            </div>
                        </div>
                        
                        <p class="description" style="color: #dc2626; margin-bottom: 16px;">
                            <span class="dashicons dashicons-warning"></span>
                            <?php _e('Warning: This will modify your database. Create a backup first!', 'assertivlogix-backup-and-migration'); ?>
                        </p>
                        
                        <button class="button button-primary" id="btn-search-replace"><?php esc_html_e('Run Search & Replace', 'assertivlogix-backup-and-migration'); ?></button>
                    </section>

                    <!-- Migration Cleanup -->
                    <section class="assertivlogix-backup-card">
                        <div class="moon-feature-header">
                            <div class="moon-feature-icon-inline"><span class="dashicons dashicons-trash"></span></div>
                            <h2><?php esc_html_e('Migration Cleanup', 'assertivlogix-backup-and-migration'); ?></h2>
                        </div>
                        <p><?php esc_html_e('Run cleanup tasks after migration to ensure everything works correctly.', 'assertivlogix-backup-and-migration'); ?></p>
                        
                        <ul class="moon-cleanup-tasks">
                            <li><span class="dashicons dashicons-yes"></span> Flush rewrite rules</li>
                            <li><span class="dashicons dashicons-yes"></span> Clear WordPress cache</li>
                            <li><span class="dashicons dashicons-yes"></span> Regenerate .htaccess</li>
                            <li><span class="dashicons dashicons-yes"></span> Synchronize site URLs</li>
                            <li><span class="dashicons dashicons-yes"></span> Clear expired transients</li>
                        </ul>
                        
                        <button class="button button-primary" id="btn-cleanup"><?php esc_html_e('Run Cleanup Tasks', 'assertivlogix-backup-and-migration'); ?></button>
                    </section>

                    <!-- Selective Migration -->
                    <section class="assertivlogix-backup-card">
                        <div class="moon-feature-header">
                            <div class="moon-feature-icon-inline"><span class="dashicons dashicons-filter"></span></div>
                            <h2><?php esc_html_e('Selective Migration', 'assertivlogix-backup-and-migration'); ?></h2>
                        </div>
                        <p><?php esc_html_e('Choose exactly what to include in your backup.', 'assertivlogix-backup-and-migration'); ?></p>
                        
                        <div class="moon-selective-options">
                            <div class="moon-option-group">
                                <h4><?php _e('Plugins', 'assertivlogix-backup-and-migration'); ?></h4>
                                <button class="button" id="btn-select-plugins"><?php _e('Select Plugins', 'assertivlogix-backup-and-migration'); ?></button>
                                <span id="selected-plugins-count" class="description">0 selected</span>
                            </div>
                            
                            <div class="moon-option-group">
                                <h4><?php _e('Themes', 'assertivlogix-backup-and-migration'); ?></h4>
                                <button class="button" id="btn-select-themes"><?php _e('Select Themes', 'assertivlogix-backup-and-migration'); ?></button>
                                <span id="selected-themes-count" class="description">0 selected</span>
                            </div>
                            
                            <div class="moon-option-group">
                                <h4><?php _e('Database Tables', 'assertivlogix-backup-and-migration'); ?></h4>
                                <button class="button" id="btn-select-tables"><?php _e('Select Tables', 'assertivlogix-backup-and-migration'); ?></button>
                                <span id="selected-tables-count" class="description">0 selected</span>
                            </div>
                            
                            <div class="moon-option-group">
                                <label>
                                    <input type="checkbox" id="include-uploads" checked />
                                    <?php _e('Include Uploads Folder', 'assertivlogix-backup-and-migration'); ?>
                                </label>
                            </div>
                        </div>
                        
                        <button class="button button-primary" id="btn-selective-backup"><?php esc_html_e('Create Selective Backup', 'assertivlogix-backup-and-migration'); ?></button>
                    </section>

                    <!-- Premium Features Info -->
                    <section class="assertivlogix-backup-card moon-premium-info">
                        <div class="moon-feature-header">
                            <div class="moon-feature-icon-inline"><span class="dashicons dashicons-star-filled"></span></div>
                            <h2><?php esc_html_e('Premium Features', 'assertivlogix-backup-and-migration'); ?></h2>
                        </div>
                        <p><?php esc_html_e('Unlock advanced migration features with Assertivlogix Backup Pro:', 'assertivlogix-backup-and-migration'); ?></p>
                        
                        <ul class="moon-premium-features">
                            <li><span class="dashicons dashicons-networking"></span> <strong>Direct Site-to-Site Migration</strong> - Link two sites and migrate back and forth instantly</li>
                            <li><span class="dashicons dashicons-admin-multisite"></span> <strong>Multisite Support</strong> - Migrate single sites into multisite networks</li>
                            <li><span class="dashicons dashicons-cloud"></span> <strong>Cloud Storage</strong> - Automatic backups to Google Drive, Dropbox, S3, and more</li>
                            <li><span class="dashicons dashicons-admin-generic"></span> <strong>Advanced Scheduling</strong> - Custom backup schedules and retention policies</li>
                        </ul>
                        
                        <button class="button button-large" disabled><?php esc_html_e('Upgrade to Pro', 'assertivlogix-backup-and-migration'); ?></button>
                    </section>


                <?php elseif ($active_tab == 'storage'): ?>
                    <section class="assertivlogix-backup-card">
                        <h2><?php esc_html_e('Remote Storage Locations', 'assertivlogix-backup-and-migration'); ?></h2>
                        <p><?php esc_html_e('Connect your cloud storage accounts.', 'assertivlogix-backup-and-migration'); ?></p>
                        <ul class="storage-list">
                            <li><label><input type="checkbox" disabled> Google Drive (Pro)</label></li>
                            <li><label><input type="checkbox" disabled> Dropbox (Pro)</label></li>
                            <li><label><input type="checkbox" disabled> Amazon S3 (Pro)</label></li>
                            <li><label><input type="checkbox" disabled> OneDrive (Pro)</label></li>
                        </ul>
                    </section>

                <?php elseif ($active_tab == 'settings'): ?>
                    <section class="assertivlogix-backup-card">
                        <h2><?php esc_html_e('General Settings', 'assertivlogix-backup-and-migration'); ?></h2>
                        <p><label><input type="checkbox" id="setting-auto-backup" <?php checked(get_option('assertivlogix_backup_auto_update'), '1'); ?>> <?php _e('Backup automatically before updates', 'assertivlogix-backup-and-migration'); ?></label></p>
                        <p><label><input type="checkbox" id="setting-anonymise" <?php checked(get_option('assertivlogix_backup_anonymise'), '1'); ?>> <?php _e('Anonymise data (GDPR)', 'assertivlogix-backup-and-migration'); ?></label></p>
                        <p><label><input type="checkbox" id="setting-encrypt" <?php checked(get_option('assertivlogix_backup_encrypt'), '1'); ?>> <?php _e('Encrypt database backups', 'assertivlogix-backup-and-migration'); ?></label></p>
                        <p><button class="button button-primary" id="btn-save-settings"><?php esc_html_e('Save Changes', 'assertivlogix-backup-and-migration'); ?></button></p>
                    </section>
                <?php endif; ?>
            </div>
        </div>
        <?php
    }

    public function render_how_it_works() {
        ?>
        <div class="wrap assertivlogix-backup-wrapper">
            <h1 class="wp-heading-inline"><?php _e('How Assertivlogix Backup Works', 'assertivlogix-backup-and-migration'); ?></h1>
            <hr class="wp-header-end"/>
            <div class="assertivlogix-backup-dashboard">
                <section class="assertivlogix-backup-card">
                    <h2><?php _e('1. Choose where to back up', 'assertivlogix-backup-and-migration'); ?></h2>
                    <p><?php _e('Select from a wide range of cloud storage options including Google Drive, Dropbox, S3, and more.', 'assertivlogix-backup-and-migration'); ?></p>
                </section>
                <section class="assertivlogix-backup-card">
                    <h2><?php _e('2. Choose what to back up', 'assertivlogix-backup-and-migration'); ?></h2>
                    <p><?php _e('Back up your WordPress database and content files, including plugins, themes, and uploads.', 'assertivlogix-backup-and-migration'); ?></p>
                </section>
                <section class="assertivlogix-backup-card">
                    <h2><?php _e('3. Choose when to back up', 'assertivlogix-backup-and-migration'); ?></h2>
                    <p><?php _e('Run backups manually or schedule them to run automatically in just a few clicks.', 'assertivlogix-backup-and-migration'); ?></p>
                </section>
                <section class="assertivlogix-backup-card">
                    <h2><?php _e('4. Rapid restore', 'assertivlogix-backup-and-migration'); ?></h2>
                    <p><?php _e('Quickly restore your WordPress site from a backup in just minutes.', 'assertivlogix-backup-and-migration'); ?></p>
                </section>
            </div>
        </div>
        <?php
    }

    // AJAX Handlers

    public function ajax_run_backup() {
        check_ajax_referer('assertivlogix_backup_nonce');
        if (!current_user_can('manage_options')) wp_send_json_error('Unauthorized');

        $backup = new Backup();
        $result = $backup->run();

        if (is_wp_error($result)) {
            wp_send_json_error($result->get_error_message());
        } else {
            wp_send_json_success(['filename' => $result]);
        }
    }

    public function ajax_save_schedule() {
        check_ajax_referer('assertivlogix_backup_nonce');
        if (!current_user_can('manage_options')) wp_send_json_error('Unauthorized');

        $schedule = sanitize_text_field($_POST['schedule']);
        update_option('assertivlogix_backup_schedule', $schedule);
        
        // Re-init scheduler to apply changes immediately
        $scheduler = new Scheduler();
        $scheduler->setup_cron_schedule();

        wp_send_json_success();
    }

    public function ajax_list_backups() {
        check_ajax_referer('assertivlogix_backup_nonce');
        if (!current_user_can('manage_options')) wp_send_json_error('Unauthorized');

        $backup = new Backup();
        $backups = $backup->get_backups();
        
        // Format for display
        $html = '';
        if (empty($backups)) {
            $html = '<p>' . __('No backups found.', 'assertivlogix-backup-and-migration') . '</p>';
        } else {
            $html .= '<table class="widefat fixed striped">';
            $html .= '<thead><tr><th>' . __('Date', 'assertivlogix-backup-and-migration') . '</th><th>' . __('Size', 'assertivlogix-backup-and-migration') . '</th><th>' . __('Actions', 'assertivlogix-backup-and-migration') . '</th></tr></thead>';
            $html .= '<tbody>';
            foreach ($backups as $b) {
                $download_url = content_url('uploads/assertivlogix-backups/' . $b['filename']);
                $html .= '<tr>';
                $html .= '<td>' . esc_html($b['date']) . '</td>';
                $html .= '<td>' . esc_html($b['size']) . '</td>';
                $html .= '<td>';
                $html .= '<a href="' . esc_url($download_url) . '" class="button" download>' . __('Download', 'assertivlogix-backup-and-migration') . '</a> ';
                $html .= '<button class="button button-primary btn-restore" data-filename="' . esc_attr($b['filename']) . '">' . __('Restore', 'assertivlogix-backup-and-migration') . '</button> ';
                $html .= '<button class="button button-link-delete btn-delete" data-filename="' . esc_attr($b['filename']) . '" style="color: #a00;">' . __('Delete', 'assertivlogix-backup-and-migration') . '</button>';
                $html .= '</td>';
                $html .= '</tr>';
            }
            $html .= '</tbody></table>';
        }

        wp_send_json_success($html);
    }

    public function ajax_restore_backup() {
        check_ajax_referer('assertivlogix_backup_nonce');
        if (!current_user_can('manage_options')) wp_send_json_error('Unauthorized');

        $filename = sanitize_text_field($_POST['filename']);
        $restore = new Restore();
        $result = $restore->run($filename);

        if (is_wp_error($result)) {
            wp_send_json_error($result->get_error_message());
        } else {
            wp_send_json_success();
        }
    }

    public function ajax_delete_backup() {
        check_ajax_referer('assertivlogix_backup_nonce');
        if (!current_user_can('manage_options')) wp_send_json_error('Unauthorized');

        $filename = sanitize_text_field($_POST['filename']);
        $backup = new Backup();
        $result = $backup->delete($filename);

        if (is_wp_error($result)) {
            wp_send_json_error($result->get_error_message());
        } else {
            wp_send_json_success();
        }
    }

    public function ajax_save_settings() {
        check_ajax_referer('assertivlogix_backup_nonce');
        if (!current_user_can('manage_options')) wp_send_json_error('Unauthorized');

        $auto_update = isset($_POST['auto_update']) && $_POST['auto_update'] === 'true' ? '1' : '0';
        $anonymise   = isset($_POST['anonymise']) && $_POST['anonymise'] === 'true' ? '1' : '0';
        $encrypt     = isset($_POST['encrypt']) && $_POST['encrypt'] === 'true' ? '1' : '0';

        update_option('assertivlogix_backup_auto_update', $auto_update);
        update_option('assertivlogix_backup_anonymise', $anonymise);
        update_option('assertivlogix_backup_encrypt', $encrypt);

        wp_send_json_success();
    }
    
    // Migration AJAX Handlers
    
    public function ajax_upload_migrate() {
        check_ajax_referer('assertivlogix_backup_nonce');
        if (!current_user_can('manage_options')) wp_send_json_error('Unauthorized');
        
        // Check if file was uploaded
        if (!isset($_FILES['backup_file']) || $_FILES['backup_file']['error'] !== UPLOAD_ERR_OK) {
            wp_send_json_error(__('No file uploaded or upload error occurred.', 'assertivlogix-backup-and-migration'));
        }
        
        $file = $_FILES['backup_file'];
        
        // Validate file type
        if (pathinfo($file['name'], PATHINFO_EXTENSION) !== 'zip') {
            wp_send_json_error(__('Only ZIP files are allowed.', 'assertivlogix-backup-and-migration'));
        }
        
        // Validate file size (512MB max)
        $max_size = 512 * 1024 * 1024; // 512MB in bytes
        if ($file['size'] > $max_size) {
            wp_send_json_error(__('File size exceeds 512MB limit.', 'assertivlogix-backup-and-migration'));
        }
        
        // Get backup directory
        $upload_dir = wp_upload_dir();
        $backup_dir = $upload_dir['basedir'] . '/assertivlogix-backups/';
        
        // Create directory if it doesn't exist
        if (!file_exists($backup_dir)) {
            wp_mkdir_p($backup_dir);
        }
        
        // Generate unique filename
        $filename = 'uploaded-' . date('Y-m-d-His') . '-' . sanitize_file_name($file['name']);
        $destination = $backup_dir . $filename;
        
        // Move uploaded file
        if (!move_uploaded_file($file['tmp_name'], $destination)) {
            wp_send_json_error(__('Failed to move uploaded file.', 'assertivlogix-backup-and-migration'));
        }
        
        // Now restore the backup
        $restore = new Restore();
        $result = $restore->run($filename);
        
        if (is_wp_error($result)) {
            // Delete the uploaded file if restore failed
            @unlink($destination);
            wp_send_json_error($result->get_error_message());
        } else {
            wp_send_json_success([
                'message' => __('Backup uploaded and migrated successfully! The page will reload in 3 seconds.', 'assertivlogix-backup-and-migration'),
                'filename' => $filename
            ]);
        }
    }
    
    public function ajax_search_replace() {
        check_ajax_referer('assertivlogix_backup_nonce');
        if (!current_user_can('manage_options')) wp_send_json_error('Unauthorized');
        
        $search = sanitize_text_field($_POST['search']);
        $replace = sanitize_text_field($_POST['replace']);
        
        $migration = new Migration();
        $result = $migration->search_replace($search, $replace);
        
        if (is_wp_error($result)) {
            wp_send_json_error($result->get_error_message());
        } else {
            $total_rows = array_sum(array_column($result, 'rows'));
            wp_send_json_success([
                'message' => sprintf(__('Replaced %d occurrences across %d locations.', 'assertivlogix-backup-and-migration'), $total_rows, count($result)),
                'details' => $result
            ]);
        }
    }
    
    public function ajax_cleanup() {
        check_ajax_referer('assertivlogix_backup_nonce');
        if (!current_user_can('manage_options')) wp_send_json_error('Unauthorized');
        
        $migration = new Migration();
        $tasks = $migration->cleanup_after_migration();
        
        wp_send_json_success([
            'message' => sprintf(__('Completed %d cleanup tasks.', 'assertivlogix-backup-and-migration'), count($tasks)),
            'tasks' => $tasks
        ]);
    }
    
    public function ajax_selective_backup() {
        check_ajax_referer('assertivlogix_backup_nonce');
        if (!current_user_can('manage_options')) wp_send_json_error('Unauthorized');
        
        $options = [
            'plugins' => isset($_POST['plugins']) ? json_decode(stripslashes($_POST['plugins']), true) : [],
            'themes' => isset($_POST['themes']) ? json_decode(stripslashes($_POST['themes']), true) : [],
            'tables' => isset($_POST['tables']) ? json_decode(stripslashes($_POST['tables']), true) : [],
            'uploads' => isset($_POST['uploads']) && $_POST['uploads'] === 'true'
        ];
        
        $migration = new Migration();
        $result = $migration->selective_export($options);
        
        if (is_wp_error($result)) {
            wp_send_json_error($result->get_error_message());
        } else {
            wp_send_json_success(['filename' => $result]);
        }
    }
    
    public function ajax_get_plugins() {
        check_ajax_referer('assertivlogix_backup_nonce');
        if (!current_user_can('manage_options')) wp_send_json_error('Unauthorized');
        
        $migration = new Migration();
        $plugins = $migration->get_installed_plugins();
        
        wp_send_json_success($plugins);
    }
    
    public function ajax_get_themes() {
        check_ajax_referer('assertivlogix_backup_nonce');
        if (!current_user_can('manage_options')) wp_send_json_error('Unauthorized');
        
        $migration = new Migration();
        $themes = $migration->get_installed_themes();
        
        wp_send_json_success($themes);
    }
    
    public function ajax_get_tables() {
        check_ajax_referer('assertivlogix_backup_nonce');
        if (!current_user_can('manage_options')) wp_send_json_error('Unauthorized');
        
        $migration = new Migration();
        $tables = $migration->get_database_tables();
        
        wp_send_json_success($tables);
    }
}
?>
