<?php
namespace AssertivlogixBackup;

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

/**
 * Admin class handling the plugin settings page and UI.
 */
class Admin {
    /**
     * Initialize admin hooks.
     */
    public function init() {
        add_action('admin_menu', [$this, 'add_admin_menu']);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_assets']);
        
        // AJAX actions
        add_action('wp_ajax_assertivlogix_backup_run', [$this, 'ajax_run_backup']);
        add_action('wp_ajax_assertivlogix_backup_save_schedule', [$this, 'ajax_save_schedule']);
        add_action('wp_ajax_assertivlogix_backup_list', [$this, 'ajax_list_backups']);
        add_action('wp_ajax_assertivlogix_backup_restore', [$this, 'ajax_restore_backup']);
        add_action('wp_ajax_assertivlogix_backup_delete', [$this, 'ajax_delete_backup']);
        add_action('wp_ajax_assertivlogix_backup_save_settings', [$this, 'ajax_save_settings']);
        
        // Migration actions
        add_action('wp_ajax_assertivlogix_backup_upload_migrate', [$this, 'ajax_upload_migrate']);
        add_action('wp_ajax_assertivlogix_backup_search_replace', [$this, 'ajax_search_replace']);
        add_action('wp_ajax_assertivlogix_backup_cleanup', [$this, 'ajax_cleanup']);
        add_action('wp_ajax_assertivlogix_backup_selective', [$this, 'ajax_selective_backup']);
        add_action('wp_ajax_assertivlogix_backup_get_plugins', [$this, 'ajax_get_plugins']);
        add_action('wp_ajax_assertivlogix_backup_get_themes', [$this, 'ajax_get_themes']);
        add_action('wp_ajax_assertivlogix_backup_get_tables', [$this, 'ajax_get_tables']);
        
        // License management
        add_action('wp_ajax_assertivlogix_backup_validate_license', [$this, 'ajax_validate_license']);
        add_action('wp_ajax_assertivlogix_backup_deactivate_license', [$this, 'ajax_deactivate_license']);
        
        // API Key management (for site-to-site migration)
        add_action('wp_ajax_assertivlogix_backup_regenerate_api_key', [$this, 'ajax_regenerate_api_key']);
        
        // Cloud storage settings
        add_action('wp_ajax_assertivlogix_backup_save_cloud_settings', [$this, 'ajax_save_cloud_settings']);
        add_action('wp_ajax_assertivlogix_backup_test_cloud_upload', [$this, 'ajax_test_cloud_upload']);
    }

    /**
     * Register the plugin menu page.
     */
    public function add_admin_menu() {
        add_menu_page(
            __('Assertivlogix Backup & Migration', 'assertivlogix-backup-and-migration'),
            __('Assertivlogix Backup', 'assertivlogix-backup-and-migration'),
            'manage_options',
            'assertivlogix-backup',
            [$this, 'render_settings_page'],
            'dashicons-backup',
            80
        );
        
        add_submenu_page(
            'assertivlogix-backup',
            __('How Assertivlogix Backup Works', 'assertivlogix-backup-and-migration'),
            __('How it works', 'assertivlogix-backup-and-migration'),
            'manage_options',
            'assertivlogix-backup-how-it-works',
            [$this, 'render_how_it_works']
        );
    }

    /**
     * Enqueue CSS/JS for the admin UI.
     */
    public function enqueue_assets($hook) {
        if (strpos($hook, 'assertivlogix-backup') === false) {
            return;
        }
        
        // Get active tab
        $active_tab = isset($_GET['tab']) ? $_GET['tab'] : 'backup';
        
        // Base admin CSS (always loaded)
        wp_enqueue_style(
            'assertivlogix-backup-admin-base',
            ASSERTIVLOGIX_BM_PLUGIN_URL . 'assets/css/admin-base.css',
            [],
            ASSERTIVLOGIX_BM_VERSION
        );
        
        // Page-specific CSS
        $page_css_map = [
            'backup' => 'admin-backup.css',
            'migration' => 'admin-migration.css',
            'storage' => 'admin-storage.css',
            'settings' => 'admin-settings.css',
        ];
        
        if (isset($page_css_map[$active_tab])) {
            wp_enqueue_style(
                'assertivlogix-backup-admin-' . $active_tab,
                ASSERTIVLOGIX_BM_PLUGIN_URL . 'assets/css/' . $page_css_map[$active_tab],
                ['assertivlogix-backup-admin-base'],
                ASSERTIVLOGIX_BM_VERSION
            );
        }
        
        // Modal CSS (always loaded for storage tab)
        wp_enqueue_style(
            'assertivlogix-backup-modal',
            ASSERTIVLOGIX_BM_PLUGIN_URL . 'assets/css/admin-modal.css',
            ['assertivlogix-backup-admin-base'],
            ASSERTIVLOGIX_BM_VERSION
        );
        
        // JavaScript
        wp_enqueue_script(
            'assertivlogix-backup-admin',
            ASSERTIVLOGIX_BM_PLUGIN_URL . 'assets/js/admin.js',
            ['jquery'],
            ASSERTIVLOGIX_BM_VERSION,
            true
        );
        
        // Localize script for AJAX actions.
        $license_info = class_exists('AssertivlogixBackup\\License') ? \AssertivlogixBackup\License::get_info() : ['is_valid' => false];
        wp_localize_script('assertivlogix-backup-admin', 'assertivlogixBackup', [
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'nonce'   => wp_create_nonce('assertivlogix_backup_nonce'),
            'license' => $license_info,
            'debug'   => defined('WP_DEBUG') && WP_DEBUG,
            'strings' => [
                'confirmRestore' => __('Are you sure you want to restore this backup? This will overwrite your current site content and database.', 'assertivlogix-backup-and-migration'),
                'backupSuccess' => __('Backup completed successfully!', 'assertivlogix-backup-and-migration'),
                'restoreSuccess' => __('Restore completed successfully!', 'assertivlogix-backup-and-migration'),
            ]
        ]);
    }

    /**
     * Render the plugin settings page.
     */
    public function render_settings_page() {
        $active_tab = isset($_GET['tab']) ? $_GET['tab'] : 'backup';
        $backup = new Backup();
        $backups = $backup->get_backups();
        $total_backups = count($backups);
        $total_size = 0;
        foreach ($backups as $b) {
            if (isset($b['path']) && file_exists($b['path'])) {
                $total_size += filesize($b['path']);
            }
        }
        $license_info = class_exists('AssertivlogixBackup\\License') ? \AssertivlogixBackup\License::get_info() : ['is_valid' => false];
        $is_license_valid = isset($license_info['is_valid']) && $license_info['is_valid'];
        ?>
        <!-- Connection Modal - Moved outside wrap to prevent CSS conflicts -->
        <div id="assertivlogix-storage-modal" class="assertivlogix-modal-overlay" style="display: none;">
            <div class="assertivlogix-modal assertivlogix-storage-modal">
                <div class="assertivlogix-modal-header">
                    <h3 id="storage-modal-title"><?php _e('Connect Storage', 'assertivlogix-backup-and-migration'); ?></h3>
                    <button class="assertivlogix-modal-close">&times;</button>
                </div>
                <div class="assertivlogix-modal-body">
                    <form id="storage-connection-form">
                        <!-- Google Drive Form -->
                        <div id="storage-form-google-drive" style="display: none;">
                            <div class="assertivlogix-credential-guide">
                                <h4><span class="dashicons dashicons-info"></span> <?php _e('How to Get Google Drive Access Token', 'assertivlogix-backup-and-migration'); ?></h4>
                                <ol>
                                    <li><?php _e('Go to', 'assertivlogix-backup-and-migration'); ?> <a href="https://console.cloud.google.com/apis/credentials" target="_blank"><?php _e('Google Cloud Console', 'assertivlogix-backup-and-migration'); ?></a></li>
                                    <li><?php _e('Create a new project or select an existing one', 'assertivlogix-backup-and-migration'); ?></li>
                                    <li><?php _e('Enable the Google Drive API', 'assertivlogix-backup-and-migration'); ?></li>
                                    <li><?php _e('Go to Credentials → Create Credentials → OAuth client ID', 'assertivlogix-backup-and-migration'); ?></li>
                                    <li><?php _e('Set Application type to "Web application"', 'assertivlogix-backup-and-migration'); ?></li>
                                    <li><?php _e('Add authorized redirect URI: ', 'assertivlogix-backup-and-migration'); ?><code><?php echo admin_url('admin.php?page=assertivlogix-backup'); ?></code></li>
                                    <li><?php _e('Use OAuth 2.0 Playground to generate access token:', 'assertivlogix-backup-and-migration'); ?> <a href="https://developers.google.com/oauthplayground/" target="_blank"><?php _e('OAuth 2.0 Playground', 'assertivlogix-backup-and-migration'); ?></a></li>
                                    <li><?php _e('Select "Drive API v3" and authorize to get your access token', 'assertivlogix-backup-and-migration'); ?></li>
                                </ol>
                            </div>
                            <p>
                                <label for="storage-access-token"><?php _e('Access Token', 'assertivlogix-backup-and-migration'); ?> <span class="required">*</span></label>
                                <input type="text" id="storage-access-token" name="access_token" class="regular-text" placeholder="<?php esc_attr_e('Enter Google Drive access token', 'assertivlogix-backup-and-migration'); ?>" required>
                                <span class="description"><?php _e('Paste your OAuth 2.0 access token here. Note: Access tokens expire after 1 hour. For permanent access, use a refresh token.', 'assertivlogix-backup-and-migration'); ?></span>
                            </p>
                        </div>
                        
                        <!-- Dropbox Form -->
                        <div id="storage-form-dropbox" style="display: none;">
                            <div class="assertivlogix-credential-guide">
                                <h4><span class="dashicons dashicons-info"></span> <?php _e('How to Get Dropbox Access Token', 'assertivlogix-backup-and-migration'); ?></h4>
                                <ol>
                                    <li><?php _e('Go to', 'assertivlogix-backup-and-migration'); ?> <a href="https://www.dropbox.com/developers/apps" target="_blank"><?php _e('Dropbox App Console', 'assertivlogix-backup-and-migration'); ?></a></li>
                                    <li><?php _e('Click "Create app"', 'assertivlogix-backup-and-migration'); ?></li>
                                    <li><?php _e('Choose "Scoped access" → "Full Dropbox" or "App folder"', 'assertivlogix-backup-and-migration'); ?></li>
                                    <li><?php _e('Enter an app name and click "Create app"', 'assertivlogix-backup-and-migration'); ?></li>
                                    <li><?php _e('Go to the "Permissions" tab and enable: files.content.write, files.content.read', 'assertivlogix-backup-and-migration'); ?></li>
                                    <li><?php _e('Go to the "Settings" tab → "OAuth 2" section', 'assertivlogix-backup-and-migration'); ?></li>
                                    <li><?php _e('Click "Generate" under "Access token expiration" to create a long-lived token', 'assertivlogix-backup-and-migration'); ?></li>
                                    <li><?php _e('Copy the generated access token', 'assertivlogix-backup-and-migration'); ?></li>
                                </ol>
                            </div>
                            <p>
                                <label for="storage-access-token-dropbox"><?php _e('Access Token', 'assertivlogix-backup-and-migration'); ?> <span class="required">*</span></label>
                                <input type="text" id="storage-access-token-dropbox" name="access_token" class="regular-text" placeholder="<?php esc_attr_e('Enter Dropbox access token', 'assertivlogix-backup-and-migration'); ?>" required>
                                <span class="description"><?php _e('Paste your Dropbox access token here. Long-lived tokens don\'t expire.', 'assertivlogix-backup-and-migration'); ?></span>
                            </p>
                        </div>
                        
                        <!-- OneDrive Form -->
                        <div id="storage-form-onedrive" style="display: none;">
                            <div class="assertivlogix-credential-guide">
                                <h4><span class="dashicons dashicons-info"></span> <?php _e('How to Get OneDrive Access Token', 'assertivlogix-backup-and-migration'); ?></h4>
                                <ol>
                                    <li><?php _e('Go to', 'assertivlogix-backup-and-migration'); ?> <a href="https://portal.azure.com/#blade/Microsoft_AAD_RegisteredApps/ApplicationsListBlade" target="_blank"><?php _e('Azure Portal', 'assertivlogix-backup-and-migration'); ?></a></li>
                                    <li><?php _e('Click "New registration" to create a new app', 'assertivlogix-backup-and-migration'); ?></li>
                                    <li><?php _e('Enter a name and set redirect URI to:', 'assertivlogix-backup-and-migration'); ?> <code><?php echo admin_url('admin.php?page=assertivlogix-backup'); ?></code></li>
                                    <li><?php _e('Go to "API permissions" → Add permission → Microsoft Graph → Delegated permissions', 'assertivlogix-backup-and-migration'); ?></li>
                                    <li><?php _e('Add permissions: Files.ReadWrite, User.Read', 'assertivlogix-backup-and-migration'); ?></li>
                                    <li><?php _e('Go to "Certificates & secrets" → Create a client secret', 'assertivlogix-backup-and-migration'); ?></li>
                                    <li><?php _e('Use Microsoft Graph Explorer or OAuth 2.0 to get access token:', 'assertivlogix-backup-and-migration'); ?> <a href="https://developer.microsoft.com/en-us/graph/graph-explorer" target="_blank"><?php _e('Graph Explorer', 'assertivlogix-backup-and-migration'); ?></a></li>
                                    <li><?php _e('Authorize and copy the access token', 'assertivlogix-backup-and-migration'); ?></li>
                                </ol>
                            </div>
                            <p>
                                <label for="storage-access-token-onedrive"><?php _e('Access Token', 'assertivlogix-backup-and-migration'); ?> <span class="required">*</span></label>
                                <input type="text" id="storage-access-token-onedrive" name="access_token" class="regular-text" placeholder="<?php esc_attr_e('Enter OneDrive access token', 'assertivlogix-backup-and-migration'); ?>" required>
                                <span class="description"><?php _e('Paste your Microsoft Graph access token here. Access tokens typically expire after 1 hour.', 'assertivlogix-backup-and-migration'); ?></span>
                            </p>
                        </div>
                        
                        <!-- S3 Form -->
                        <div id="storage-form-s3" style="display: none;">
                            <div class="assertivlogix-credential-guide">
                                <h4><span class="dashicons dashicons-info"></span> <?php _e('How to Get Amazon S3 Credentials', 'assertivlogix-backup-and-migration'); ?></h4>
                                <ol>
                                    <li><?php _e('Log in to', 'assertivlogix-backup-and-migration'); ?> <a href="https://console.aws.amazon.com/iam/" target="_blank"><?php _e('AWS IAM Console', 'assertivlogix-backup-and-migration'); ?></a></li>
                                    <li><?php _e('Click "Users" → "Add users" to create a new IAM user', 'assertivlogix-backup-and-migration'); ?></li>
                                    <li><?php _e('Enable "Programmatic access" (Access key ID and secret access key)', 'assertivlogix-backup-and-migration'); ?></li>
                                    <li><?php _e('Attach policy: "AmazonS3FullAccess" or create custom policy with:', 'assertivlogix-backup-and-migration'); ?>
                                        <ul style="margin-left: 20px; margin-top: 8px;">
                                            <li><code>s3:PutObject</code></li>
                                            <li><code>s3:GetObject</code></li>
                                            <li><code>s3:ListBucket</code></li>
                                            <li><code>s3:DeleteObject</code></li>
                                        </ul>
                                    </li>
                                    <li><?php _e('Go to', 'assertivlogix-backup-and-migration'); ?> <a href="https://s3.console.aws.amazon.com/s3/home" target="_blank"><?php _e('S3 Console', 'assertivlogix-backup-and-migration'); ?></a> <?php _e('and create a bucket (or use existing)', 'assertivlogix-backup-and-migration'); ?></li>
                                    <li><?php _e('After creating user, click "Security credentials" tab', 'assertivlogix-backup-and-migration'); ?></li>
                                    <li><?php _e('Click "Create access key" → Choose "Application running outside AWS"', 'assertivlogix-backup-and-migration'); ?></li>
                                    <li><?php _e('Copy the Access Key ID and Secret Access Key (shown only once)', 'assertivlogix-backup-and-migration'); ?></li>
                                </ol>
                                <p style="margin-top: 16px; padding: 12px; background: #fff3cd; border-left: 4px solid #ffc107; border-radius: 4px;">
                                    <strong><?php _e('Security Tip:', 'assertivlogix-backup-and-migration'); ?></strong> <?php _e('Never share your AWS credentials. Use IAM users with minimal required permissions.', 'assertivlogix-backup-and-migration'); ?>
                                </p>
                            </div>
                            <p>
                                <label for="storage-access-key"><?php _e('Access Key ID', 'assertivlogix-backup-and-migration'); ?> <span class="required">*</span></label>
                                <input type="text" id="storage-access-key" name="access_key" class="regular-text" placeholder="<?php esc_attr_e('Enter AWS Access Key ID', 'assertivlogix-backup-and-migration'); ?>" required>
                                <span class="description"><?php _e('Your AWS IAM user Access Key ID (starts with AKIA...)', 'assertivlogix-backup-and-migration'); ?></span>
                            </p>
                            <p>
                                <label for="storage-secret-key"><?php _e('Secret Access Key', 'assertivlogix-backup-and-migration'); ?> <span class="required">*</span></label>
                                <input type="password" id="storage-secret-key" name="secret_key" class="regular-text" placeholder="<?php esc_attr_e('Enter AWS Secret Access Key', 'assertivlogix-backup-and-migration'); ?>" required>
                                <span class="description"><?php _e('Your AWS IAM user Secret Access Key (keep this secure)', 'assertivlogix-backup-and-migration'); ?></span>
                            </p>
                            <p>
                                <label for="storage-bucket"><?php _e('Bucket Name', 'assertivlogix-backup-and-migration'); ?> <span class="required">*</span></label>
                                <input type="text" id="storage-bucket" name="bucket" class="regular-text" placeholder="<?php esc_attr_e('Enter S3 bucket name', 'assertivlogix-backup-and-migration'); ?>" required>
                                <span class="description"><?php _e('The name of your S3 bucket (must be globally unique)', 'assertivlogix-backup-and-migration'); ?></span>
                            </p>
                            <p>
                                <label for="storage-region"><?php _e('Region', 'assertivlogix-backup-and-migration'); ?> <span class="required">*</span></label>
                                <select id="storage-region" name="region" class="regular-text" required>
                                    <option value="us-east-1">US East (N. Virginia) - us-east-1</option>
                                    <option value="us-east-2">US East (Ohio) - us-east-2</option>
                                    <option value="us-west-1">US West (N. California) - us-west-1</option>
                                    <option value="us-west-2">US West (Oregon) - us-west-2</option>
                                    <option value="eu-west-1">Europe (Ireland) - eu-west-1</option>
                                    <option value="eu-west-2">Europe (London) - eu-west-2</option>
                                    <option value="eu-central-1">Europe (Frankfurt) - eu-central-1</option>
                                    <option value="ap-southeast-1">Asia Pacific (Singapore) - ap-southeast-1</option>
                                    <option value="ap-southeast-2">Asia Pacific (Sydney) - ap-southeast-2</option>
                                    <option value="ap-northeast-1">Asia Pacific (Tokyo) - ap-northeast-1</option>
                                </select>
                                <span class="description"><?php _e('The AWS region where your bucket is located', 'assertivlogix-backup-and-migration'); ?></span>
                            </p>
                            <p>
                                <label for="storage-folder"><?php _e('Folder (optional)', 'assertivlogix-backup-and-migration'); ?></label>
                                <input type="text" id="storage-folder" name="folder" class="regular-text" placeholder="<?php esc_attr_e('e.g., backups/', 'assertivlogix-backup-and-migration'); ?>">
                                <span class="description"><?php _e('Optional folder path within the bucket (e.g., "backups/" or "wordpress-backups/")', 'assertivlogix-backup-and-migration'); ?></span>
                            </p>
                        </div>
                        
                        <input type="hidden" id="storage-provider" name="provider" value="">
                    </form>
                </div>
                <div class="assertivlogix-modal-footer">
                    <button class="button button-primary" id="btn-save-storage-connection"><?php _e('Connect', 'assertivlogix-backup-and-migration'); ?></button>
                    <button class="button assertivlogix-modal-cancel"><?php _e('Cancel', 'assertivlogix-backup-and-migration'); ?></button>
                </div>
            </div>
        </div>

        <div class="wrap assertivlogix-backup-wrapper">
            <!-- Premium Hero Banner -->
            <div class="assertivlogix-hero-banner">
                <div class="assertivlogix-hero-content">
                    <div class="assertivlogix-hero-text">
                        <h1><?php esc_html_e('Assertivlogix Backup & Migration', 'assertivlogix-backup-and-migration'); ?></h1>
                        <p class="assertivlogix-hero-subtitle"><?php esc_html_e('Professional backup and migration solution for WordPress', 'assertivlogix-backup-and-migration'); ?></p>
                    </div>
                    <?php if (!$is_license_valid): ?>
                    <div class="assertivlogix-hero-cta">
                        <a href="https://plugins.assertivlogix.com/plugin/assertivlogix-backup-pro" target="_blank" class="assertivlogix-premium-btn">
                            <span class="dashicons dashicons-star-filled"></span>
                            <?php _e('Upgrade to Pro', 'assertivlogix-backup-and-migration'); ?>
                        </a>
                    </div>
                    <?php else: ?>
                    <div class="assertivlogix-hero-badge assertivlogix-license-activated">
                        <span class="dashicons dashicons-yes-alt"></span>
                        <div>
                            <strong><?php _e('License Activated', 'assertivlogix-backup-and-migration'); ?></strong>
                            <?php if (!empty($license_info['expires']) && $license_info['expires'] > 0): ?>
                                <div class="assertivlogix-license-expires">
                                    <?php printf(__('Expires: %s', 'assertivlogix-backup-and-migration'), date('M d, Y', $license_info['expires'])); ?>
                                </div>
                            <?php endif; ?>
                        </div>
                    </div>
                    <?php endif; ?>
                </div>
            </div>

            <!-- Stats Dashboard -->
            <div class="assertivlogix-stats-grid">
                <div class="assertivlogix-stat-card">
                    <div class="assertivlogix-stat-icon" style="background: linear-gradient(135deg, #3b82f6 0%, #2563eb 100%);">
                        <span class="dashicons dashicons-database"></span>
                    </div>
                    <div class="assertivlogix-stat-content">
                        <div class="assertivlogix-stat-value"><?php echo esc_html($total_backups); ?></div>
                        <div class="assertivlogix-stat-label"><?php _e('Total Backups', 'assertivlogix-backup-and-migration'); ?></div>
                    </div>
                </div>
                <div class="assertivlogix-stat-card">
                    <div class="assertivlogix-stat-icon" style="background: linear-gradient(135deg, #10b981 0%, #059669 100%);">
                        <span class="dashicons dashicons-media-archive"></span>
                    </div>
                    <div class="assertivlogix-stat-content">
                        <div class="assertivlogix-stat-value"><?php echo esc_html(size_format($total_size, 2)); ?></div>
                        <div class="assertivlogix-stat-label"><?php _e('Total Size', 'assertivlogix-backup-and-migration'); ?></div>
                    </div>
                </div>
                <div class="assertivlogix-stat-card">
                    <div class="assertivlogix-stat-icon" style="background: linear-gradient(135deg, #f59e0b 0%, #d97706 100%);">
                        <span class="dashicons dashicons-calendar-alt"></span>
                    </div>
                    <div class="assertivlogix-stat-content">
                        <div class="assertivlogix-stat-value"><?php echo esc_html(ucfirst(get_option('assertivlogix_backup_schedule', 'none'))); ?></div>
                        <div class="assertivlogix-stat-label"><?php _e('Schedule', 'assertivlogix-backup-and-migration'); ?></div>
                    </div>
                </div>
                <div class="assertivlogix-stat-card">
                    <div class="assertivlogix-stat-icon" style="background: linear-gradient(135deg, #8b5cf6 0%, #7c3aed 100%);">
                        <span class="dashicons dashicons-cloud"></span>
                    </div>
                    <div class="assertivlogix-stat-content">
                        <div class="assertivlogix-stat-value"><?php echo $is_license_valid ? __('Active', 'assertivlogix-backup-and-migration') : __('Free', 'assertivlogix-backup-and-migration'); ?></div>
                        <div class="assertivlogix-stat-label"><?php _e('License Status', 'assertivlogix-backup-and-migration'); ?></div>
                    </div>
                </div>
            </div>
            
            <nav class="nav-tab-wrapper assertivlogix-nav-tabs">
                <a href="?page=assertivlogix-backup&tab=backup" class="nav-tab <?php echo $active_tab == 'backup' ? 'nav-tab-active' : ''; ?>">
                    <span class="dashicons dashicons-backup" style="font-size: 16px; width: 16px; height: 16px; vertical-align: middle; margin-right: 6px;"></span>
                    <?php _e('Backup / Restore', 'assertivlogix-backup-and-migration'); ?>
                </a>
                <a href="?page=assertivlogix-backup&tab=migration" class="nav-tab <?php echo $active_tab == 'migration' ? 'nav-tab-active' : ''; ?>">
                    <span class="dashicons dashicons-update" style="font-size: 16px; width: 16px; height: 16px; vertical-align: middle; margin-right: 6px;"></span>
                    <?php _e('Migration', 'assertivlogix-backup-and-migration'); ?>
                </a>
                <a href="?page=assertivlogix-backup&tab=storage" class="nav-tab <?php echo $active_tab == 'storage' ? 'nav-tab-active' : ''; ?>">
                    <span class="dashicons dashicons-cloud" style="font-size: 16px; width: 16px; height: 16px; vertical-align: middle; margin-right: 6px;"></span>
                    <?php _e('Storage', 'assertivlogix-backup-and-migration'); ?>
                </a>
                <a href="?page=assertivlogix-backup&tab=settings" class="nav-tab <?php echo $active_tab == 'settings' ? 'nav-tab-active' : ''; ?>">
                    <span class="dashicons dashicons-admin-settings" style="font-size: 16px; width: 16px; height: 16px; vertical-align: middle; margin-right: 6px;"></span>
                    <?php _e('Settings', 'assertivlogix-backup-and-migration'); ?>
                </a>
            </nav>

            <div class="assertivlogix-backup-dashboard">
                <?php if ($active_tab == 'backup'): ?>
                    <?php if (!\AssertivlogixBackup\Backup::is_zip_available()): ?>
                        <div class="notice notice-error" style="margin: 20px 20px 0 0; padding: 20px; border-left: 4px solid #dc3232;">
                            <p style="margin: 0; font-size: 16px; font-weight: 600; color: #dc3232;">
                                <span class="dashicons dashicons-warning" style="font-size: 20px; width: 20px; height: 20px; vertical-align: middle; margin-right: 8px;"></span>
                                <?php _e('PHP ZipArchive Extension Not Available', 'assertivlogix-backup-and-migration'); ?>
                            </p>
                            <p style="margin: 12px 0 0 0; font-size: 14px; line-height: 1.6;">
                                <?php _e('The PHP ZipArchive extension is required to create backups. Please contact your hosting provider to enable the PHP zip extension.', 'assertivlogix-backup-and-migration'); ?>
                            </p>
                            <p style="margin: 12px 0 0 0; font-size: 14px; line-height: 1.6;">
                                <strong><?php _e('How to enable:', 'assertivlogix-backup-and-migration'); ?></strong><br>
                                <?php _e('1. Contact your hosting provider and ask them to enable the PHP zip extension', 'assertivlogix-backup-and-migration'); ?><br>
                                <?php _e('2. Or edit your php.ini file and add/ensure this line: extension=zip', 'assertivlogix-backup-and-migration'); ?><br>
                                <?php _e('3. Restart your web server after making changes', 'assertivlogix-backup-and-migration'); ?>
                            </p>
                            <p style="margin: 12px 0 0 0; font-size: 13px; color: #666;">
                                <a href="https://www.php.net/manual/en/zip.installation.php" target="_blank" style="text-decoration: none;">
                                    <?php _e('Learn more about PHP ZipArchive installation →', 'assertivlogix-backup-and-migration'); ?>
                                </a>
                            </p>
                        </div>
                    <?php endif; ?>
                    <section class="assertivlogix-backup-card" id="backup-now">
                        <h2><?php esc_html_e('Back Up Now', 'assertivlogix-backup-and-migration'); ?></h2>
                        <p><?php esc_html_e('Create an immediate backup of your site files and database.', 'assertivlogix-backup-and-migration'); ?></p>
                        <button class="button button-primary" id="btn-backup-now" <?php echo !\AssertivlogixBackup\Backup::is_zip_available() ? 'disabled' : ''; ?>>
                            <?php esc_html_e('Run Backup', 'assertivlogix-backup-and-migration'); ?>
                        </button>
                        <?php if (!\AssertivlogixBackup\Backup::is_zip_available()): ?>
                            <p class="description" style="color: #dc3232; margin-top: 12px;">
                                <span class="dashicons dashicons-warning" style="font-size: 16px; width: 16px; height: 16px; vertical-align: middle; margin-right: 4px;"></span>
                                <?php _e('Backup is disabled because PHP ZipArchive extension is not available.', 'assertivlogix-backup-and-migration'); ?>
                            </p>
                        <?php endif; ?>
                    </section>
                    <section class="assertivlogix-backup-card" id="schedule-backup">
                        <h2><?php esc_html_e('Schedule Backups', 'assertivlogix-backup-and-migration'); ?></h2>
                        <p><?php esc_html_e('Choose a schedule interval for automatic backups.', 'assertivlogix-backup-and-migration'); ?></p>
                        <div style="display: flex; flex-wrap: wrap; gap: 12px; align-items: center;">
                            <select id="backup-interval" class="regular-text">
                                <option value="2" <?php selected(get_option('assertivlogix_backup_schedule'), '2'); ?>><?php esc_html_e('Every 2 hours', 'assertivlogix-backup-and-migration'); ?></option>
                                <option value="4" <?php selected(get_option('assertivlogix_backup_schedule'), '4'); ?>><?php esc_html_e('Every 4 hours', 'assertivlogix-backup-and-migration'); ?></option>
                                <option value="8" <?php selected(get_option('assertivlogix_backup_schedule'), '8'); ?>><?php esc_html_e('Every 8 hours', 'assertivlogix-backup-and-migration'); ?></option>
                                <option value="12" <?php selected(get_option('assertivlogix_backup_schedule'), '12'); ?>><?php esc_html_e('Every 12 hours', 'assertivlogix-backup-and-migration'); ?></option>
                                <option value="daily" <?php selected(get_option('assertivlogix_backup_schedule'), 'daily'); ?>><?php esc_html_e('Daily', 'assertivlogix-backup-and-migration'); ?></option>
                                <option value="weekly" <?php selected(get_option('assertivlogix_backup_schedule'), 'weekly'); ?>><?php esc_html_e('Weekly', 'assertivlogix-backup-and-migration'); ?></option>
                                <option value="fortnightly" <?php selected(get_option('assertivlogix_backup_schedule'), 'fortnightly'); ?>><?php esc_html_e('Fortnightly', 'assertivlogix-backup-and-migration'); ?></option>
                                <option value="monthly" <?php selected(get_option('assertivlogix_backup_schedule'), 'monthly'); ?>><?php esc_html_e('Monthly', 'assertivlogix-backup-and-migration'); ?></option>
                            </select>
                            <button class="button button-primary" id="btn-save-schedule"><?php esc_html_e('Save Schedule', 'assertivlogix-backup-and-migration'); ?></button>
                        </div>
                    </section>
                    <section class="assertivlogix-backup-card" id="restore-backup">
                        <h2><?php esc_html_e('Existing Backups', 'assertivlogix-backup-and-migration'); ?></h2>
                        <p><?php esc_html_e('Download or restore previous backups.', 'assertivlogix-backup-and-migration'); ?></p>
                        <div id="backup-list">
                            <!-- Dynamically populated via AJAX -->
                            <p class="description"><?php _e('Loading backups...', 'assertivlogix-backup-and-migration'); ?></p>
                        </div>
                    </section>
                
                <?php elseif ($active_tab == 'migration'): ?>
                    
                    <!-- Migration Header -->
                    <div class="assertivlogix-migration-header">
                        <h2><?php esc_html_e('Site Migration Tools', 'assertivlogix-backup-and-migration'); ?></h2>
                        <p><?php esc_html_e('Migrate your WordPress site with ease using our comprehensive migration tools.', 'assertivlogix-backup-and-migration'); ?></p>
                    </div>
                    
                    <!-- Basic Migration -->
                    <section class="assertivlogix-backup-card assertivlogix-migration-card">
                        <div class="assertivlogix-feature-header">
                            <div class="assertivlogix-feature-icon-inline" style="background: linear-gradient(135deg, #3b82f6 0%, #2563eb 100%);">
                                <span class="dashicons dashicons-update"></span>
                            </div>
                            <div>
                                <h2><?php esc_html_e('Basic Migration', 'assertivlogix-backup-and-migration'); ?></h2>
                                <p class="assertivlogix-feature-description"><?php esc_html_e('Upload a backup file from your source site and restore it here.', 'assertivlogix-backup-and-migration'); ?></p>
                            </div>
                        </div>
                        
                        <form id="assertivlogix-upload-backup-form" enctype="multipart/form-data">
                            <div class="assertivlogix-upload-area">
                                <input type="file" id="backup-file-upload" name="backup_file" accept=".zip" style="display: none;" />
                                <div class="assertivlogix-upload-box" id="upload-box">
                                    <span class="dashicons dashicons-upload"></span>
                                    <p><strong><?php _e('Click to upload', 'assertivlogix-backup-and-migration'); ?></strong> <?php _e('or drag and drop', 'assertivlogix-backup-and-migration'); ?></p>
                                    <p class="description"><?php _e('ZIP file only (Max size: 512MB)', 'assertivlogix-backup-and-migration'); ?></p>
                                </div>
                                <div id="file-info" class="assertivlogix-file-info" style="display: none;">
                                    <span class="dashicons dashicons-media-archive"></span>
                                    <span id="file-name"></span>
                                    <span id="file-size"></span>
                                    <button type="button" id="remove-file" class="button"><?php _e('Remove', 'assertivlogix-backup-and-migration'); ?></button>
                                </div>
                            </div>
                            
                            <div class="assertivlogix-migration-actions">
                                <button type="submit" class="button button-primary button-large" id="btn-upload-migrate" disabled>
                                    <span class="dashicons dashicons-upload"></span>
                                    <?php esc_html_e('Upload & Migrate', 'assertivlogix-backup-and-migration'); ?>
                                </button>
                            </div>
                        </form>
                        
                        <div class="assertivlogix-migration-steps">
                            <h3><?php _e('How it works:', 'assertivlogix-backup-and-migration'); ?></h3>
                            <ol class="assertivlogix-steps-list">
                                <li><?php _e('Create a backup on your source site (Backup/Restore tab)', 'assertivlogix-backup-and-migration'); ?></li>
                                <li><?php _e('Download the backup ZIP file', 'assertivlogix-backup-and-migration'); ?></li>
                                <li><?php _e('Upload the ZIP file here', 'assertivlogix-backup-and-migration'); ?></li>
                                <li><?php _e('Click "Upload & Migrate" to restore', 'assertivlogix-backup-and-migration'); ?></li>
                            </ol>
                        </div>
                    </section>

                    <!-- Migration Tools Grid -->
                    <div class="assertivlogix-migration-tools-grid">
                        <!-- Search and Replace -->
                        <section class="assertivlogix-backup-card assertivlogix-migration-tool-card">
                            <div class="assertivlogix-feature-header">
                                <div class="assertivlogix-feature-icon-inline" style="background: linear-gradient(135deg, #10b981 0%, #059669 100%);">
                                    <span class="dashicons dashicons-search"></span>
                                </div>
                                <div>
                                    <h2><?php esc_html_e('Search & Replace', 'assertivlogix-backup-and-migration'); ?></h2>
                                    <p class="assertivlogix-feature-description"><?php esc_html_e('Find and replace text in your database. Useful for changing URLs after migration.', 'assertivlogix-backup-and-migration'); ?></p>
                                </div>
                            </div>
                            
                            <div class="assertivlogix-form-row">
                                <div class="assertivlogix-form-col">
                                    <label for="search-text"><?php _e('Search for:', 'assertivlogix-backup-and-migration'); ?></label>
                                    <input type="text" id="search-text" class="regular-text" placeholder="https://oldsite.com" />
                                </div>
                                <div class="assertivlogix-form-col">
                                    <label for="replace-text"><?php _e('Replace with:', 'assertivlogix-backup-and-migration'); ?></label>
                                    <input type="text" id="replace-text" class="regular-text" placeholder="https://newsite.com" />
                                </div>
                            </div>
                            
                            <div class="assertivlogix-warning-box">
                                <span class="dashicons dashicons-warning"></span>
                                <span><?php _e('Warning: This will modify your database. Create a backup first!', 'assertivlogix-backup-and-migration'); ?></span>
                            </div>
                            
                            <button class="button button-primary" id="btn-search-replace">
                                <span class="dashicons dashicons-search"></span>
                                <?php esc_html_e('Run Search & Replace', 'assertivlogix-backup-and-migration'); ?>
                            </button>
                        </section>

                        <!-- Migration Cleanup -->
                        <section class="assertivlogix-backup-card assertivlogix-migration-tool-card">
                            <div class="assertivlogix-feature-header">
                                <div class="assertivlogix-feature-icon-inline" style="background: linear-gradient(135deg, #f59e0b 0%, #d97706 100%);">
                                    <span class="dashicons dashicons-admin-tools"></span>
                                </div>
                                <div>
                                    <h2><?php esc_html_e('Migration Cleanup', 'assertivlogix-backup-and-migration'); ?></h2>
                                    <p class="assertivlogix-feature-description"><?php esc_html_e('Run cleanup tasks after migration to ensure everything works correctly.', 'assertivlogix-backup-and-migration'); ?></p>
                                </div>
                            </div>
                            
                            <ul class="assertivlogix-cleanup-tasks">
                                <li><span class="dashicons dashicons-yes"></span> Flush rewrite rules</li>
                                <li><span class="dashicons dashicons-yes"></span> Clear WordPress cache</li>
                                <li><span class="dashicons dashicons-yes"></span> Regenerate .htaccess</li>
                                <li><span class="dashicons dashicons-yes"></span> Synchronize site URLs</li>
                                <li><span class="dashicons dashicons-yes"></span> Clear expired transients</li>
                            </ul>
                            
                            <button class="button button-primary" id="btn-cleanup">
                                <span class="dashicons dashicons-admin-tools"></span>
                                <?php esc_html_e('Run Cleanup Tasks', 'assertivlogix-backup-and-migration'); ?>
                            </button>
                        </section>

                        <!-- Selective Migration -->
                        <section class="assertivlogix-backup-card assertivlogix-migration-tool-card">
                            <div class="assertivlogix-feature-header">
                                <div class="assertivlogix-feature-icon-inline" style="background: linear-gradient(135deg, #8b5cf6 0%, #7c3aed 100%);">
                                    <span class="dashicons dashicons-filter"></span>
                                </div>
                                <div>
                                    <h2><?php esc_html_e('Selective Migration', 'assertivlogix-backup-and-migration'); ?></h2>
                                    <p class="assertivlogix-feature-description"><?php esc_html_e('Choose exactly what to include in your backup.', 'assertivlogix-backup-and-migration'); ?></p>
                                </div>
                            </div>
                            
                            <div class="assertivlogix-selective-options">
                                <div class="assertivlogix-option-group">
                                    <h4><?php _e('Plugins', 'assertivlogix-backup-and-migration'); ?></h4>
                                    <button class="button" id="btn-select-plugins"><?php _e('Select Plugins', 'assertivlogix-backup-and-migration'); ?></button>
                                    <span id="selected-plugins-count" class="description">0 selected</span>
                                </div>
                                
                                <div class="assertivlogix-option-group">
                                    <h4><?php _e('Themes', 'assertivlogix-backup-and-migration'); ?></h4>
                                    <button class="button" id="btn-select-themes"><?php _e('Select Themes', 'assertivlogix-backup-and-migration'); ?></button>
                                    <span id="selected-themes-count" class="description">0 selected</span>
                                </div>
                                
                                <div class="assertivlogix-option-group">
                                    <h4><?php _e('Database Tables', 'assertivlogix-backup-and-migration'); ?></h4>
                                    <button class="button" id="btn-select-tables"><?php _e('Select Tables', 'assertivlogix-backup-and-migration'); ?></button>
                                    <span id="selected-tables-count" class="description">0 selected</span>
                                </div>
                                
                                <div class="assertivlogix-option-group">
                                    <label>
                                        <input type="checkbox" id="include-uploads" checked />
                                        <?php _e('Include Uploads Folder', 'assertivlogix-backup-and-migration'); ?>
                                    </label>
                                </div>
                            </div>
                            
                            <button class="button button-primary" id="btn-selective-backup">
                                <span class="dashicons dashicons-filter"></span>
                                <?php esc_html_e('Create Selective Backup', 'assertivlogix-backup-and-migration'); ?>
                            </button>
                        </section>
                    </div>

                    <?php
                    $license_info = class_exists('AssertivlogixBackup\\License') ? \AssertivlogixBackup\License::get_info() : ['is_valid' => false];
                    $is_license_valid = isset($license_info['is_valid']) && $license_info['is_valid'];
                    ?>
                    
                    <!-- Direct Site-to-Site Migration (Premium) -->
                    <?php if ($is_license_valid && class_exists('AssertivlogixBackup\\License') && \AssertivlogixBackup\License::has_feature('site_to_site')): ?>
                    <?php
                    $linked_site_url = get_option('assertivlogix_backup_linked_site_url', '');
                    $has_linked_site = !empty($linked_site_url);
                    ?>
                    <section class="assertivlogix-backup-card">
                        <div class="assertivlogix-feature-header">
                            <div class="assertivlogix-feature-icon-inline" style="background: linear-gradient(135deg, #8b5cf6 0%, #7c3aed 100%);">
                                <span class="dashicons dashicons-networking"></span>
                            </div>
                            <div>
                                <h2><?php esc_html_e('Direct Site-to-Site Migration', 'assertivlogix-backup-and-migration'); ?></h2>
                                <p class="assertivlogix-feature-description"><?php esc_html_e('Link two sites and migrate back and forth instantly.', 'assertivlogix-backup-and-migration'); ?></p>
                            </div>
                        </div>
                        
                        <?php if ($has_linked_site): ?>
                        <div class="assertivlogix-warning-box" style="background: linear-gradient(135deg, #ecfdf5 0%, #d1fae5 100%); border-left-color: var(--success); color: #065f46; margin-bottom: var(--spacing-lg);">
                            <span class="dashicons dashicons-yes-alt" style="color: var(--success);"></span>
                            <span><strong><?php _e('Site Linked:', 'assertivlogix-backup-and-migration'); ?></strong> <?php echo esc_html($linked_site_url); ?></span>
                        </div>
                        <?php endif; ?>
                        
                        <div class="assertivlogix-form-row">
                            <div class="assertivlogix-form-col">
                                <label for="linked-site-url"><?php _e('Remote Site URL:', 'assertivlogix-backup-and-migration'); ?></label>
                                <input type="url" id="linked-site-url" class="regular-text" placeholder="https://example.com" value="<?php echo esc_attr($linked_site_url); ?>" />
                                <p class="description"><?php _e('Enter the full URL of the remote WordPress site (e.g., https://example.com)', 'assertivlogix-backup-and-migration'); ?></p>
                            </div>
                            <div class="assertivlogix-form-col">
                                <label for="linked-site-api-key"><?php _e('API Key:', 'assertivlogix-backup-and-migration'); ?></label>
                                <input type="password" id="linked-site-api-key" class="regular-text" placeholder="<?php esc_attr_e('API Key', 'assertivlogix-backup-and-migration'); ?>" />
                                <p class="description">
                                    <?php _e('Get the API key from the remote site:', 'assertivlogix-backup-and-migration'); ?>
                                    <br/>
                                    <strong><?php _e('1. Go to the remote site → Settings tab', 'assertivlogix-backup-and-migration'); ?></strong>
                                    <br/>
                                    <strong><?php _e('2. Find "API Key for Remote Access" section', 'assertivlogix-backup-and-migration'); ?></strong>
                                    <br/>
                                    <strong><?php _e('3. Copy the API key and paste it here', 'assertivlogix-backup-and-migration'); ?></strong>
                                </p>
                            </div>
                        </div>
                        
                        <div class="assertivlogix-migration-actions">
                            <button class="button button-primary" id="btn-link-site">
                                <span class="dashicons dashicons-networking"></span>
                                <?php esc_html_e($has_linked_site ? 'Update Linked Site' : 'Link Site', 'assertivlogix-backup-and-migration'); ?>
                            </button>
                            <?php if ($has_linked_site): ?>
                            <button class="button button-primary" id="btn-migrate-to-linked">
                                <span class="dashicons dashicons-upload"></span>
                                <?php esc_html_e('Migrate to Linked Site', 'assertivlogix-backup-and-migration'); ?>
                            </button>
                            <?php endif; ?>
                        </div>
                    </section>
                    <?php endif; ?>
                    
                    <!-- Multisite Support (Premium) -->
                    <?php if ($is_license_valid && class_exists('AssertivlogixBackup\\License') && \AssertivlogixBackup\License::has_feature('multisite') && is_multisite()): ?>
                    <section class="assertivlogix-backup-card">
                        <div class="assertivlogix-feature-header">
                            <div class="assertivlogix-feature-icon-inline"><span class="dashicons dashicons-admin-multisite"></span></div>
                            <h2><?php esc_html_e('Multisite Migration', 'assertivlogix-backup-and-migration'); ?></h2>
                            <span class="assertivlogix-premium-badge"><?php _e('Pro', 'assertivlogix-backup-and-migration'); ?></span>
                        </div>
                        <p><?php esc_html_e('Migrate single sites into multisite networks.', 'assertivlogix-backup-and-migration'); ?></p>
                        
                        <div class="assertivlogix-form-row">
                            <div class="assertivlogix-form-col">
                                <label for="multisite-backup-file"><?php _e('Backup File:', 'assertivlogix-backup-and-migration'); ?></label>
                                <select id="multisite-backup-file" class="regular-text">
                                    <option value=""><?php _e('Select backup file...', 'assertivlogix-backup-and-migration'); ?></option>
                                </select>
                            </div>
                            <div class="assertivlogix-form-col">
                                <label for="multisite-domain"><?php _e('New Site Domain:', 'assertivlogix-backup-and-migration'); ?></label>
                                <input type="text" id="multisite-domain" class="regular-text" placeholder="newsite.example.com" />
                            </div>
                        </div>
                        <button class="button button-primary" id="btn-migrate-multisite"><?php esc_html_e('Migrate to Multisite', 'assertivlogix-backup-and-migration'); ?></button>
                    </section>
                    <?php endif; ?>
                    
                    <!-- Premium Features Info (if not licensed) -->
                    <?php if (!$is_license_valid): ?>
                    <section class="assertivlogix-backup-card assertivlogix-premium-info">
                        <div class="assertivlogix-feature-header">
                            <div class="assertivlogix-feature-icon-inline"><span class="dashicons dashicons-star-filled"></span></div>
                            <h2><?php esc_html_e('Premium Features', 'assertivlogix-backup-and-migration'); ?></h2>
                        </div>
                        <p><?php esc_html_e('Unlock advanced migration features with Assertivlogix Backup Pro:', 'assertivlogix-backup-and-migration'); ?></p>
                        
                        <ul class="assertivlogix-premium-features">
                            <li><span class="dashicons dashicons-networking"></span> <strong>Direct Site-to-Site Migration</strong> - Link two sites and migrate back and forth instantly</li>
                            <li><span class="dashicons dashicons-admin-multisite"></span> <strong>Multisite Support</strong> - Migrate single sites into multisite networks</li>
                            <li><span class="dashicons dashicons-cloud"></span> <strong>Cloud Storage</strong> - Automatic backups to Google Drive, Dropbox, S3, and more</li>
                            <li><span class="dashicons dashicons-admin-generic"></span> <strong>Advanced Scheduling</strong> - Custom backup schedules and retention policies</li>
                        </ul>
                        
                        <p><a href="?page=assertivlogix-backup&tab=settings" class="button button-large button-primary"><?php esc_html_e('Enter License Key', 'assertivlogix-backup-and-migration'); ?></a></p>
                    </section>
                    <?php endif; ?>


                <?php elseif ($active_tab == 'storage'): ?>
                    <?php
                    $license_info = class_exists('AssertivlogixBackup\\License') ? \AssertivlogixBackup\License::get_info() : ['is_valid' => false];
                    $is_license_valid = isset($license_info['is_valid']) && $license_info['is_valid'];
                    $has_cloud_storage = $is_license_valid && class_exists('AssertivlogixBackup\\License') && \AssertivlogixBackup\License::has_feature('cloud_storage');
                    $connections = get_option('assertivlogix_backup_cloud_connections', []);
                    
                    // Check if any cloud storage is connected
                    $has_connected_storage = false;
                    $connected_providers = [];
                    foreach ($connections as $provider => $connection) {
                        if (isset($connection['connected']) && $connection['connected']) {
                            $has_connected_storage = true;
                            $connected_providers[] = $provider;
                        }
                    }
                    
                    // Get cloud storage settings
                    $cloud_settings = get_option('assertivlogix_backup_cloud_settings', [
                        'auto_upload' => true,
                        'instant_backup' => false,
                        'upload_after_scheduled' => true,
                        'upload_after_manual' => true,
                        'delete_local_after_upload' => false,
                        'retention_days' => 30,
                        'upload_only_to' => [], // Which providers to upload to
                    ]);
                    ?>
                    
                    <?php if ($has_cloud_storage): ?>
                    <section class="assertivlogix-backup-card">
                        <h2><?php esc_html_e('Remote Storage Locations', 'assertivlogix-backup-and-migration'); ?></h2>
                        <p><?php esc_html_e('Connect your cloud storage accounts for automatic backup uploads.', 'assertivlogix-backup-and-migration'); ?></p>
                        
                        <div class="assertivlogix-storage-providers">
                            <!-- Google Drive -->
                            <div class="assertivlogix-storage-item">
                                <div class="assertivlogix-storage-header">
                                    <h3><span class="dashicons dashicons-google"></span> <?php _e('Google Drive', 'assertivlogix-backup-and-migration'); ?></h3>
                                    <?php if (isset($connections['google_drive']['connected']) && $connections['google_drive']['connected']): ?>
                                        <span class="assertivlogix-status-badge connected"><span class="dashicons dashicons-yes"></span> <?php _e('Connected', 'assertivlogix-backup-and-migration'); ?></span>
                                    <?php else: ?>
                                        <span class="assertivlogix-status-badge disconnected"><?php _e('Not Connected', 'assertivlogix-backup-and-migration'); ?></span>
                                    <?php endif; ?>
                                </div>
                                <p class="description"><?php _e('Store backups securely in your Google Drive account.', 'assertivlogix-backup-and-migration'); ?></p>
                                <?php if (isset($connections['google_drive']['connected']) && $connections['google_drive']['connected']): ?>
                                    <div class="assertivlogix-storage-actions">
                                        <button class="button button-secondary" data-provider="google_drive" data-action="test"><?php _e('Test Connection', 'assertivlogix-backup-and-migration'); ?></button>
                                        <button class="button" data-provider="google_drive" data-action="disconnect"><?php _e('Disconnect', 'assertivlogix-backup-and-migration'); ?></button>
                                    </div>
                                <?php else: ?>
                                    <button class="button button-primary" data-provider="google_drive" data-action="connect" id="btn-connect-google-drive"><?php _e('Connect Google Drive', 'assertivlogix-backup-and-migration'); ?></button>
                                <?php endif; ?>
                            </div>
                            
                            <!-- Dropbox -->
                            <div class="assertivlogix-storage-item">
                                <div class="assertivlogix-storage-header">
                                    <h3><span class="dashicons dashicons-cloud"></span> <?php _e('Dropbox', 'assertivlogix-backup-and-migration'); ?></h3>
                                    <?php if (isset($connections['dropbox']['connected']) && $connections['dropbox']['connected']): ?>
                                        <span class="assertivlogix-status-badge connected"><span class="dashicons dashicons-yes"></span> <?php _e('Connected', 'assertivlogix-backup-and-migration'); ?></span>
                                    <?php else: ?>
                                        <span class="assertivlogix-status-badge disconnected"><?php _e('Not Connected', 'assertivlogix-backup-and-migration'); ?></span>
                                    <?php endif; ?>
                                </div>
                                <p class="description"><?php _e('Upload backups to your Dropbox account automatically.', 'assertivlogix-backup-and-migration'); ?></p>
                                <?php if (isset($connections['dropbox']['connected']) && $connections['dropbox']['connected']): ?>
                                    <div class="assertivlogix-storage-actions">
                                        <button class="button button-secondary" data-provider="dropbox" data-action="test"><?php _e('Test Connection', 'assertivlogix-backup-and-migration'); ?></button>
                                        <button class="button" data-provider="dropbox" data-action="disconnect"><?php _e('Disconnect', 'assertivlogix-backup-and-migration'); ?></button>
                                    </div>
                                <?php else: ?>
                                    <button class="button button-primary" data-provider="dropbox" data-action="connect" id="btn-connect-dropbox"><?php _e('Connect Dropbox', 'assertivlogix-backup-and-migration'); ?></button>
                                <?php endif; ?>
                            </div>
                            
                            <!-- Amazon S3 -->
                            <div class="assertivlogix-storage-item">
                                <div class="assertivlogix-storage-header">
                                    <h3><span class="dashicons dashicons-amazon"></span> <?php _e('Amazon S3', 'assertivlogix-backup-and-migration'); ?></h3>
                                    <?php if (isset($connections['s3']['connected']) && $connections['s3']['connected']): ?>
                                        <span class="assertivlogix-status-badge connected"><span class="dashicons dashicons-yes"></span> <?php _e('Connected', 'assertivlogix-backup-and-migration'); ?></span>
                                    <?php else: ?>
                                        <span class="assertivlogix-status-badge disconnected"><?php _e('Not Connected', 'assertivlogix-backup-and-migration'); ?></span>
                                    <?php endif; ?>
                                </div>
                                <p class="description"><?php _e('Store backups in Amazon S3 buckets for scalable cloud storage.', 'assertivlogix-backup-and-migration'); ?></p>
                                <?php if (isset($connections['s3']['connected']) && $connections['s3']['connected']): ?>
                                    <div class="assertivlogix-storage-actions">
                                        <button class="button button-secondary" data-provider="s3" data-action="test"><?php _e('Test Connection', 'assertivlogix-backup-and-migration'); ?></button>
                                        <button class="button" data-provider="s3" data-action="disconnect"><?php _e('Disconnect', 'assertivlogix-backup-and-migration'); ?></button>
                                    </div>
                                <?php else: ?>
                                    <button class="button button-primary" data-provider="s3" data-action="connect" id="btn-connect-s3"><?php _e('Connect Amazon S3', 'assertivlogix-backup-and-migration'); ?></button>
                                <?php endif; ?>
                            </div>
                            
                            <!-- OneDrive -->
                            <div class="assertivlogix-storage-item">
                                <div class="assertivlogix-storage-header">
                                    <h3><span class="dashicons dashicons-cloud"></span> <?php _e('OneDrive', 'assertivlogix-backup-and-migration'); ?></h3>
                                    <?php if (isset($connections['onedrive']['connected']) && $connections['onedrive']['connected']): ?>
                                        <span class="assertivlogix-status-badge connected"><span class="dashicons dashicons-yes"></span> <?php _e('Connected', 'assertivlogix-backup-and-migration'); ?></span>
                                    <?php else: ?>
                                        <span class="assertivlogix-status-badge disconnected"><?php _e('Not Connected', 'assertivlogix-backup-and-migration'); ?></span>
                                    <?php endif; ?>
                                </div>
                                <p class="description"><?php _e('Backup to Microsoft OneDrive for seamless integration.', 'assertivlogix-backup-and-migration'); ?></p>
                                <?php if (isset($connections['onedrive']['connected']) && $connections['onedrive']['connected']): ?>
                                    <div class="assertivlogix-storage-actions">
                                        <button class="button button-secondary" data-provider="onedrive" data-action="test"><?php _e('Test Connection', 'assertivlogix-backup-and-migration'); ?></button>
                                        <button class="button" data-provider="onedrive" data-action="disconnect"><?php _e('Disconnect', 'assertivlogix-backup-and-migration'); ?></button>
                                    </div>
                                <?php else: ?>
                                    <button class="button button-primary" data-provider="onedrive" data-action="connect" id="btn-connect-onedrive"><?php _e('Connect OneDrive', 'assertivlogix-backup-and-migration'); ?></button>
                                <?php endif; ?>
                            </div>
                        </div>
                    </section>
                    
                    <?php if ($has_connected_storage): ?>
                    <!-- Cloud Storage Settings (shown when at least one provider is connected) -->
                    <section class="assertivlogix-backup-card assertivlogix-cloud-settings-card">
                        <div class="assertivlogix-feature-header">
                            <div class="assertivlogix-feature-icon-inline" style="background: linear-gradient(135deg, #10b981 0%, #059669 100%);">
                                <span class="dashicons dashicons-admin-settings"></span>
                            </div>
                            <div>
                                <h2><?php esc_html_e('Cloud Backup Settings', 'assertivlogix-backup-and-migration'); ?></h2>
                                <p class="assertivlogix-feature-description"><?php esc_html_e('Configure when and how backups are uploaded to your connected cloud storage.', 'assertivlogix-backup-and-migration'); ?></p>
                            </div>
                        </div>
                        
                        <div class="assertivlogix-cloud-settings-grid">
                            <!-- Upload Options -->
                            <div class="assertivlogix-settings-group">
                                <h3><span class="dashicons dashicons-upload"></span> <?php _e('Upload Options', 'assertivlogix-backup-and-migration'); ?></h3>
                                
                                <div class="assertivlogix-setting-item">
                                    <label>
                                        <input type="checkbox" id="cloud-auto-upload" <?php checked($cloud_settings['auto_upload'], true); ?> />
                                        <strong><?php _e('Auto-upload to Cloud', 'assertivlogix-backup-and-migration'); ?></strong>
                                    </label>
                                    <p class="description"><?php _e('Automatically upload backups to cloud storage when created.', 'assertivlogix-backup-and-migration'); ?></p>
                                </div>
                                
                                <div class="assertivlogix-setting-item">
                                    <label>
                                        <input type="checkbox" id="cloud-instant-backup" <?php checked($cloud_settings['instant_backup'], true); ?> />
                                        <strong><?php _e('Instant Backup to Cloud', 'assertivlogix-backup-and-migration'); ?></strong>
                                    </label>
                                    <p class="description"><?php _e('Create and upload backup immediately without waiting for scheduled time.', 'assertivlogix-backup-and-migration'); ?></p>
                                </div>
                                
                                <div class="assertivlogix-setting-item">
                                    <label>
                                        <input type="checkbox" id="cloud-upload-after-scheduled" <?php checked($cloud_settings['upload_after_scheduled'], true); ?> />
                                        <strong><?php _e('Upload After Scheduled Backup', 'assertivlogix-backup-and-migration'); ?></strong>
                                    </label>
                                    <p class="description"><?php _e('Automatically upload backups created by scheduled tasks.', 'assertivlogix-backup-and-migration'); ?></p>
                                </div>
                                
                                <div class="assertivlogix-setting-item">
                                    <label>
                                        <input type="checkbox" id="cloud-upload-after-manual" <?php checked($cloud_settings['upload_after_manual'], true); ?> />
                                        <strong><?php _e('Upload After Manual Backup', 'assertivlogix-backup-and-migration'); ?></strong>
                                    </label>
                                    <p class="description"><?php _e('Automatically upload backups created manually from Backup tab.', 'assertivlogix-backup-and-migration'); ?></p>
                                </div>
                                
                                <div class="assertivlogix-setting-item">
                                    <label>
                                        <input type="checkbox" id="cloud-delete-local-after-upload" <?php checked($cloud_settings['delete_local_after_upload'], true); ?> />
                                        <strong><?php _e('Delete Local Backup After Upload', 'assertivlogix-backup-and-migration'); ?></strong>
                                    </label>
                                    <p class="description"><?php _e('Remove local backup file after successful cloud upload (saves disk space).', 'assertivlogix-backup-and-migration'); ?></p>
                                </div>
                            </div>
                            
                            <!-- Provider Selection -->
                            <div class="assertivlogix-settings-group">
                                <h3><span class="dashicons dashicons-filter"></span> <?php _e('Upload To', 'assertivlogix-backup-and-migration'); ?></h3>
                                <p class="description"><?php _e('Select which connected cloud storage providers should receive backups.', 'assertivlogix-backup-and-migration'); ?></p>
                                
                                <?php
                                $provider_names = [
                                    'google_drive' => __('Google Drive', 'assertivlogix-backup-and-migration'),
                                    'dropbox' => __('Dropbox', 'assertivlogix-backup-and-migration'),
                                    's3' => __('Amazon S3', 'assertivlogix-backup-and-migration'),
                                    'onedrive' => __('OneDrive', 'assertivlogix-backup-and-migration'),
                                ];
                                ?>
                                
                                <?php foreach ($connected_providers as $provider): ?>
                                <div class="assertivlogix-setting-item">
                                    <label>
                                        <input type="checkbox" 
                                               class="cloud-provider-checkbox" 
                                               value="<?php echo esc_attr($provider); ?>"
                                               <?php checked(in_array($provider, $cloud_settings['upload_only_to'] ?? []), true); ?> />
                                        <strong><?php echo esc_html($provider_names[$provider] ?? ucfirst($provider)); ?></strong>
                                    </label>
                                </div>
                                <?php endforeach; ?>
                                
                                <?php if (empty($connected_providers)): ?>
                                <p class="description" style="color: var(--text-secondary); font-style: italic;">
                                    <?php _e('Connect at least one cloud storage provider above to enable upload options.', 'assertivlogix-backup-and-migration'); ?>
                                </p>
                                <?php endif; ?>
                            </div>
                            
                            <!-- Retention Settings -->
                            <div class="assertivlogix-settings-group">
                                <h3><span class="dashicons dashicons-calendar-alt"></span> <?php _e('Retention Policy', 'assertivlogix-backup-and-migration'); ?></h3>
                                
                                <div class="assertivlogix-setting-item">
                                    <label for="cloud-retention-days">
                                        <strong><?php _e('Keep Backups in Cloud (days)', 'assertivlogix-backup-and-migration'); ?></strong>
                                    </label>
                                    <input 
                                        type="number" 
                                        id="cloud-retention-days" 
                                        value="<?php echo esc_attr($cloud_settings['retention_days'] ?? 30); ?>" 
                                        min="1" 
                                        max="365" 
                                        class="small-text"
                                    />
                                    <p class="description"><?php _e('Automatically delete cloud backups older than this many days. Set to 0 for unlimited retention.', 'assertivlogix-backup-and-migration'); ?></p>
                                </div>
                            </div>
                            
                            <!-- Advanced Options -->
                            <div class="assertivlogix-settings-group">
                                <h3><span class="dashicons dashicons-admin-tools"></span> <?php _e('Advanced Options', 'assertivlogix-backup-and-migration'); ?></h3>
                                
                                <div class="assertivlogix-setting-item">
                                    <label>
                                        <input type="checkbox" id="cloud-notify-on-failure" <?php checked(get_option('assertivlogix_backup_cloud_notify_failure', false), true); ?> />
                                        <strong><?php _e('Notify on Upload Failure', 'assertivlogix-backup-and-migration'); ?></strong>
                                    </label>
                                    <p class="description"><?php _e('Send email notification if cloud backup upload fails.', 'assertivlogix-backup-and-migration'); ?></p>
                                </div>
                                
                                <div class="assertivlogix-setting-item">
                                    <label>
                                        <input type="checkbox" id="cloud-compress-before-upload" <?php checked(get_option('assertivlogix_backup_cloud_compress', false), true); ?> />
                                        <strong><?php _e('Compress Before Upload', 'assertivlogix-backup-and-migration'); ?></strong>
                                    </label>
                                    <p class="description"><?php _e('Compress backup files before uploading to reduce upload time and storage space.', 'assertivlogix-backup-and-migration'); ?></p>
                                </div>
                                
                                <div class="assertivlogix-setting-item">
                                    <label for="cloud-upload-timeout">
                                        <strong><?php _e('Upload Timeout (seconds)', 'assertivlogix-backup-and-migration'); ?></strong>
                                    </label>
                                    <input 
                                        type="number" 
                                        id="cloud-upload-timeout" 
                                        value="<?php echo esc_attr(get_option('assertivlogix_backup_cloud_timeout', 300)); ?>" 
                                        min="60" 
                                        max="3600" 
                                        class="small-text"
                                    />
                                    <p class="description"><?php _e('Maximum time to wait for upload to complete (default: 300 seconds).', 'assertivlogix-backup-and-migration'); ?></p>
                                </div>
                            </div>
                        </div>
                        
                        <div class="assertivlogix-cloud-settings-actions">
                            <button class="button button-primary button-large" id="btn-save-cloud-settings">
                                <span class="dashicons dashicons-yes-alt"></span>
                                <?php esc_html_e('Save Cloud Settings', 'assertivlogix-backup-and-migration'); ?>
                            </button>
                            <button class="button button-secondary" id="btn-test-cloud-upload">
                                <span class="dashicons dashicons-admin-generic"></span>
                                <?php esc_html_e('Test Cloud Upload', 'assertivlogix-backup-and-migration'); ?>
                            </button>
                        </div>
                    </section>
                    <?php endif; ?>
                    
                    <?php else: ?>
                    <section class="assertivlogix-backup-card">
                        <h2><?php esc_html_e('Remote Storage Locations', 'assertivlogix-backup-and-migration'); ?></h2>
                        <p><?php esc_html_e('Cloud storage integration is a premium feature. Please activate your license to use this feature.', 'assertivlogix-backup-and-migration'); ?></p>
                        <ul class="storage-list">
                            <li><span class="dashicons dashicons-lock"></span> Google Drive (Pro)</li>
                            <li><span class="dashicons dashicons-lock"></span> Dropbox (Pro)</li>
                            <li><span class="dashicons dashicons-lock"></span> Amazon S3 (Pro)</li>
                            <li><span class="dashicons dashicons-lock"></span> OneDrive (Pro)</li>
                        </ul>
                        <div class="assertivlogix-premium-cta-buttons">
                            <a href="https://plugins.assertivlogix.com/plugin/assertivlogix-backup-pro" target="_blank" class="button button-primary">
                                <span class="dashicons dashicons-star-filled"></span>
                                <?php esc_html_e('Upgrade to Pro', 'assertivlogix-backup-and-migration'); ?>
                            </a>
                            <a href="?page=assertivlogix-backup&tab=settings" class="button">
                                <?php esc_html_e('Enter License Key', 'assertivlogix-backup-and-migration'); ?>
                            </a>
                        </div>
                    </section>
                    <?php endif; ?>

                <?php elseif ($active_tab == 'settings'): ?>
                    <?php
                    $license_info = class_exists('AssertivlogixBackup\\License') ? \AssertivlogixBackup\License::get_info() : ['is_valid' => false, 'status' => 'invalid'];
                    $is_license_valid = isset($license_info['is_valid']) && $license_info['is_valid'];
                    
                    // Debug logging
                    if (defined('WP_DEBUG') && WP_DEBUG) {
                        error_log('=== License Status Check (Settings Tab) ===');
                        error_log('License Info: ' . print_r($license_info, true));
                        error_log('Is Valid: ' . ($is_license_valid ? 'true' : 'false'));
                        error_log('License Status Option: ' . get_option('assertivlogix_backup_license_status', 'not set'));
                        error_log('License Key Option: ' . substr(get_option('assertivlogix_backup_license_key', ''), 0, 10) . '...');
                        error_log('License Expires Option: ' . get_option('assertivlogix_backup_license_expires', 'not set'));
                    }
                    ?>
                    
                    <!-- License Management -->
                    <section class="assertivlogix-backup-card">
                        <h2><?php esc_html_e('License Key', 'assertivlogix-backup-and-migration'); ?></h2>
                        <p><?php esc_html_e('Enter your license key to unlock premium features.', 'assertivlogix-backup-and-migration'); ?></p>
                        
                        <div id="license-status" class="assertivlogix-license-status" style="margin-bottom: 20px;">
                            <?php if ($is_license_valid): ?>
                                <div class="notice notice-success inline" style="padding: 16px 20px;">
                                    <span class="dashicons dashicons-yes-alt" style="font-size: 24px; width: 24px; height: 24px; color: #10b981;"></span>
                                    <div style="display: inline-block; margin-left: 12px; vertical-align: middle;">
                                        <strong style="display: block; font-size: 1.125rem; margin-bottom: 4px;"><?php _e('License Activated', 'assertivlogix-backup-and-migration'); ?></strong>
                                        <?php if (!empty($license_info['expires']) && $license_info['expires'] > 0): ?>
                                            <p class="description" style="margin: 0;"><?php printf(__('Expires: %s', 'assertivlogix-backup-and-migration'), date('M d, Y', $license_info['expires'])); ?></p>
                                        <?php else: ?>
                                            <p class="description" style="margin: 0;"><?php _e('Lifetime License', 'assertivlogix-backup-and-migration'); ?></p>
                                        <?php endif; ?>
                                    </div>
                                </div>
                            <?php else: ?>
                                <div class="notice notice-error inline" style="padding: 16px 20px;">
                                    <span class="dashicons dashicons-warning" style="font-size: 24px; width: 24px; height: 24px; color: #ef4444;"></span>
                                    <div style="display: inline-block; margin-left: 12px; vertical-align: middle;">
                                        <strong style="display: block; font-size: 1.125rem; margin-bottom: 4px;"><?php _e('License Not Active', 'assertivlogix-backup-and-migration'); ?></strong>
                                        <p class="description" style="margin: 0;"><?php _e('Premium features are disabled.', 'assertivlogix-backup-and-migration'); ?></p>
                                    </div>
                                </div>
                            <?php endif; ?>
                        </div>
                        
                        <div class="assertivlogix-license-form">
                            <input 
                                type="text" 
                                id="license-key" 
                                class="regular-text" 
                                placeholder="<?php esc_attr_e('Enter your license key', 'assertivlogix-backup-and-migration'); ?>"
                                value="<?php echo esc_attr($license_info['key'] ?? ''); ?>"
                                <?php echo $is_license_valid ? 'style="display:none;"' : ''; ?>
                            />
                            <button class="button button-primary" id="btn-validate-license" <?php echo $is_license_valid ? 'style="display:none;"' : ''; ?>>
                                <?php esc_html_e('Activate License', 'assertivlogix-backup-and-migration'); ?>
                            </button>
                            <?php if ($is_license_valid): ?>
                                <button class="button" id="btn-deactivate-license">
                                    <?php esc_html_e('Deactivate License', 'assertivlogix-backup-and-migration'); ?>
                                </button>
                            <?php endif; ?>
                            <span id="license-message" class="description" style="display: block; margin-top: 10px;"></span>
                        </div>
                    </section>
                    
                    <!-- API Key for Remote Access -->
                    <?php if ($is_license_valid && class_exists('AssertivlogixBackup\\License') && \AssertivlogixBackup\License::has_feature('site_to_site')): ?>
                    <section class="assertivlogix-backup-card">
                        <div class="assertivlogix-feature-header">
                            <div class="assertivlogix-feature-icon-inline" style="background: linear-gradient(135deg, #8b5cf6 0%, #7c3aed 100%);">
                                <span class="dashicons dashicons-admin-network"></span>
                            </div>
                            <div>
                                <h2><?php esc_html_e('API Key for Remote Access', 'assertivlogix-backup-and-migration'); ?></h2>
                                <p class="assertivlogix-feature-description"><?php esc_html_e('Generate an API key to allow other sites to connect to this site for direct site-to-site migration.', 'assertivlogix-backup-and-migration'); ?></p>
                            </div>
                        </div>
                        
                        <?php
                        $api_key = get_option('assertivlogix_backup_api_key', '');
                        if (empty($api_key)) {
                            // Generate a new API key if none exists
                            $api_key = bin2hex(random_bytes(32));
                            update_option('assertivlogix_backup_api_key', $api_key);
                        }
                        ?>
                        
                        <div class="assertivlogix-api-key-section">
                            <label for="api-key-display"><?php _e('Your API Key:', 'assertivlogix-backup-and-migration'); ?></label>
                            <div class="assertivlogix-api-key-wrapper">
                                <input 
                                    type="text" 
                                    id="api-key-display" 
                                    class="regular-text assertivlogix-api-key-input" 
                                    value="<?php echo esc_attr($api_key); ?>" 
                                    readonly 
                                    onclick="this.select();"
                                />
                                <button type="button" class="button" id="btn-copy-api-key" title="<?php esc_attr_e('Copy to clipboard', 'assertivlogix-backup-and-migration'); ?>">
                                    <span class="dashicons dashicons-admin-page"></span>
                                    <?php _e('Copy', 'assertivlogix-backup-and-migration'); ?>
                                </button>
                                <button type="button" class="button" id="btn-regenerate-api-key" title="<?php esc_attr_e('Generate a new API key', 'assertivlogix-backup-and-migration'); ?>">
                                    <span class="dashicons dashicons-update"></span>
                                    <?php _e('Regenerate', 'assertivlogix-backup-and-migration'); ?>
                                </button>
                            </div>
                            <p class="description">
                                <?php _e('Share this API key with the source site to enable direct site-to-site migration. Keep it secure!', 'assertivlogix-backup-and-migration'); ?>
                            </p>
                            <div class="assertivlogix-info-box" style="margin-top: 16px;">
                                <strong><?php _e('How to use:', 'assertivlogix-backup-and-migration'); ?></strong>
                                <ol style="margin: 8px 0 0 20px; padding: 0;">
                                    <li><?php _e('Copy the API key above', 'assertivlogix-backup-and-migration'); ?></li>
                                    <li><?php _e('Go to the source site\'s Migration page', 'assertivlogix-backup-and-migration'); ?></li>
                                    <li><?php _e('Enter this site\'s URL and paste the API key', 'assertivlogix-backup-and-migration'); ?></li>
                                    <li><?php _e('Click "Link Site" to establish connection', 'assertivlogix-backup-and-migration'); ?></li>
                                </ol>
                            </div>
                        </div>
                    </section>
                    <?php endif; ?>
                    
                    <!-- General Settings -->
                    <section class="assertivlogix-backup-card">
                        <h2><?php esc_html_e('General Settings', 'assertivlogix-backup-and-migration'); ?></h2>
                        <p><label><input type="checkbox" id="setting-auto-backup" <?php checked(get_option('assertivlogix_backup_auto_update'), '1'); ?>> <?php _e('Backup automatically before updates', 'assertivlogix-backup-and-migration'); ?></label></p>
                        <p><label><input type="checkbox" id="setting-anonymise" <?php checked(get_option('assertivlogix_backup_anonymise'), '1'); ?>> <?php _e('Anonymise data (GDPR)', 'assertivlogix-backup-and-migration'); ?></label></p>
                        <p><label><input type="checkbox" id="setting-encrypt" <?php checked(get_option('assertivlogix_backup_encrypt'), '1'); ?>> <?php _e('Encrypt database backups', 'assertivlogix-backup-and-migration'); ?></label></p>
                        <p><button class="button button-primary" id="btn-save-settings"><?php esc_html_e('Save Changes', 'assertivlogix-backup-and-migration'); ?></button></p>
                    </section>
                    
                    <?php if ($is_license_valid && class_exists('AssertivlogixBackup\\License') && \AssertivlogixBackup\License::has_feature('advanced_scheduling')): ?>
                    <!-- Advanced Scheduling Settings -->
                    <section class="assertivlogix-backup-card">
                        <h2><?php esc_html_e('Advanced Scheduling', 'assertivlogix-backup-and-migration'); ?></h2>
                        <p><?php esc_html_e('Configure retention policies for your backups.', 'assertivlogix-backup-and-migration'); ?></p>
                        
                        <?php
                        $retention = get_option('assertivlogix_backup_retention_policy', []);
                        ?>
                        <table class="form-table">
                            <tr>
                                <th><label for="retention-daily"><?php _e('Keep Daily Backups (days)', 'assertivlogix-backup-and-migration'); ?></label></th>
                                <td><input type="number" id="retention-daily" value="<?php echo esc_attr($retention['keep_daily'] ?? 7); ?>" min="1" /></td>
                            </tr>
                            <tr>
                                <th><label for="retention-weekly"><?php _e('Keep Weekly Backups (weeks)', 'assertivlogix-backup-and-migration'); ?></label></th>
                                <td><input type="number" id="retention-weekly" value="<?php echo esc_attr($retention['keep_weekly'] ?? 4); ?>" min="1" /></td>
                            </tr>
                            <tr>
                                <th><label for="retention-monthly"><?php _e('Keep Monthly Backups (months)', 'assertivlogix-backup-and-migration'); ?></label></th>
                                <td><input type="number" id="retention-monthly" value="<?php echo esc_attr($retention['keep_monthly'] ?? 6); ?>" min="1" /></td>
                            </tr>
                            <tr>
                                <th><label for="retention-max"><?php _e('Maximum Total Backups', 'assertivlogix-backup-and-migration'); ?></label></th>
                                <td><input type="number" id="retention-max" value="<?php echo esc_attr($retention['max_backups'] ?? 50); ?>" min="1" /></td>
                            </tr>
                        </table>
                        <p><button class="button button-primary" id="btn-save-retention"><?php esc_html_e('Save Retention Policy', 'assertivlogix-backup-and-migration'); ?></button></p>
                    </section>
                    <?php endif; ?>
                <?php endif; ?>
            </div>
        </div>
        <?php
    }

    public function render_how_it_works() {
        ?>
        <div class="wrap assertivlogix-backup-wrapper">
            <h1 class="wp-heading-inline"><?php _e('How Assertivlogix Backup Works', 'assertivlogix-backup-and-migration'); ?></h1>
            <hr class="wp-header-end"/>
            <div class="assertivlogix-backup-dashboard">
                <section class="assertivlogix-backup-card">
                    <h2><?php _e('1. Choose where to back up', 'assertivlogix-backup-and-migration'); ?></h2>
                    <p><?php _e('Select from a wide range of cloud storage options including Google Drive, Dropbox, S3, and more.', 'assertivlogix-backup-and-migration'); ?></p>
                </section>
                <section class="assertivlogix-backup-card">
                    <h2><?php _e('2. Choose what to back up', 'assertivlogix-backup-and-migration'); ?></h2>
                    <p><?php _e('Back up your WordPress database and content files, including plugins, themes, and uploads.', 'assertivlogix-backup-and-migration'); ?></p>
                </section>
                <section class="assertivlogix-backup-card">
                    <h2><?php _e('3. Choose when to back up', 'assertivlogix-backup-and-migration'); ?></h2>
                    <p><?php _e('Run backups manually or schedule them to run automatically in just a few clicks.', 'assertivlogix-backup-and-migration'); ?></p>
                </section>
                <section class="assertivlogix-backup-card">
                    <h2><?php _e('4. Rapid restore', 'assertivlogix-backup-and-migration'); ?></h2>
                    <p><?php _e('Quickly restore your WordPress site from a backup in just minutes.', 'assertivlogix-backup-and-migration'); ?></p>
                </section>
            </div>
        </div>
        <?php
    }

    // AJAX Handlers

    public function ajax_run_backup() {
        check_ajax_referer('assertivlogix_backup_nonce');
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Unauthorized']);
            return;
        }

        try {
            if (!class_exists('AssertivlogixBackup\\Backup')) {
                wp_send_json_error(['message' => 'Backup class not found. Please refresh the page.']);
                return;
            }

            $backup = new \AssertivlogixBackup\Backup();
            $result = $backup->run();

            if (is_wp_error($result)) {
                wp_send_json_error(['message' => $result->get_error_message()]);
            } else {
                wp_send_json_success([
                    'filename' => $result,
                    'message' => __('Backup created successfully!', 'assertivlogix-backup-and-migration')
                ]);
            }
        } catch (\Exception $e) {
            error_log('Assertivlogix Backup Error: ' . $e->getMessage());
            wp_send_json_error(['message' => __('Backup failed: ', 'assertivlogix-backup-and-migration') . $e->getMessage()]);
        } catch (\Error $e) {
            error_log('Assertivlogix Backup Fatal Error: ' . $e->getMessage());
            wp_send_json_error(['message' => __('Critical error occurred. Please check error logs.', 'assertivlogix-backup-and-migration')]);
        }
    }

    public function ajax_save_schedule() {
        check_ajax_referer('assertivlogix_backup_nonce');
        if (!current_user_can('manage_options')) wp_send_json_error('Unauthorized');

        $schedule = sanitize_text_field($_POST['schedule']);
        update_option('assertivlogix_backup_schedule', $schedule);
        
        // Re-init scheduler to apply changes immediately
        $scheduler = new Scheduler();
        $scheduler->setup_cron_schedule();

        wp_send_json_success();
    }

    public function ajax_list_backups() {
        check_ajax_referer('assertivlogix_backup_nonce');
        if (!current_user_can('manage_options')) wp_send_json_error('Unauthorized');

        $backup = new Backup();
        $backups = $backup->get_backups();
        
        // Format for display
        $html = '';
        if (empty($backups)) {
            $html = '<p>' . __('No backups found.', 'assertivlogix-backup-and-migration') . '</p>';
        } else {
            $html .= '<table class="widefat fixed striped">';
            $html .= '<thead><tr><th>' . __('Date', 'assertivlogix-backup-and-migration') . '</th><th>' . __('Size', 'assertivlogix-backup-and-migration') . '</th><th>' . __('Actions', 'assertivlogix-backup-and-migration') . '</th></tr></thead>';
            $html .= '<tbody>';
            foreach ($backups as $b) {
                $download_url = content_url('uploads/assertivlogix-backups/' . $b['filename']);
                $html .= '<tr>';
                $html .= '<td>' . esc_html($b['date']) . '</td>';
                $html .= '<td>' . esc_html($b['size']) . '</td>';
                $html .= '<td>';
                $html .= '<a href="' . esc_url($download_url) . '" class="button" download>' . __('Download', 'assertivlogix-backup-and-migration') . '</a> ';
                $html .= '<button class="button button-primary btn-restore" data-filename="' . esc_attr($b['filename']) . '">' . __('Restore', 'assertivlogix-backup-and-migration') . '</button> ';
                $html .= '<button class="button button-link-delete btn-delete" data-filename="' . esc_attr($b['filename']) . '" style="color: #a00;">' . __('Delete', 'assertivlogix-backup-and-migration') . '</button>';
                $html .= '</td>';
                $html .= '</tr>';
            }
            $html .= '</tbody></table>';
        }

        wp_send_json_success($html);
    }

    public function ajax_restore_backup() {
        check_ajax_referer('assertivlogix_backup_nonce');
        if (!current_user_can('manage_options')) wp_send_json_error('Unauthorized');

        $filename = sanitize_text_field($_POST['filename']);
        $restore = new Restore();
        $result = $restore->run($filename);

        if (is_wp_error($result)) {
            wp_send_json_error($result->get_error_message());
        } else {
            wp_send_json_success();
        }
    }

    public function ajax_delete_backup() {
        check_ajax_referer('assertivlogix_backup_nonce');
        if (!current_user_can('manage_options')) wp_send_json_error('Unauthorized');

        $filename = sanitize_text_field($_POST['filename']);
        $backup = new Backup();
        $result = $backup->delete($filename);

        if (is_wp_error($result)) {
            wp_send_json_error($result->get_error_message());
        } else {
            wp_send_json_success();
        }
    }

    public function ajax_save_settings() {
        check_ajax_referer('assertivlogix_backup_nonce');
        if (!current_user_can('manage_options')) wp_send_json_error('Unauthorized');

        $auto_update = isset($_POST['auto_update']) && $_POST['auto_update'] === 'true' ? '1' : '0';
        $anonymise   = isset($_POST['anonymise']) && $_POST['anonymise'] === 'true' ? '1' : '0';
        $encrypt     = isset($_POST['encrypt']) && $_POST['encrypt'] === 'true' ? '1' : '0';

        update_option('assertivlogix_backup_auto_update', $auto_update);
        update_option('assertivlogix_backup_anonymise', $anonymise);
        update_option('assertivlogix_backup_encrypt', $encrypt);

        wp_send_json_success();
    }
    
    // Migration AJAX Handlers
    
    public function ajax_upload_migrate() {
        check_ajax_referer('assertivlogix_backup_nonce');
        if (!current_user_can('manage_options')) wp_send_json_error('Unauthorized');
        
        // Check if file was uploaded
        if (!isset($_FILES['backup_file']) || $_FILES['backup_file']['error'] !== UPLOAD_ERR_OK) {
            wp_send_json_error(__('No file uploaded or upload error occurred.', 'assertivlogix-backup-and-migration'));
        }
        
        $file = $_FILES['backup_file'];
        
        // Validate file type
        if (pathinfo($file['name'], PATHINFO_EXTENSION) !== 'zip') {
            wp_send_json_error(__('Only ZIP files are allowed.', 'assertivlogix-backup-and-migration'));
        }
        
        // Validate file size (512MB max)
        $max_size = 512 * 1024 * 1024; // 512MB in bytes
        if ($file['size'] > $max_size) {
            wp_send_json_error(__('File size exceeds 512MB limit.', 'assertivlogix-backup-and-migration'));
        }
        
        // Get backup directory
        $upload_dir = wp_upload_dir();
        $backup_dir = $upload_dir['basedir'] . '/assertivlogix-backups/';
        
        // Create directory if it doesn't exist
        if (!file_exists($backup_dir)) {
            wp_mkdir_p($backup_dir);
        }
        
        // Generate unique filename
        $filename = 'uploaded-' . date('Y-m-d-His') . '-' . sanitize_file_name($file['name']);
        $destination = $backup_dir . $filename;
        
        // Move uploaded file
        if (!move_uploaded_file($file['tmp_name'], $destination)) {
            wp_send_json_error(__('Failed to move uploaded file.', 'assertivlogix-backup-and-migration'));
        }
        
        // Now restore the backup
        $restore = new Restore();
        $result = $restore->run($filename);
        
        if (is_wp_error($result)) {
            // Delete the uploaded file if restore failed
            @unlink($destination);
            wp_send_json_error($result->get_error_message());
        } else {
            wp_send_json_success([
                'message' => __('Backup uploaded and migrated successfully! The page will reload in 3 seconds.', 'assertivlogix-backup-and-migration'),
                'filename' => $filename
            ]);
        }
    }
    
    public function ajax_search_replace() {
        check_ajax_referer('assertivlogix_backup_nonce');
        if (!current_user_can('manage_options')) wp_send_json_error('Unauthorized');
        
        $search = sanitize_text_field($_POST['search']);
        $replace = sanitize_text_field($_POST['replace']);
        
        $migration = new Migration();
        $result = $migration->search_replace($search, $replace);
        
        if (is_wp_error($result)) {
            wp_send_json_error($result->get_error_message());
        } else {
            $total_rows = array_sum(array_column($result, 'rows'));
            wp_send_json_success([
                'message' => sprintf(__('Replaced %d occurrences across %d locations.', 'assertivlogix-backup-and-migration'), $total_rows, count($result)),
                'details' => $result
            ]);
        }
    }
    
    public function ajax_cleanup() {
        check_ajax_referer('assertivlogix_backup_nonce');
        if (!current_user_can('manage_options')) wp_send_json_error('Unauthorized');
        
        $migration = new Migration();
        $tasks = $migration->cleanup_after_migration();
        
        wp_send_json_success([
            'message' => sprintf(__('Completed %d cleanup tasks.', 'assertivlogix-backup-and-migration'), count($tasks)),
            'tasks' => $tasks
        ]);
    }
    
    public function ajax_selective_backup() {
        check_ajax_referer('assertivlogix_backup_nonce');
        if (!current_user_can('manage_options')) wp_send_json_error('Unauthorized');
        
        // Check if ZipArchive is available
        if (!class_exists('ZipArchive') || !extension_loaded('zip')) {
            wp_send_json_error([
                'code' => 'zip_extension_missing',
                'message' => __('The PHP ZipArchive extension is not enabled on your server. Please enable it to use selective backup functionality.', 'assertivlogix-backup-and-migration')
            ]);
        }
        
        $options = [
            'plugins' => isset($_POST['plugins']) ? json_decode(stripslashes($_POST['plugins']), true) : [],
            'themes' => isset($_POST['themes']) ? json_decode(stripslashes($_POST['themes']), true) : [],
            'tables' => isset($_POST['tables']) ? json_decode(stripslashes($_POST['tables']), true) : [],
            'uploads' => isset($_POST['uploads']) && $_POST['uploads'] === 'true'
        ];
        
        // Validate that at least one option is selected
        if (empty($options['plugins']) && empty($options['themes']) && empty($options['tables']) && !$options['uploads']) {
            wp_send_json_error(__('Please select at least one item to backup.', 'assertivlogix-backup-and-migration'));
        }
        
        try {
            $migration = new \AssertivlogixBackup\Migration();
            $result = $migration->selective_export($options);
            
            if (is_wp_error($result)) {
                wp_send_json_error([
                    'code' => $result->get_error_code(),
                    'message' => $result->get_error_message()
                ]);
            } else {
                wp_send_json_success(['filename' => $result]);
            }
        } catch (\Exception $e) {
            error_log('Selective backup error: ' . $e->getMessage());
            wp_send_json_error([
                'code' => 'exception',
                'message' => __('An unexpected error occurred: ', 'assertivlogix-backup-and-migration') . $e->getMessage()
            ]);
        }
    }
    
    public function ajax_get_plugins() {
        check_ajax_referer('assertivlogix_backup_nonce');
        if (!current_user_can('manage_options')) wp_send_json_error('Unauthorized');
        
        $migration = new Migration();
        $plugins = $migration->get_installed_plugins();
        
        wp_send_json_success($plugins);
    }
    
    public function ajax_get_themes() {
        check_ajax_referer('assertivlogix_backup_nonce');
        if (!current_user_can('manage_options')) wp_send_json_error('Unauthorized');
        
        $migration = new Migration();
        $themes = $migration->get_installed_themes();
        
        wp_send_json_success($themes);
    }
    
    public function ajax_get_tables() {
        check_ajax_referer('assertivlogix_backup_nonce');
        if (!current_user_can('manage_options')) wp_send_json_error('Unauthorized');

        $migration = new Migration();
        $tables = $migration->get_database_tables();
        
        wp_send_json_success($tables);
    }
    
    // License Management AJAX Handlers
    
    public function ajax_validate_license() {
        // Log the AJAX request
        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log('=== AJAX License Validation Request ===');
            error_log('POST Data: ' . print_r($_POST, true));
        }
        
        // Verify nonce
        if (!isset($_POST['_wpnonce']) || !wp_verify_nonce($_POST['_wpnonce'], 'assertivlogix_backup_nonce')) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('Nonce verification failed');
            }
            wp_send_json_error(['message' => __('Security check failed. Please refresh the page and try again.', 'assertivlogix-backup-and-migration')]);
        }
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Unauthorized. You do not have permission to perform this action.', 'assertivlogix-backup-and-migration')]);
        }
        
        if (!class_exists('AssertivlogixBackup\\License')) {
            wp_send_json_error(['message' => __('License class not found. Please deactivate and reactivate the plugin.', 'assertivlogix-backup-and-migration')]);
        }
        
        $license_key = isset($_POST['license_key']) ? sanitize_text_field($_POST['license_key']) : '';
        
        if (empty($license_key)) {
            wp_send_json_error(['message' => __('License key cannot be empty.', 'assertivlogix-backup-and-migration')]);
        }
        
        // Call the license validation
        $result = \AssertivlogixBackup\License::validate($license_key);
        
        if (is_wp_error($result)) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('License validation error: ' . $result->get_error_message());
            }
            wp_send_json_error([
                'message' => $result->get_error_message(),
                'code' => $result->get_error_code(),
            ]);
        } else {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('License validation success');
            }
            wp_send_json_success([
                'message' => isset($result['message']) ? $result['message'] : __('License activated successfully!', 'assertivlogix-backup-and-migration'),
                'expires' => isset($result['expires']) ? $result['expires'] : null,
                'data' => isset($result['data']) ? $result['data'] : [],
            ]);
        }
    }
    
    public function ajax_deactivate_license() {
        check_ajax_referer('assertivlogix_backup_nonce');
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Unauthorized');
        }
        
        if (!class_exists('AssertivlogixBackup\\License')) {
            wp_send_json_error(__('License class not found.', 'assertivlogix-backup-and-migration'));
        }
        
        $result = \AssertivlogixBackup\License::deactivate();
        
        if (is_wp_error($result)) {
            wp_send_json_error($result->get_error_message());
        } else {
            wp_send_json_success(['message' => __('License deactivated successfully.', 'assertivlogix-backup-and-migration')]);
        }
    }
    
    /**
     * AJAX handler for regenerating API key.
     */
    public function ajax_regenerate_api_key() {
        check_ajax_referer('assertivlogix_backup_nonce');
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Unauthorized']);
        }
        
        // Check if user has premium license
        if (!class_exists('AssertivlogixBackup\\License') || !\AssertivlogixBackup\License::has_feature('site_to_site')) {
            wp_send_json_error(['message' => __('This feature requires a valid premium license with site-to-site migration feature.', 'assertivlogix-backup-and-migration')]);
        }
        
        // Generate new API key
        $new_api_key = bin2hex(random_bytes(32));
        update_option('assertivlogix_backup_api_key', $new_api_key);
        
        wp_send_json_success([
            'message' => __('API key regenerated successfully!', 'assertivlogix-backup-and-migration'),
            'api_key' => $new_api_key
        ]);
    }
    
    /**
     * AJAX handler for saving cloud storage settings.
     */
    public function ajax_save_cloud_settings() {
        check_ajax_referer('assertivlogix_backup_nonce');
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Unauthorized']);
        }
        
        // Check if user has premium license
        if (!class_exists('AssertivlogixBackup\\License') || !\AssertivlogixBackup\License::has_feature('cloud_storage')) {
            wp_send_json_error(['message' => __('This feature requires a valid premium license with cloud storage feature.', 'assertivlogix-backup-and-migration')]);
        }
        
        // Get settings from POST
        $settings = [
            'auto_upload' => isset($_POST['auto_upload']) && ($_POST['auto_upload'] === 'true' || $_POST['auto_upload'] === '1'),
            'instant_backup' => isset($_POST['instant_backup']) && ($_POST['instant_backup'] === 'true' || $_POST['instant_backup'] === '1'),
            'upload_after_scheduled' => isset($_POST['upload_after_scheduled']) && ($_POST['upload_after_scheduled'] === 'true' || $_POST['upload_after_scheduled'] === '1'),
            'upload_after_manual' => isset($_POST['upload_after_manual']) && ($_POST['upload_after_manual'] === 'true' || $_POST['upload_after_manual'] === '1'),
            'delete_local_after_upload' => isset($_POST['delete_local_after_upload']) && ($_POST['delete_local_after_upload'] === 'true' || $_POST['delete_local_after_upload'] === '1'),
            'retention_days' => isset($_POST['retention_days']) ? absint($_POST['retention_days']) : 30,
            'upload_only_to' => isset($_POST['upload_only_to']) && is_array($_POST['upload_only_to']) ? array_map('sanitize_text_field', $_POST['upload_only_to']) : [],
        ];
        
        // Save settings
        update_option('assertivlogix_backup_cloud_settings', $settings);
        
        // Save advanced settings
        update_option('assertivlogix_backup_cloud_notify_failure', isset($_POST['notify_on_failure']) && ($_POST['notify_on_failure'] === 'true' || $_POST['notify_on_failure'] === '1'));
        update_option('assertivlogix_backup_cloud_compress', isset($_POST['compress_before_upload']) && ($_POST['compress_before_upload'] === 'true' || $_POST['compress_before_upload'] === '1'));
        update_option('assertivlogix_backup_cloud_timeout', isset($_POST['upload_timeout']) ? absint($_POST['upload_timeout']) : 300);
        
        wp_send_json_success([
            'message' => __('Cloud settings saved successfully!', 'assertivlogix-backup-and-migration')
        ]);
    }
    
    /**
     * AJAX handler for testing cloud upload.
     */
    public function ajax_test_cloud_upload() {
        check_ajax_referer('assertivlogix_backup_nonce');
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Unauthorized']);
        }
        
        // Check if user has premium license
        if (!class_exists('AssertivlogixBackup\\License') || !\AssertivlogixBackup\License::has_feature('cloud_storage')) {
            wp_send_json_error(['message' => __('This feature requires a valid premium license with cloud storage feature.', 'assertivlogix-backup-and-migration')]);
        }
        
        // Check if ZipArchive is available
        if (!class_exists('ZipArchive') || !extension_loaded('zip')) {
            wp_send_json_error(['message' => __('PHP ZipArchive extension is required for cloud upload testing.', 'assertivlogix-backup-and-migration')]);
        }
        
        // Check if any cloud storage is connected
        $connections = get_option('assertivlogix_backup_cloud_connections', []);
        $has_connected = false;
        foreach ($connections as $provider => $connection) {
            if (isset($connection['connected']) && $connection['connected']) {
                $has_connected = true;
                break;
            }
        }
        
        if (!$has_connected) {
            wp_send_json_error(['message' => __('Please connect at least one cloud storage provider before testing upload.', 'assertivlogix-backup-and-migration')]);
        }
        
        try {
            // Create a small test backup
            $backup = new \AssertivlogixBackup\Backup();
            
            if (!method_exists($backup, 'run')) {
                wp_send_json_error(['message' => __('Backup class method not found.', 'assertivlogix-backup-and-migration')]);
            }
            
            $backup_file = $backup->run();
            
            if (is_wp_error($backup_file)) {
                wp_send_json_error(['message' => $backup_file->get_error_message()]);
            }
            
            if (empty($backup_file)) {
                wp_send_json_error(['message' => __('Backup creation failed. No file was created.', 'assertivlogix-backup-and-migration')]);
            }
            
            // Try to upload to cloud
            if (!class_exists('AssertivlogixBackup\\Premium\\CloudStorage')) {
                wp_send_json_error(['message' => __('Cloud storage class not found.', 'assertivlogix-backup-and-migration')]);
            }
            
            $cloud_storage = new \AssertivlogixBackup\Premium\CloudStorage();
            
            if (!method_exists($cloud_storage, 'upload_to_cloud')) {
                wp_send_json_error(['message' => __('Cloud storage upload method not found.', 'assertivlogix-backup-and-migration')]);
            }
            
            $upload_result = $cloud_storage->upload_to_cloud($backup_file);
            
            if (is_wp_error($upload_result)) {
                wp_send_json_error(['message' => $upload_result->get_error_message()]);
            }
            
            // Check if upload_result is an array with results
            $upload_success = false;
            $upload_errors = [];
            
            if (is_array($upload_result)) {
                foreach ($upload_result as $provider => $result) {
                    if (is_wp_error($result)) {
                        $upload_errors[] = ucfirst(str_replace('_', ' ', $provider)) . ': ' . $result->get_error_message();
                    } else {
                        $upload_success = true;
                    }
                }
            } else {
                $upload_success = ($upload_result !== false);
            }
            
            if ($upload_success) {
                $message = __('Cloud upload test successful! Backup uploaded to connected cloud storage providers.', 'assertivlogix-backup-and-migration');
                if (!empty($upload_errors)) {
                    $message .= ' ' . __('Some providers had errors: ', 'assertivlogix-backup-and-migration') . implode(', ', $upload_errors);
                }
                wp_send_json_success(['message' => $message]);
            } else {
                $error_message = !empty($upload_errors) ? implode(', ', $upload_errors) : __('Upload test failed. Please check your cloud storage connections.', 'assertivlogix-backup-and-migration');
                wp_send_json_error(['message' => $error_message]);
            }
            
        } catch (\Exception $e) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('Cloud upload test error: ' . $e->getMessage());
                error_log('Stack trace: ' . $e->getTraceAsString());
            }
            wp_send_json_error([
                'message' => __('Test failed: ', 'assertivlogix-backup-and-migration') . $e->getMessage()
            ]);
        } catch (\Error $e) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('Cloud upload test fatal error: ' . $e->getMessage());
                error_log('Stack trace: ' . $e->getTraceAsString());
            }
            wp_send_json_error([
                'message' => __('Fatal error: ', 'assertivlogix-backup-and-migration') . $e->getMessage()
            ]);
        }
    }
}

