<?php
namespace AssertivlogixBackup;

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

/**
 * Scheduler class to handle WP-Cron backup events.
 */
class Scheduler {
    const CRON_HOOK = 'assertivlogix_backup_scheduled_event';

    public function init() {
        add_action(self::CRON_HOOK, [$this, 'run_scheduled_backup']);
        add_filter('cron_schedules', [$this, 'add_custom_cron_schedules']);
    }

    /**
     * Add custom cron schedules.
     */
    public function add_custom_cron_schedules($schedules) {
        $schedules['weekly'] = [
            'interval' => 604800,
            'display'  => __('Once Weekly', 'assertivlogix-backup-and-migration')
        ];
        $schedules['fortnightly'] = [
            'interval' => 1209600,
            'display'  => __('Once every 2 weeks', 'assertivlogix-backup-and-migration')
        ];
        $schedules['monthly'] = [
            'interval' => 2592000,
            'display'  => __('Once Monthly', 'assertivlogix-backup-and-migration')
        ];
        return $schedules;
    }

    /**
     * Set up WP-Cron schedule based on user option.
     * This should only be called when settings are saved.
     */
    public function setup_cron_schedule() {
        $schedule = get_option('assertivlogix_backup_schedule', '');
        
        // Clear any existing scheduled event.
        wp_clear_scheduled_hook(self::CRON_HOOK);

        if (empty($schedule)) {
            return;
        }

        // Map schedule to recurrence.
        $recurrence = $this->map_schedule_to_recurrence($schedule);
        
        if ($recurrence) {
            // Schedule first run for 1 hour from now to avoid immediate execution loop
            wp_schedule_event(time() + 3600, $recurrence, self::CRON_HOOK);
        }
    }

    /**
     * Convert user friendly schedule to WP-Cron recurrence.
     */
    private function map_schedule_to_recurrence($schedule) {
        $map = [
            '2'          => 'hourly', // This is imperfect, ideally we register 'every_2_hours'
            '4'          => 'hourly',
            '8'          => 'twicedaily',
            '12'         => 'twicedaily',
            'daily'      => 'daily',
            'weekly'     => 'weekly',
            'fortnightly'=> 'fortnightly',
            'monthly'    => 'monthly',
        ];
        
        return isset($map[$schedule]) ? $map[$schedule] : false;
    }

    /**
     * Callback for scheduled backup.
     */
    public function run_scheduled_backup() {
        $backup = new Backup();
        $backup->run();
    }
}
