<?php
namespace AssertivlogixBackup;

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

/**
 * Core restore logic.
 */
class Restore {

    private $backup_dir;

    public function __construct() {
        $upload_dir = wp_upload_dir();
        $this->backup_dir = trailingslashit($upload_dir['basedir']) . 'assertivlogix-backups/';
    }

    /**
     * Restore from a specific backup file.
     * 
     * @param string $filename
     * @return bool|WP_Error
     */
    public function run($filename) {
        $zip_path = $this->backup_dir . $filename;
        
        if (!file_exists($zip_path)) {
            return new \WP_Error('file_not_found', __('Backup file not found.', 'assertivlogix-backup-and-migration'));
        }

        @set_time_limit(0);
        
        $zip = new \ZipArchive();
        if ($zip->open($zip_path) === TRUE) {
            
            // 1. Extract files
            // We extract to a temporary folder first or directly overwrite?
            // For simplicity in this version, we overwrite wp-content.
            // Note: This is dangerous in production without checks, but standard for restore plugins.
            
            // We need to be careful about paths. Our zip has 'wp-content/...' structure.
            // Extracting to WP_CONTENT_DIR's parent (ABSPATH usually)
            
            $extract_path = ABSPATH;
            
            // Filter: only extract wp-content and database.sql
            for ($i = 0; $i < $zip->numFiles; $i++) {
                $entry = $zip->getNameIndex($i);
                
                if ($entry === 'database.sql') {
                    // Extract DB dump to temp
                    $zip->extractTo($this->backup_dir, 'database.sql');
                } elseif (strpos($entry, 'wp-content/') === 0) {
                    // Extract content file
                    // We need to map 'wp-content/' in zip to actual WP_CONTENT_DIR
                    // If WP_CONTENT_DIR is standard, extracting to ABSPATH works.
                    // If custom, we need logic. Assuming standard for now or relative to ABSPATH.
                    $zip->extractTo($extract_path, $entry);
                }
            }
            
            $zip->close();
            
            // 2. Import Database
            $sql_file = $this->backup_dir . 'database.sql';
            if (file_exists($sql_file)) {
                $this->import_database($sql_file);
                unlink($sql_file); // cleanup
            }
            
            return true;
        } else {
            return new \WP_Error('zip_error', __('Failed to open zip archive.', 'assertivlogix-backup-and-migration'));
        }
    }

    private function import_database($sql_file) {
        global $wpdb;
        
        // Read file
        $sql = file_get_contents($sql_file);
        if (!$sql) return;

        // Split into queries
        // This is a naive split, robust implementations use a parser or line-by-line reading
        $queries = explode(";\n", $sql);
        
        foreach ($queries as $query) {
            $query = trim($query);
            if (!empty($query)) {
                $wpdb->query($query);
            }
        }
    }
}
