<?php
namespace AssertivlogixBackup\Premium;

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

use AssertivlogixBackup\License;

/**
 * Multisite Support feature.
 * Allows migrating single sites into multisite networks.
 */
class MultisiteSupport {
    
    /**
     * Initialize the feature.
     */
    public function init() {
        if (!License::has_feature('multisite')) {
            return;
        }
        
        // Add hooks for multisite migration
        add_action('wp_ajax_assertivlogix_backup_migrate_to_multisite', [$this, 'ajax_migrate_to_multisite']);
    }
    
    /**
     * Migrate a single site backup into a multisite network.
     * 
     * @param string $backup_file Backup file name
     * @param string $new_site_domain New site domain/subdomain
     * @param string $new_site_path New site path
     * @return bool|WP_Error
     */
    public function migrate_to_multisite($backup_file, $new_site_domain, $new_site_path = '/') {
        if (!License::has_feature('multisite')) {
            return new \WP_Error('premium_required', __('This feature requires a valid premium license.', 'assertivlogix-backup-and-migration'));
        }
        
        if (!is_multisite()) {
            return new \WP_Error('not_multisite', __('This feature is only available on multisite networks.', 'assertivlogix-backup-and-migration'));
        }
        
        // Validate inputs
        if (empty($backup_file) || empty($new_site_domain)) {
            return new \WP_Error('invalid_params', __('Backup file and site domain are required.', 'assertivlogix-backup-and-migration'));
        }
        
        // Create new site in multisite network
        $site_id = wpmu_create_blog($new_site_domain, $new_site_path, '', get_current_user_id(), [
            'public' => 1,
        ]);
        
        if (is_wp_error($site_id)) {
            return $site_id;
        }
        
        // Switch to new site
        switch_to_blog($site_id);
        
        // Restore backup to new site
        $restore = new \AssertivlogixBackup\Restore();
        $result = $restore->run($backup_file);
        
        // Switch back to original site
        restore_current_blog();
        
        if (is_wp_error($result)) {
            // Delete the site if restore failed
            wpmu_delete_blog($site_id, true);
            return $result;
        }
        
        return [
            'site_id' => $site_id,
            'site_url' => get_site_url($site_id),
        ];
    }
    
    /**
     * AJAX handler for multisite migration.
     */
    public function ajax_migrate_to_multisite() {
        check_ajax_referer('assertivlogix_backup_nonce');
        if (!current_user_can('manage_network')) {
            wp_send_json_error('Unauthorized');
        }
        
        $backup_file = sanitize_text_field($_POST['backup_file'] ?? '');
        $new_site_domain = sanitize_text_field($_POST['site_domain'] ?? '');
        $new_site_path = sanitize_text_field($_POST['site_path'] ?? '/');
        
        $result = $this->migrate_to_multisite($backup_file, $new_site_domain, $new_site_path);
        
        if (is_wp_error($result)) {
            wp_send_json_error($result->get_error_message());
        } else {
            wp_send_json_success([
                'message' => __('Site migrated to multisite network successfully!', 'assertivlogix-backup-and-migration'),
                'site_url' => $result['site_url'],
            ]);
        }
    }
}

