<?php
namespace AssertivlogixBackup\Premium;

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

use AssertivlogixBackup\License;

/**
 * Direct Site-to-Site Migration feature.
 * Allows linking two sites and migrating back and forth instantly.
 */
class DirectSiteToSite {
    
    /**
     * Initialize the feature.
     */
    public function init() {
        if (!License::has_feature('site_to_site')) {
            return;
        }
        
        // Add hooks for site-to-site migration
        add_action('wp_ajax_assertivlogix_backup_link_site', [$this, 'ajax_link_site']);
        add_action('wp_ajax_assertivlogix_backup_migrate_to_linked', [$this, 'ajax_migrate_to_linked']);
        
        // Register REST API endpoints for remote access
        add_action('rest_api_init', [$this, 'register_rest_routes']);
    }
    
    /**
     * Register REST API routes for remote site access.
     */
    public function register_rest_routes() {
        register_rest_route('assertivlogix-backup/v1', '/test', [
            'methods' => 'GET',
            'callback' => [$this, 'rest_test_connection'],
            'permission_callback' => [$this, 'check_api_key'],
        ]);
        
        register_rest_route('assertivlogix-backup/v1', '/restore', [
            'methods' => 'POST',
            'callback' => [$this, 'rest_restore_backup'],
            'permission_callback' => [$this, 'check_api_key'],
        ]);
    }
    
    /**
     * Check API key for REST API requests.
     * 
     * @param WP_REST_Request $request
     * @return bool|WP_Error
     */
    public function check_api_key($request) {
        $api_key = $request->get_header('X-API-Key');
        
        if (empty($api_key)) {
            return new \WP_Error('no_api_key', __('API key is required.', 'assertivlogix-backup-and-migration'), ['status' => 401]);
        }
        
        $stored_key = get_option('assertivlogix_backup_api_key', '');
        
        if (empty($stored_key)) {
            return new \WP_Error('no_api_key_configured', __('API key is not configured on this site.', 'assertivlogix-backup-and-migration'), ['status' => 401]);
        }
        
        if (!hash_equals($stored_key, $api_key)) {
            return new \WP_Error('invalid_api_key', __('Invalid API key.', 'assertivlogix-backup-and-migration'), ['status' => 401]);
        }
        
        return true;
    }
    
    /**
     * REST API endpoint: Test connection.
     * 
     * @param WP_REST_Request $request
     * @return WP_REST_Response|WP_Error
     */
    public function rest_test_connection($request) {
        return new \WP_REST_Response([
            'success' => true,
            'message' => __('Connection successful.', 'assertivlogix-backup-and-migration'),
            'site_url' => home_url(),
            'wp_version' => get_bloginfo('version'),
        ], 200);
    }
    
    /**
     * REST API endpoint: Restore backup from remote site.
     * 
     * @param WP_REST_Request $request
     * @return WP_REST_Response|WP_Error
     */
    public function rest_restore_backup($request) {
        $params = $request->get_json_params();
        
        if (empty($params['backup_file_name']) || empty($params['backup_file_data'])) {
            return new \WP_Error('missing_data', __('Backup file name and data are required.', 'assertivlogix-backup-and-migration'), ['status' => 400]);
        }
        
        $file_name = sanitize_file_name($params['backup_file_name']);
        $file_data = $params['backup_file_data'];
        
        // Decode base64 data
        $file_contents = base64_decode($file_data);
        if ($file_contents === false) {
            return new \WP_Error('decode_failed', __('Failed to decode backup file data.', 'assertivlogix-backup-and-migration'), ['status' => 400]);
        }
        
        // Save backup file temporarily
        $upload_dir = wp_upload_dir();
        $backup_dir = str_replace('\\', '/', $upload_dir['basedir'] . '/assertivlogix-backups/');
        wp_mkdir_p($backup_dir);
        
        $temp_file = $backup_dir . 'temp-' . $file_name;
        $result = file_put_contents($temp_file, $file_contents);
        
        if ($result === false) {
            return new \WP_Error('save_failed', __('Failed to save backup file.', 'assertivlogix-backup-and-migration'), ['status' => 500]);
        }
        
        // Move to final location
        $final_file = $backup_dir . $file_name;
        if (file_exists($final_file)) {
            unlink($final_file);
        }
        rename($temp_file, $final_file);
        
        // Restore the backup
        $restore = new \AssertivlogixBackup\Restore();
        $restore_result = $restore->run($file_name);
        
        if (is_wp_error($restore_result)) {
            return new \WP_Error('restore_failed', $restore_result->get_error_message(), ['status' => 500]);
        }
        
        return new \WP_REST_Response([
            'success' => true,
            'message' => __('Backup restored successfully.', 'assertivlogix-backup-and-migration'),
            'filename' => $file_name,
        ], 200);
    }
    
    /**
     * Link a remote site for direct migration.
     * 
     * @param string $remote_url Remote site URL
     * @param string $api_key API key for authentication
     * @return bool|WP_Error
     */
    public function link_site($remote_url, $api_key) {
        if (!License::has_feature('site_to_site')) {
            return new \WP_Error('premium_required', __('This feature requires a valid premium license.', 'assertivlogix-backup-and-migration'));
        }
        
        // Validate remote URL
        $remote_url = esc_url_raw($remote_url);
        if (!filter_var($remote_url, FILTER_VALIDATE_URL)) {
            return new \WP_Error('invalid_url', __('Invalid remote site URL.', 'assertivlogix-backup-and-migration'));
        }
        
        // Test connection to remote site
        $test_url = trailingslashit($remote_url) . 'wp-json/assertivlogix-backup/v1/test';
        $response = wp_remote_get($test_url, [
            'headers' => [
                'X-API-Key' => sanitize_text_field($api_key),
            ],
            'timeout' => 10,
        ]);
        
        if (is_wp_error($response)) {
            return $response;
        }
        
        $code = wp_remote_retrieve_response_code($response);
        if ($code !== 200) {
            return new \WP_Error('connection_failed', __('Could not connect to remote site. Please check the URL and API key.', 'assertivlogix-backup-and-migration'));
        }
        
        // Save linked site information
        update_option('assertivlogix_backup_linked_site_url', $remote_url);
        update_option('assertivlogix_backup_linked_site_api_key', $api_key);
        
        return true;
    }
    
    /**
     * Migrate to linked site.
     * 
     * @return bool|WP_Error
     */
    public function migrate_to_linked() {
        if (!License::has_feature('site_to_site')) {
            return new \WP_Error('premium_required', __('This feature requires a valid premium license.', 'assertivlogix-backup-and-migration'));
        }
        
        $remote_url = get_option('assertivlogix_backup_linked_site_url');
        $api_key = get_option('assertivlogix_backup_linked_site_api_key');
        
        if (empty($remote_url) || empty($api_key)) {
            return new \WP_Error('no_linked_site', __('No linked site configured.', 'assertivlogix-backup-and-migration'));
        }
        
        // Create backup first
        $backup = new \AssertivlogixBackup\Backup();
        $backup_file = $backup->run();
        
        if (is_wp_error($backup_file)) {
            return $backup_file;
        }
        
        // Upload backup to remote site
        $upload_url = trailingslashit($remote_url) . 'wp-json/assertivlogix-backup/v1/restore';
        $backup_path = wp_upload_dir()['basedir'] . '/assertivlogix-backups/' . $backup_file;
        
        if (!file_exists($backup_path)) {
            return new \WP_Error('backup_not_found', __('Backup file not found.', 'assertivlogix-backup-and-migration'));
        }
        
        // Read file contents and send as base64 encoded string (more reliable than CURLFile)
        $file_contents = file_get_contents($backup_path);
        if ($file_contents === false) {
            return new \WP_Error('file_read_error', __('Could not read backup file.', 'assertivlogix-backup-and-migration'));
        }
        
        $file_base64 = base64_encode($file_contents);
        $file_name = basename($backup_path);
        
        $response = wp_remote_post($upload_url, [
            'headers' => [
                'X-API-Key' => $api_key,
                'Content-Type' => 'application/json',
            ],
            'body' => json_encode([
                'backup_file_name' => $file_name,
                'backup_file_data' => $file_base64,
            ]),
            'timeout' => 600, // 10 minutes for large files
        ]);
        
        if (is_wp_error($response)) {
            return new \WP_Error('upload_failed', __('Failed to upload backup: ', 'assertivlogix-backup-and-migration') . $response->get_error_message());
        }
        
        $code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);
        
        if ($code !== 200) {
            $error_message = __('Migration to remote site failed. ', 'assertivlogix-backup-and-migration');
            if (!empty($body)) {
                $error_data = json_decode($body, true);
                if (isset($error_data['message'])) {
                    $error_message .= $error_data['message'];
                } else {
                    $error_message .= __('HTTP Status: ', 'assertivlogix-backup-and-migration') . $code;
                }
            } else {
                $error_message .= __('HTTP Status: ', 'assertivlogix-backup-and-migration') . $code;
            }
            return new \WP_Error('migration_failed', $error_message);
        }
        
        return true;
    }
    
    /**
     * AJAX handler for linking a site.
     */
    public function ajax_link_site() {
        check_ajax_referer('assertivlogix_backup_nonce');
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Unauthorized']);
        }
        
        $remote_url = sanitize_text_field($_POST['remote_url'] ?? '');
        $api_key = sanitize_text_field($_POST['api_key'] ?? '');
        
        if (empty($remote_url) || empty($api_key)) {
            wp_send_json_error(['message' => __('Please provide both remote site URL and API key.', 'assertivlogix-backup-and-migration')]);
        }
        
        $result = $this->link_site($remote_url, $api_key);
        
        if (is_wp_error($result)) {
            wp_send_json_error(['message' => $result->get_error_message()]);
        } else {
            wp_send_json_success([
                'message' => __('Site linked successfully!', 'assertivlogix-backup-and-migration'),
                'linked_url' => $remote_url
            ]);
        }
    }
    
    /**
     * AJAX handler for migrating to linked site.
     */
    public function ajax_migrate_to_linked() {
        check_ajax_referer('assertivlogix_backup_nonce');
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Unauthorized']);
        }
        
        // Check if ZipArchive is available
        if (!class_exists('ZipArchive') || !extension_loaded('zip')) {
            wp_send_json_error([
                'message' => __('The PHP ZipArchive extension is not enabled on your server. Please enable it to use migration functionality.', 'assertivlogix-backup-and-migration')
            ]);
        }
        
        try {
            $result = $this->migrate_to_linked();
            
            if (is_wp_error($result)) {
                wp_send_json_error(['message' => $result->get_error_message()]);
            } else {
                wp_send_json_success(['message' => __('Migration completed successfully!', 'assertivlogix-backup-and-migration')]);
            }
        } catch (\Exception $e) {
            error_log('Direct migration error: ' . $e->getMessage());
            wp_send_json_error([
                'message' => __('An unexpected error occurred: ', 'assertivlogix-backup-and-migration') . $e->getMessage()
            ]);
        }
    }
}

