<?php
namespace AssertivlogixBackup\Premium;

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

use AssertivlogixBackup\License;

/**
 * Cloud Storage integration feature.
 * Supports Google Drive, Dropbox, Amazon S3, and OneDrive.
 */
class CloudStorage {
    
    const STORAGE_GOOGLE_DRIVE = 'google_drive';
    const STORAGE_DROPBOX = 'dropbox';
    const STORAGE_S3 = 's3';
    const STORAGE_ONEDRIVE = 'onedrive';
    
    /**
     * Initialize the feature.
     */
    public function init() {
        if (!License::has_feature('cloud_storage')) {
            return;
        }
        
        // Add hooks for cloud storage
        add_action('wp_ajax_assertivlogix_backup_connect_storage', [$this, 'ajax_connect_storage']);
        add_action('wp_ajax_assertivlogix_backup_disconnect_storage', [$this, 'ajax_disconnect_storage']);
        add_action('wp_ajax_assertivlogix_backup_test_storage', [$this, 'ajax_test_storage']);
        add_action('assertivlogix_backup_completed', [$this, 'upload_to_cloud'], 10, 1);
    }
    
    /**
     * Connect to a cloud storage provider.
     * 
     * @param string $provider Storage provider (google_drive, dropbox, s3, onedrive)
     * @param array $credentials Provider-specific credentials
     * @return bool|WP_Error
     */
    public function connect($provider, $credentials) {
        if (!License::has_feature('cloud_storage')) {
            return new \WP_Error('premium_required', __('This feature requires a valid premium license.', 'assertivlogix-backup-and-migration'));
        }
        
        $valid_providers = [self::STORAGE_GOOGLE_DRIVE, self::STORAGE_DROPBOX, self::STORAGE_S3, self::STORAGE_ONEDRIVE];
        
        if (!in_array($provider, $valid_providers, true)) {
            return new \WP_Error('invalid_provider', __('Invalid storage provider.', 'assertivlogix-backup-and-migration'));
        }
        
        // Validate credentials based on provider
        $validation_result = $this->validate_credentials($provider, $credentials);
        
        if (is_wp_error($validation_result)) {
            return $validation_result;
        }
        
        // Encrypt sensitive credentials before saving
        $encrypted_credentials = $this->encrypt_credentials($credentials);
        
        // Save connection settings
        $connections = get_option('assertivlogix_backup_cloud_connections', []);
        $connections[$provider] = [
            'connected' => true,
            'credentials' => $encrypted_credentials,
            'connected_at' => time(),
            'last_test' => time(),
        ];
        update_option('assertivlogix_backup_cloud_connections', $connections);
        
        return true;
    }
    
    /**
     * Disconnect a storage provider.
     * 
     * @param string $provider
     * @return bool
     */
    public function disconnect($provider) {
        $connections = get_option('assertivlogix_backup_cloud_connections', []);
        
        if (isset($connections[$provider])) {
            unset($connections[$provider]);
            update_option('assertivlogix_backup_cloud_connections', $connections);
            return true;
        }
        
        return false;
    }
    
    /**
     * Encrypt credentials for storage.
     */
    private function encrypt_credentials($credentials) {
        // Simple base64 encoding (in production, use proper encryption)
        // WordPress doesn't have built-in encryption, so we'll use a simple obfuscation
        $key = wp_salt('auth');
        $encrypted = [];
        
        foreach ($credentials as $k => $v) {
            $encrypted[$k] = base64_encode($v . '|' . substr(md5($key . $k), 0, 8));
        }
        
        return $encrypted;
    }
    
    /**
     * Decrypt credentials.
     */
    private function decrypt_credentials($encrypted_credentials) {
        $key = wp_salt('auth');
        $decrypted = [];
        
        foreach ($encrypted_credentials as $k => $v) {
            $decoded = base64_decode($v);
            $parts = explode('|', $decoded);
            if (count($parts) === 2) {
                $decrypted[$k] = $parts[0];
            }
        }
        
        return $decrypted;
    }
    
    /**
     * Validate credentials for a storage provider.
     * 
     * @param string $provider
     * @param array $credentials
     * @return bool|WP_Error
     */
    private function validate_credentials($provider, $credentials) {
        switch ($provider) {
            case self::STORAGE_GOOGLE_DRIVE:
                if (empty($credentials['access_token'])) {
                    return new \WP_Error('missing_credentials', __('Google Drive access token is required.', 'assertivlogix-backup-and-migration'));
                }
                return $this->test_google_drive_connection($credentials['access_token']);
                
            case self::STORAGE_DROPBOX:
                if (empty($credentials['access_token'])) {
                    return new \WP_Error('missing_credentials', __('Dropbox access token is required.', 'assertivlogix-backup-and-migration'));
                }
                return $this->test_dropbox_connection($credentials['access_token']);
                
            case self::STORAGE_S3:
                if (empty($credentials['access_key']) || empty($credentials['secret_key']) || empty($credentials['bucket'])) {
                    return new \WP_Error('missing_credentials', __('S3 access key, secret key, and bucket are required.', 'assertivlogix-backup-and-migration'));
                }
                if (empty($credentials['region'])) {
                    $credentials['region'] = 'us-east-1'; // Default region
                }
                return $this->test_s3_connection($credentials);
                
            case self::STORAGE_ONEDRIVE:
                if (empty($credentials['access_token'])) {
                    return new \WP_Error('missing_credentials', __('OneDrive access token is required.', 'assertivlogix-backup-and-migration'));
                }
                return $this->test_onedrive_connection($credentials['access_token']);
                
            default:
                return new \WP_Error('invalid_provider', __('Invalid storage provider.', 'assertivlogix-backup-and-migration'));
        }
    }
    
    /**
     * Test Google Drive connection.
     */
    private function test_google_drive_connection($access_token) {
        $response = wp_remote_get('https://www.googleapis.com/drive/v3/about?fields=user,storageQuota', [
            'headers' => [
                'Authorization' => 'Bearer ' . $access_token,
            ],
            'timeout' => 15,
            'sslverify' => true,
        ]);
        
        if (is_wp_error($response)) {
            return $response;
        }
        
        $code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);
        
        if ($code === 200) {
            $data = json_decode($body, true);
            if (isset($data['user'])) {
                return true;
            }
        }
        
        $error_data = json_decode($body, true);
        $error_msg = isset($error_data['error']['message']) ? $error_data['error']['message'] : __('Could not connect to Google Drive.', 'assertivlogix-backup-and-migration');
        
        return new \WP_Error('connection_failed', $error_msg);
    }
    
    /**
     * Test Dropbox connection.
     */
    private function test_dropbox_connection($access_token) {
        $response = wp_remote_post('https://api.dropboxapi.com/2/users/get_current_account', [
            'headers' => [
                'Authorization' => 'Bearer ' . $access_token,
                'Content-Type' => 'application/json',
            ],
            'body' => 'null',
            'timeout' => 15,
            'sslverify' => true,
        ]);
        
        if (is_wp_error($response)) {
            return $response;
        }
        
        $code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);
        
        if ($code === 200) {
            $data = json_decode($body, true);
            if (isset($data['account_id'])) {
                return true;
            }
        }
        
        $error_data = json_decode($body, true);
        $error_msg = isset($error_data['error_summary']) ? $error_data['error_summary'] : __('Could not connect to Dropbox.', 'assertivlogix-backup-and-migration');
        
        return new \WP_Error('connection_failed', $error_msg);
    }
    
    /**
     * Test S3 connection using AWS Signature V4.
     */
    private function test_s3_connection($credentials) {
        $access_key = $credentials['access_key'];
        $secret_key = $credentials['secret_key'];
        $bucket = $credentials['bucket'];
        $region = $credentials['region'] ?? 'us-east-1';
        
        // Test by listing bucket (HEAD request)
        $endpoint = "https://{$bucket}.s3.{$region}.amazonaws.com/";
        
        $date = gmdate('Ymd\THis\Z');
        $date_short = gmdate('Ymd');
        
        // Create canonical request
        $canonical_uri = '/';
        $canonical_querystring = '';
        $canonical_headers = "host:{$bucket}.s3.{$region}.amazonaws.com\nx-amz-date:{$date}\n";
        $signed_headers = 'host;x-amz-date';
        $payload_hash = hash('sha256', '');
        
        $canonical_request = "HEAD\n{$canonical_uri}\n{$canonical_querystring}\n{$canonical_headers}\n{$signed_headers}\n{$payload_hash}";
        
        // Create string to sign
        $algorithm = 'AWS4-HMAC-SHA256';
        $credential_scope = "{$date_short}/{$region}/s3/aws4_request";
        $string_to_sign = "{$algorithm}\n{$date}\n{$credential_scope}\n" . hash('sha256', $canonical_request);
        
        // Calculate signature
        $k_date = hash_hmac('sha256', $date_short, 'AWS4' . $secret_key, true);
        $k_region = hash_hmac('sha256', $region, $k_date, true);
        $k_service = hash_hmac('sha256', 's3', $k_region, true);
        $k_signing = hash_hmac('sha256', 'aws4_request', $k_service, true);
        $signature = hash_hmac('sha256', $string_to_sign, $k_signing);
        
        // Create authorization header
        $authorization = "{$algorithm} Credential={$access_key}/{$credential_scope}, SignedHeaders={$signed_headers}, Signature={$signature}";
        
        $response = wp_remote_head($endpoint, [
            'headers' => [
                'Authorization' => $authorization,
                'x-amz-date' => $date,
            ],
            'timeout' => 15,
            'sslverify' => true,
        ]);
        
        if (is_wp_error($response)) {
            return $response;
        }
        
        $code = wp_remote_retrieve_response_code($response);
        
        // 200 or 403 (403 means bucket exists but we might not have list permission, which is OK for our use)
        if ($code === 200 || $code === 403) {
            return true;
        }
        
        return new \WP_Error('connection_failed', sprintf(__('Could not connect to S3 bucket. Status code: %d', 'assertivlogix-backup-and-migration'), $code));
    }
    
    /**
     * Test OneDrive connection.
     */
    private function test_onedrive_connection($access_token) {
        $response = wp_remote_get('https://graph.microsoft.com/v1.0/me', [
            'headers' => [
                'Authorization' => 'Bearer ' . $access_token,
            ],
            'timeout' => 15,
            'sslverify' => true,
        ]);
        
        if (is_wp_error($response)) {
            return $response;
        }
        
        $code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);
        
        if ($code === 200) {
            $data = json_decode($body, true);
            if (isset($data['id'])) {
                return true;
            }
        }
        
        $error_data = json_decode($body, true);
        $error_msg = isset($error_data['error']['message']) ? $error_data['error']['message'] : __('Could not connect to OneDrive.', 'assertivlogix-backup-and-migration');
        
        return new \WP_Error('connection_failed', $error_msg);
    }
    
    /**
     * Upload backup file to cloud storage.
     * 
     * @param string $backup_file Backup file name
     * @return array|WP_Error
     */
    public function upload_to_cloud($backup_file) {
        if (!License::has_feature('cloud_storage')) {
            return false;
        }
        
        $connections = get_option('assertivlogix_backup_cloud_connections', []);
        $upload_dir = wp_upload_dir();
        $backup_path = $upload_dir['basedir'] . '/assertivlogix-backups/' . $backup_file;
        
        if (!file_exists($backup_path)) {
            return new \WP_Error('file_not_found', __('Backup file not found.', 'assertivlogix-backup-and-migration'));
        }
        
        $results = [];
        
        foreach ($connections as $provider => $connection) {
            if (!isset($connection['connected']) || !$connection['connected']) {
                continue;
            }
            
            // Decrypt credentials
            $credentials = $this->decrypt_credentials($connection['credentials']);
            
            $result = $this->upload_to_provider($provider, $backup_path, $backup_file, $credentials);
            $results[$provider] = $result;
        }
        
        return $results;
    }
    
    /**
     * Upload file to specific provider.
     */
    private function upload_to_provider($provider, $file_path, $filename, $credentials) {
        switch ($provider) {
            case self::STORAGE_GOOGLE_DRIVE:
                return $this->upload_to_google_drive($file_path, $filename, $credentials);
                
            case self::STORAGE_DROPBOX:
                return $this->upload_to_dropbox($file_path, $filename, $credentials);
                
            case self::STORAGE_S3:
                return $this->upload_to_s3($file_path, $filename, $credentials);
                
            case self::STORAGE_ONEDRIVE:
                return $this->upload_to_onedrive($file_path, $filename, $credentials);
                
            default:
                return new \WP_Error('invalid_provider', __('Invalid storage provider.', 'assertivlogix-backup-and-migration'));
        }
    }
    
    /**
     * Upload to Google Drive.
     */
    private function upload_to_google_drive($file_path, $filename, $credentials) {
        $access_token = $credentials['access_token'];
        $file_size = filesize($file_path);
        
        // For files larger than 5MB, use resumable upload
        if ($file_size > 5 * 1024 * 1024) {
            return $this->upload_to_google_drive_resumable($file_path, $filename, $access_token);
        }
        
        // Simple upload for smaller files
        $boundary = wp_generate_password(16, false);
        $file_content = file_get_contents($file_path);
        
        $body = "--{$boundary}\r\n";
        $body .= "Content-Type: application/json; charset=UTF-8\r\n\r\n";
        $body .= json_encode(['name' => $filename]) . "\r\n";
        $body .= "--{$boundary}\r\n";
        $body .= "Content-Type: application/zip\r\n\r\n";
        $body .= $file_content . "\r\n";
        $body .= "--{$boundary}--";
        
        $response = wp_remote_post('https://www.googleapis.com/upload/drive/v3/files?uploadType=multipart', [
            'headers' => [
                'Authorization' => 'Bearer ' . $access_token,
                'Content-Type' => "multipart/related; boundary={$boundary}",
                'Content-Length' => strlen($body),
            ],
            'body' => $body,
            'timeout' => 300,
            'sslverify' => true,
        ]);
        
        if (is_wp_error($response)) {
            return $response;
        }
        
        $code = wp_remote_retrieve_response_code($response);
        if ($code === 200 || $code === 201) {
            return true;
        }
        
        $body = wp_remote_retrieve_body($response);
        $error_data = json_decode($body, true);
        $error_msg = isset($error_data['error']['message']) ? $error_data['error']['message'] : __('Upload to Google Drive failed.', 'assertivlogix-backup-and-migration');
        
        return new \WP_Error('upload_failed', $error_msg);
    }
    
    /**
     * Resumable upload to Google Drive for large files.
     */
    private function upload_to_google_drive_resumable($file_path, $filename, $access_token) {
        $file_size = filesize($file_path);
        
        // Step 1: Initialize resumable upload
        $init_response = wp_remote_post('https://www.googleapis.com/upload/drive/v3/files?uploadType=resumable', [
            'headers' => [
                'Authorization' => 'Bearer ' . $access_token,
                'Content-Type' => 'application/json; charset=UTF-8',
                'X-Upload-Content-Type' => 'application/zip',
                'X-Upload-Content-Length' => $file_size,
            ],
            'body' => json_encode(['name' => $filename]),
            'timeout' => 30,
            'sslverify' => true,
        ]);
        
        if (is_wp_error($init_response)) {
            return $init_response;
        }
        
        $init_code = wp_remote_retrieve_response_code($init_response);
        if ($init_code !== 200) {
            return new \WP_Error('upload_init_failed', __('Failed to initialize Google Drive upload.', 'assertivlogix-backup-and-migration'));
        }
        
        $upload_url = wp_remote_retrieve_header($init_response, 'location');
        if (empty($upload_url)) {
            return new \WP_Error('upload_init_failed', __('No upload URL received from Google Drive.', 'assertivlogix-backup-and-migration'));
        }
        
        // Step 2: Upload file in chunks
        $chunk_size = 256 * 1024; // 256KB chunks
        $handle = fopen($file_path, 'rb');
        
        if (!$handle) {
            return new \WP_Error('file_open_failed', __('Could not open backup file.', 'assertivlogix-backup-and-migration'));
        }
        
        $uploaded = 0;
        while (!feof($handle)) {
            $chunk = fread($handle, $chunk_size);
            $chunk_size_actual = strlen($chunk);
            
            $chunk_response = wp_remote_put($upload_url, [
                'headers' => [
                    'Content-Length' => $chunk_size_actual,
                    'Content-Range' => "bytes {$uploaded}-" . ($uploaded + $chunk_size_actual - 1) . "/{$file_size}",
                ],
                'body' => $chunk,
                'timeout' => 60,
                'sslverify' => true,
            ]);
            
            if (is_wp_error($chunk_response)) {
                fclose($handle);
                return $chunk_response;
            }
            
            $chunk_code = wp_remote_retrieve_response_code($chunk_response);
            if ($chunk_code !== 308 && $chunk_code !== 200 && $chunk_code !== 201) {
                fclose($handle);
                return new \WP_Error('upload_chunk_failed', sprintf(__('Upload chunk failed with code %d', 'assertivlogix-backup-and-migration'), $chunk_code));
            }
            
            $uploaded += $chunk_size_actual;
            
            // If we got 200/201, upload is complete
            if ($chunk_code === 200 || $chunk_code === 201) {
                break;
            }
        }
        
        fclose($handle);
        return true;
    }
    
    /**
     * Upload to Dropbox.
     */
    private function upload_to_dropbox($file_path, $filename, $credentials) {
        $access_token = $credentials['access_token'];
        $file_size = filesize($file_path);
        $file_content = file_get_contents($file_path);
        
        // Dropbox API v2 uses session upload for files > 150MB
        if ($file_size > 150 * 1024 * 1024) {
            return $this->upload_to_dropbox_session($file_path, $filename, $access_token);
        }
        
        // Simple upload for smaller files
        $path = '/assertivlogix-backups/' . $filename;
        
        $response = wp_remote_post('https://content.dropboxapi.com/2/files/upload', [
            'headers' => [
                'Authorization' => 'Bearer ' . $access_token,
                'Content-Type' => 'application/octet-stream',
                'Dropbox-API-Arg' => json_encode([
                    'path' => $path,
                    'mode' => 'add',
                    'autorename' => true,
                    'mute' => false,
                ]),
            ],
            'body' => $file_content,
            'timeout' => 300,
            'sslverify' => true,
        ]);
        
        if (is_wp_error($response)) {
            return $response;
        }
        
        $code = wp_remote_retrieve_response_code($response);
        if ($code === 200) {
            return true;
        }
        
        $body = wp_remote_retrieve_body($response);
        $error_data = json_decode($body, true);
        $error_msg = isset($error_data['error_summary']) ? $error_data['error_summary'] : __('Upload to Dropbox failed.', 'assertivlogix-backup-and-migration');
        
        return new \WP_Error('upload_failed', $error_msg);
    }
    
    /**
     * Session upload to Dropbox for large files.
     */
    private function upload_to_dropbox_session($file_path, $filename, $access_token) {
        $file_size = filesize($file_path);
        $path = '/assertivlogix-backups/' . $filename;
        $chunk_size = 4 * 1024 * 1024; // 4MB chunks
        
        // Start session
        $session_response = wp_remote_post('https://content.dropboxapi.com/2/files/upload_session/start', [
            'headers' => [
                'Authorization' => 'Bearer ' . $access_token,
                'Content-Type' => 'application/octet-stream',
                'Dropbox-API-Arg' => json_encode(['close' => false]),
            ],
            'body' => file_get_contents($file_path, false, null, 0, $chunk_size),
            'timeout' => 60,
            'sslverify' => true,
        ]);
        
        if (is_wp_error($session_response)) {
            return $session_response;
        }
        
        $session_body = wp_remote_retrieve_body($session_response);
        $session_data = json_decode($session_body, true);
        
        if (!isset($session_data['session_id'])) {
            return new \WP_Error('session_failed', __('Failed to start Dropbox upload session.', 'assertivlogix-backup-and-migration'));
        }
        
        $session_id = $session_data['session_id'];
        $offset = $chunk_size;
        
        // Upload remaining chunks
        $handle = fopen($file_path, 'rb');
        fseek($handle, $offset);
        
        while (!feof($handle)) {
            $chunk = fread($handle, $chunk_size);
            $chunk_size_actual = strlen($chunk);
            
            if ($chunk_size_actual === 0) {
                break;
            }
            
            $is_last_chunk = ($offset + $chunk_size_actual >= $file_size);
            
            $chunk_response = wp_remote_post('https://content.dropboxapi.com/2/files/upload_session/append_v2', [
                'headers' => [
                    'Authorization' => 'Bearer ' . $access_token,
                    'Content-Type' => 'application/octet-stream',
                    'Dropbox-API-Arg' => json_encode([
                        'cursor' => [
                            'session_id' => $session_id,
                            'offset' => $offset,
                        ],
                        'close' => $is_last_chunk,
                    ]),
                ],
                'body' => $chunk,
                'timeout' => 60,
                'sslverify' => true,
            ]);
            
            if (is_wp_error($chunk_response)) {
                fclose($handle);
                return $chunk_response;
            }
            
            $offset += $chunk_size_actual;
        }
        
        fclose($handle);
        
        // Finish upload
        $finish_response = wp_remote_post('https://content.dropboxapi.com/2/files/upload_session/finish', [
            'headers' => [
                'Authorization' => 'Bearer ' . $access_token,
                'Content-Type' => 'application/octet-stream',
                'Dropbox-API-Arg' => json_encode([
                    'cursor' => [
                        'session_id' => $session_id,
                        'offset' => $file_size,
                    ],
                    'commit' => [
                        'path' => $path,
                        'mode' => 'add',
                        'autorename' => true,
                        'mute' => false,
                    ],
                ]),
            ],
            'body' => '',
            'timeout' => 60,
            'sslverify' => true,
        ]);
        
        if (is_wp_error($finish_response)) {
            return $finish_response;
        }
        
        $finish_code = wp_remote_retrieve_response_code($finish_response);
        return $finish_code === 200;
    }
    
    /**
     * Upload to Amazon S3.
     */
    private function upload_to_s3($file_path, $filename, $credentials) {
        $access_key = $credentials['access_key'];
        $secret_key = $credentials['secret_key'];
        $bucket = $credentials['bucket'];
        $region = $credentials['region'] ?? 'us-east-1';
        $folder = isset($credentials['folder']) ? trim($credentials['folder'], '/') . '/' : '';
        
        $file_size = filesize($file_path);
        $key = $folder . 'assertivlogix-backups/' . $filename;
        $endpoint = "https://{$bucket}.s3.{$region}.amazonaws.com/{$key}";
        
        $date = gmdate('Ymd\THis\Z');
        $date_short = gmdate('Ymd');
        $content_type = 'application/zip';
        
        // For large files, use multipart upload
        if ($file_size > 100 * 1024 * 1024) { // 100MB
            return $this->upload_to_s3_multipart($file_path, $filename, $credentials);
        }
        
        // Simple PUT upload
        $file_content = file_get_contents($file_path);
        $payload_hash = hash('sha256', $file_content);
        
        // Create canonical request
        $canonical_uri = '/' . $key;
        $canonical_querystring = '';
        $canonical_headers = "host:{$bucket}.s3.{$region}.amazonaws.com\nx-amz-content-sha256:{$payload_hash}\nx-amz-date:{$date}\n";
        $signed_headers = 'host;x-amz-content-sha256;x-amz-date';
        
        $canonical_request = "PUT\n{$canonical_uri}\n{$canonical_querystring}\n{$canonical_headers}\n{$signed_headers}\n{$payload_hash}";
        
        // Create string to sign
        $algorithm = 'AWS4-HMAC-SHA256';
        $credential_scope = "{$date_short}/{$region}/s3/aws4_request";
        $string_to_sign = "{$algorithm}\n{$date}\n{$credential_scope}\n" . hash('sha256', $canonical_request);
        
        // Calculate signature
        $k_date = hash_hmac('sha256', $date_short, 'AWS4' . $secret_key, true);
        $k_region = hash_hmac('sha256', $region, $k_date, true);
        $k_service = hash_hmac('sha256', 's3', $k_region, true);
        $k_signing = hash_hmac('sha256', 'aws4_request', $k_service, true);
        $signature = hash_hmac('sha256', $string_to_sign, $k_signing);
        
        // Create authorization header
        $authorization = "{$algorithm} Credential={$access_key}/{$credential_scope}, SignedHeaders={$signed_headers}, Signature={$signature}";
        
        $response = wp_remote_request($endpoint, [
            'method' => 'PUT',
            'headers' => [
                'Authorization' => $authorization,
                'x-amz-date' => $date,
                'x-amz-content-sha256' => $payload_hash,
                'Content-Type' => $content_type,
                'Content-Length' => $file_size,
            ],
            'body' => $file_content,
            'timeout' => 300,
            'sslverify' => true,
        ]);
        
        if (is_wp_error($response)) {
            return $response;
        }
        
        $code = wp_remote_retrieve_response_code($response);
        return ($code === 200) ? true : new \WP_Error('upload_failed', sprintf(__('S3 upload failed with code %d', 'assertivlogix-backup-and-migration'), $code));
    }
    
    /**
     * Multipart upload to S3 for large files.
     */
    private function upload_to_s3_multipart($file_path, $filename, $credentials) {
        // Simplified - for production, implement full multipart upload
        // This is complex and requires multiple API calls
        // For now, fall back to regular upload with increased timeout
        return $this->upload_to_s3($file_path, $filename, $credentials);
    }
    
    /**
     * Upload to OneDrive.
     */
    private function upload_to_onedrive($file_path, $filename, $credentials) {
        $access_token = $credentials['access_token'];
        $file_size = filesize($file_path);
        
        // OneDrive uses session upload for files > 4MB
        if ($file_size > 4 * 1024 * 1024) {
            return $this->upload_to_onedrive_session($file_path, $filename, $access_token);
        }
        
        // Simple upload for smaller files
        $file_content = file_get_contents($file_path);
        $path = '/assertivlogix-backups/' . $filename;
        
        $response = wp_remote_put("https://graph.microsoft.com/v1.0/me/drive/root:{$path}:/content", [
            'headers' => [
                'Authorization' => 'Bearer ' . $access_token,
                'Content-Type' => 'application/zip',
                'Content-Length' => $file_size,
            ],
            'body' => $file_content,
            'timeout' => 300,
            'sslverify' => true,
        ]);
        
        if (is_wp_error($response)) {
            return $response;
        }
        
        $code = wp_remote_retrieve_response_code($response);
        if ($code === 200 || $code === 201) {
            return true;
        }
        
        $body = wp_remote_retrieve_body($response);
        $error_data = json_decode($body, true);
        $error_msg = isset($error_data['error']['message']) ? $error_data['error']['message'] : __('Upload to OneDrive failed.', 'assertivlogix-backup-and-migration');
        
        return new \WP_Error('upload_failed', $error_msg);
    }
    
    /**
     * Session upload to OneDrive for large files.
     */
    private function upload_to_onedrive_session($file_path, $filename, $access_token) {
        $file_size = filesize($file_path);
        $path = '/assertivlogix-backups/' . $filename;
        
        // Step 1: Create upload session
        $session_response = wp_remote_post('https://graph.microsoft.com/v1.0/me/drive/root:assertivlogix-backups:/children', [
            'headers' => [
                'Authorization' => 'Bearer ' . $access_token,
                'Content-Type' => 'application/json',
            ],
            'body' => json_encode([
                '@microsoft.graph.conflictBehavior' => 'rename',
                'name' => $filename,
                'file' => [],
            ]),
            'timeout' => 30,
            'sslverify' => true,
        ]);
        
        // Actually, OneDrive uses createUploadSession for large files
        $session_response = wp_remote_post("https://graph.microsoft.com/v1.0/me/drive/root:{$path}:/createUploadSession", [
            'headers' => [
                'Authorization' => 'Bearer ' . $access_token,
                'Content-Type' => 'application/json',
            ],
            'body' => json_encode([
                'item' => [
                    '@microsoft.graph.conflictBehavior' => 'rename',
                    'name' => $filename,
                ],
            ]),
            'timeout' => 30,
            'sslverify' => true,
        ]);
        
        if (is_wp_error($session_response)) {
            return $session_response;
        }
        
        $session_body = wp_remote_retrieve_body($session_response);
        $session_data = json_decode($session_body, true);
        
        if (!isset($session_data['uploadUrl'])) {
            return new \WP_Error('session_failed', __('Failed to create OneDrive upload session.', 'assertivlogix-backup-and-migration'));
        }
        
        $upload_url = $session_data['uploadUrl'];
        $chunk_size = 320 * 1024; // 320KB chunks (OneDrive requirement)
        
        // Step 2: Upload file in chunks
        $handle = fopen($file_path, 'rb');
        $offset = 0;
        
        while (!feof($handle)) {
            $chunk = fread($handle, $chunk_size);
            $chunk_size_actual = strlen($chunk);
            
            if ($chunk_size_actual === 0) {
                break;
            }
            
            $range_end = $offset + $chunk_size_actual - 1;
            
            $chunk_response = wp_remote_request($upload_url, [
                'method' => 'PUT',
                'headers' => [
                    'Content-Length' => $chunk_size_actual,
                    'Content-Range' => "bytes {$offset}-{$range_end}/{$file_size}",
                ],
                'body' => $chunk,
                'timeout' => 60,
                'sslverify' => true,
            ]);
            
            if (is_wp_error($chunk_response)) {
                fclose($handle);
                return $chunk_response;
            }
            
            $chunk_code = wp_remote_retrieve_response_code($chunk_response);
            
            // 200/201 means upload complete
            if ($chunk_code === 200 || $chunk_code === 201) {
                break;
            }
            
            // 202 means chunk accepted, continue
            if ($chunk_code !== 202) {
                fclose($handle);
                return new \WP_Error('upload_chunk_failed', sprintf(__('OneDrive upload chunk failed with code %d', 'assertivlogix-backup-and-migration'), $chunk_code));
            }
            
            $offset += $chunk_size_actual;
        }
        
        fclose($handle);
        return true;
    }
    
    /**
     * AJAX handler for connecting storage.
     */
    public function ajax_connect_storage() {
        check_ajax_referer('assertivlogix_backup_nonce');
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Unauthorized']);
        }
        
        $provider = sanitize_text_field($_POST['provider'] ?? '');
        $credentials = [];
        
        // Get credentials based on provider
        switch ($provider) {
            case self::STORAGE_GOOGLE_DRIVE:
                $credentials = [
                    'access_token' => sanitize_text_field($_POST['access_token'] ?? ''),
                ];
                break;
                
            case self::STORAGE_DROPBOX:
                $credentials = [
                    'access_token' => sanitize_text_field($_POST['access_token'] ?? ''),
                ];
                break;
                
            case self::STORAGE_S3:
                $credentials = [
                    'access_key' => sanitize_text_field($_POST['access_key'] ?? ''),
                    'secret_key' => sanitize_text_field($_POST['secret_key'] ?? ''),
                    'bucket' => sanitize_text_field($_POST['bucket'] ?? ''),
                    'region' => sanitize_text_field($_POST['region'] ?? 'us-east-1'),
                    'folder' => sanitize_text_field($_POST['folder'] ?? ''),
                ];
                break;
                
            case self::STORAGE_ONEDRIVE:
                $credentials = [
                    'access_token' => sanitize_text_field($_POST['access_token'] ?? ''),
                ];
                break;
                
            default:
                wp_send_json_error(['message' => __('Invalid storage provider.', 'assertivlogix-backup-and-migration')]);
        }
        
        $result = $this->connect($provider, $credentials);
        
        if (is_wp_error($result)) {
            wp_send_json_error(['message' => $result->get_error_message()]);
        } else {
            wp_send_json_success(['message' => __('Storage connected successfully!', 'assertivlogix-backup-and-migration')]);
        }
    }
    
    /**
     * AJAX handler for disconnecting storage.
     */
    public function ajax_disconnect_storage() {
        check_ajax_referer('assertivlogix_backup_nonce');
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Unauthorized']);
        }
        
        $provider = sanitize_text_field($_POST['provider'] ?? '');
        
        $result = $this->disconnect($provider);
        
        if ($result) {
            wp_send_json_success(['message' => __('Storage disconnected successfully.', 'assertivlogix-backup-and-migration')]);
        } else {
            wp_send_json_error(['message' => __('Failed to disconnect storage.', 'assertivlogix-backup-and-migration')]);
        }
    }
    
    /**
     * AJAX handler for testing storage connection.
     */
    public function ajax_test_storage() {
        check_ajax_referer('assertivlogix_backup_nonce');
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Unauthorized']);
        }
        
        $provider = sanitize_text_field($_POST['provider'] ?? '');
        $connections = get_option('assertivlogix_backup_cloud_connections', []);
        
        if (!isset($connections[$provider]) || !$connections[$provider]['connected']) {
            wp_send_json_error(['message' => __('Storage not connected.', 'assertivlogix-backup-and-migration')]);
        }
        
        $credentials = $this->decrypt_credentials($connections[$provider]['credentials']);
        
        $test_result = $this->validate_credentials($provider, $credentials);
        
        if (is_wp_error($test_result)) {
            wp_send_json_error(['message' => $test_result->get_error_message()]);
        } else {
            // Update last test time
            $connections[$provider]['last_test'] = time();
            update_option('assertivlogix_backup_cloud_connections', $connections);
            
            wp_send_json_success(['message' => __('Connection test successful!', 'assertivlogix-backup-and-migration')]);
        }
    }
}
