<?php
namespace AssertivlogixBackup\Premium;

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

use AssertivlogixBackup\License;
use AssertivlogixBackup\Scheduler;

/**
 * Advanced Scheduling feature.
 * Custom backup schedules and retention policies.
 */
class AdvancedScheduler extends \AssertivlogixBackup\Scheduler {
    
    /**
     * Initialize the feature.
     */
    public function init() {
        if (!License::has_feature('advanced_scheduling')) {
            return parent::init(); // Fall back to basic scheduler
        }
        
        parent::init();
        
        // Add advanced scheduling hooks
        add_action('wp_ajax_assertivlogix_backup_save_retention', [$this, 'ajax_save_retention']);
        add_action('assertivlogix_backup_scheduled_event', [$this, 'cleanup_old_backups'], 5);
    }
    
    /**
     * Set retention policy.
     * 
     * @param array $policy Retention policy settings
     * @return bool
     */
    public function set_retention_policy($policy) {
        if (!License::has_feature('advanced_scheduling')) {
            return false;
        }
        
        $defaults = [
            'keep_daily' => 7,      // Keep daily backups for 7 days
            'keep_weekly' => 4,     // Keep weekly backups for 4 weeks
            'keep_monthly' => 6,    // Keep monthly backups for 6 months
            'max_backups' => 50,    // Maximum total backups
        ];
        
        $policy = wp_parse_args($policy, $defaults);
        update_option('assertivlogix_backup_retention_policy', $policy);
        
        return true;
    }
    
    /**
     * Cleanup old backups based on retention policy.
     */
    public function cleanup_old_backups() {
        if (!License::has_feature('advanced_scheduling')) {
            return;
        }
        
        $policy = get_option('assertivlogix_backup_retention_policy', []);
        
        if (empty($policy)) {
            return;
        }
        
        $backup = new \AssertivlogixBackup\Backup();
        $backups = $backup->get_backups();
        
        if (empty($backups)) {
            return;
        }
        
        $now = time();
        $to_delete = [];
        
        foreach ($backups as $backup_item) {
            $backup_time = strtotime($backup_item['date']);
            $age_days = ($now - $backup_time) / DAY_IN_SECONDS;
            
            // Determine backup type based on schedule
            $schedule = get_option('assertivlogix_backup_schedule', 'daily');
            
            // Apply retention rules
            if ($schedule === 'daily' && $age_days > ($policy['keep_daily'] ?? 7)) {
                $to_delete[] = $backup_item['filename'];
            } elseif ($schedule === 'weekly' && $age_days > (($policy['keep_weekly'] ?? 4) * 7)) {
                $to_delete[] = $backup_item['filename'];
            } elseif ($schedule === 'monthly' && $age_days > (($policy['keep_monthly'] ?? 6) * 30)) {
                $to_delete[] = $backup_item['filename'];
            }
        }
        
        // Enforce maximum backups limit
        $max_backups = $policy['max_backups'] ?? 50;
        if (count($backups) > $max_backups) {
            // Sort by date, oldest first
            usort($backups, function($a, $b) {
                return strtotime($a['date']) - strtotime($b['date']);
            });
            
            $excess = array_slice($backups, 0, count($backups) - $max_backups);
            foreach ($excess as $item) {
                if (!in_array($item['filename'], $to_delete, true)) {
                    $to_delete[] = $item['filename'];
                }
            }
        }
        
        // Delete old backups
        foreach ($to_delete as $filename) {
            $backup->delete($filename);
        }
    }
    
    /**
     * Create custom schedule.
     * 
     * @param string $name Schedule name
     * @param int $interval Interval in seconds
     * @param string $display_name Display name
     * @return bool
     */
    public function create_custom_schedule($name, $interval, $display_name) {
        if (!License::has_feature('advanced_scheduling')) {
            return false;
        }
        
        add_filter('cron_schedules', function($schedules) use ($name, $interval, $display_name) {
            $schedules[$name] = [
                'interval' => $interval,
                'display' => $display_name,
            ];
            return $schedules;
        });
        
        return true;
    }
    
    /**
     * AJAX handler for saving retention policy.
     */
    public function ajax_save_retention() {
        check_ajax_referer('assertivlogix_backup_nonce');
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Unauthorized');
        }
        
        $policy = [
            'keep_daily' => intval($_POST['keep_daily'] ?? 7),
            'keep_weekly' => intval($_POST['keep_weekly'] ?? 4),
            'keep_monthly' => intval($_POST['keep_monthly'] ?? 6),
            'max_backups' => intval($_POST['max_backups'] ?? 50),
        ];
        
        $result = $this->set_retention_policy($policy);
        
        if ($result) {
            wp_send_json_success(['message' => __('Retention policy saved successfully!', 'assertivlogix-backup-and-migration')]);
        } else {
            wp_send_json_error(__('Failed to save retention policy.', 'assertivlogix-backup-and-migration'));
        }
    }
}

