<?php
namespace AssertivlogixBackup;

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

/**
 * Migration class to handle site migrations.
 */
class Migration {
    
    private $backup_dir;
    
    public function __construct() {
        $upload_dir = wp_upload_dir();
        $this->backup_dir = str_replace('\\', '/', $upload_dir['basedir'] . '/assertivlogix-backups/');
    }
    
    /**
     * Basic migration: Download backup from source, restore on destination
     */
    public function basic_migrate($backup_file) {
        // This is handled by existing backup/restore functionality
        $restore = new Restore();
        return $restore->run($backup_file);
    }
    
    /**
     * Search and replace in database
     */
    public function search_replace($search, $replace, $tables = []) {
        global $wpdb;
        
        if (empty($search) || empty($replace)) {
            return new \WP_Error('invalid_params', __('Search and replace strings cannot be empty.', 'assertivlogix-backup-and-migration'));
        }
        
        // If no tables specified, get all tables
        if (empty($tables)) {
            $tables = $wpdb->tables('all');
        }
        
        $results = [];
        
        foreach ($tables as $table) {
            // Get all columns for this table
            $columns = $wpdb->get_results("DESCRIBE {$table}");
            
            if (!$columns) continue;
            
            foreach ($columns as $column) {
                $column_name = $column->Field;
                
                // Skip binary columns
                if (stripos($column->Type, 'blob') !== false || stripos($column->Type, 'binary') !== false) {
                    continue;
                }
                
                // Perform search and replace
                $updated = $wpdb->query(
                    $wpdb->prepare(
                        "UPDATE {$table} SET {$column_name} = REPLACE({$column_name}, %s, %s)",
                        $search,
                        $replace
                    )
                );
                
                if ($updated !== false && $updated > 0) {
                    $results[] = [
                        'table' => $table,
                        'column' => $column_name,
                        'rows' => $updated
                    ];
                }
            }
        }
        
        // Clear all caches
        wp_cache_flush();
        
        return $results;
    }
    
    /**
     * Migration cleanup tasks
     */
    public function cleanup_after_migration() {
        $tasks_completed = [];
        
        // 1. Flush rewrite rules
        flush_rewrite_rules();
        $tasks_completed[] = 'Flushed rewrite rules';
        
        // 2. Clear all caches
        wp_cache_flush();
        $tasks_completed[] = 'Cleared WordPress cache';
        
        // 3. Regenerate .htaccess
        if (function_exists('save_mod_rewrite_rules')) {
            save_mod_rewrite_rules();
            $tasks_completed[] = 'Regenerated .htaccess';
        }
        
        // 4. Update site URL options if needed
        $siteurl = get_option('siteurl');
        $home = get_option('home');
        
        if ($siteurl !== $home) {
            update_option('siteurl', $home);
            $tasks_completed[] = 'Synchronized site URL';
        }
        
        // 5. Clear transients
        delete_expired_transients();
        $tasks_completed[] = 'Cleared expired transients';
        
        return $tasks_completed;
    }
    
    /**
     * Selective migration - export specific plugins/themes/tables
     */
    public function selective_export($options = []) {
        $backup = new Backup();
        
        $defaults = [
            'plugins' => [],      // Array of plugin slugs
            'themes' => [],       // Array of theme slugs
            'tables' => [],       // Array of table names
            'uploads' => true,    // Include uploads folder
        ];
        
        $options = wp_parse_args($options, $defaults);
        
        // Create a temporary directory for selective backup
        $temp_dir = $this->backup_dir . 'temp-' . time() . '/';
        wp_mkdir_p($temp_dir);
        
        $zip = new \ZipArchive();
        $zip_filename = 'selective-backup-' . date('Y-m-d-His') . '.zip';
        $zip_path = $this->backup_dir . $zip_filename;
        
        if ($zip->open($zip_path, \ZipArchive::CREATE) !== true) {
            return new \WP_Error('zip_failed', __('Could not create zip file.', 'assertivlogix-backup-and-migration'));
        }
        
        // Add selected plugins
        if (!empty($options['plugins'])) {
            foreach ($options['plugins'] as $plugin_slug) {
                $plugin_path = WP_PLUGIN_DIR . '/' . $plugin_slug;
                if (is_dir($plugin_path)) {
                    $this->add_directory_to_zip($zip, $plugin_path, 'wp-content/plugins/' . $plugin_slug);
                }
            }
        }
        
        // Add selected themes
        if (!empty($options['themes'])) {
            foreach ($options['themes'] as $theme_slug) {
                $theme_path = get_theme_root() . '/' . $theme_slug;
                if (is_dir($theme_path)) {
                    $this->add_directory_to_zip($zip, $theme_path, 'wp-content/themes/' . $theme_slug);
                }
            }
        }
        
        // Add uploads if requested
        if ($options['uploads']) {
            $upload_dir = wp_upload_dir();
            $this->add_directory_to_zip($zip, $upload_dir['basedir'], 'wp-content/uploads');
        }
        
        // Export selected database tables
        if (!empty($options['tables'])) {
            $sql_content = $this->export_selected_tables($options['tables']);
            $zip->addFromString('database.sql', $sql_content);
        }
        
        // Add manifest
        $manifest = [
            'type' => 'selective',
            'date' => date('Y-m-d H:i:s'),
            'site_url' => get_site_url(),
            'wp_version' => get_bloginfo('version'),
            'options' => $options
        ];
        $zip->addFromString('manifest.json', json_encode($manifest, JSON_PRETTY_PRINT));
        
        $zip->close();
        
        // Clean up temp directory
        if (is_dir($temp_dir)) {
            $this->delete_directory($temp_dir);
        }
        
        return $zip_filename;
    }
    
    /**
     * Export selected database tables
     */
    private function export_selected_tables($tables) {
        global $wpdb;
        
        $sql = '';
        
        foreach ($tables as $table) {
            // Add DROP TABLE statement
            $sql .= "DROP TABLE IF EXISTS `{$table}`;\n\n";
            
            // Get CREATE TABLE statement
            $create_table = $wpdb->get_row("SHOW CREATE TABLE `{$table}`", ARRAY_N);
            if ($create_table) {
                $sql .= $create_table[1] . ";\n\n";
            }
            
            // Get table data
            $rows = $wpdb->get_results("SELECT * FROM `{$table}`", ARRAY_A);
            
            if ($rows) {
                foreach ($rows as $row) {
                    $values = array_map(function($value) use ($wpdb) {
                        return is_null($value) ? 'NULL' : $wpdb->prepare('%s', $value);
                    }, array_values($row));
                    
                    $sql .= "INSERT INTO `{$table}` VALUES (" . implode(', ', $values) . ");\n";
                }
                $sql .= "\n";
            }
        }
        
        return $sql;
    }
    
    /**
     * Helper: Add directory to zip
     */
    private function add_directory_to_zip($zip, $source, $destination) {
        if (!is_dir($source)) return;
        
        $files = new \RecursiveIteratorIterator(
            new \RecursiveDirectoryIterator($source),
            \RecursiveIteratorIterator::LEAVES_ONLY
        );
        
        foreach ($files as $file) {
            if (!$file->isDir()) {
                $file_path = $file->getRealPath();
                $relative_path = $destination . '/' . substr($file_path, strlen($source) + 1);
                $zip->addFile($file_path, $relative_path);
            }
        }
    }
    
    /**
     * Helper: Delete directory recursively
     */
    private function delete_directory($dir) {
        if (!is_dir($dir)) return;
        
        $files = array_diff(scandir($dir), ['.', '..']);
        foreach ($files as $file) {
            $path = $dir . '/' . $file;
            is_dir($path) ? $this->delete_directory($path) : unlink($path);
        }
        rmdir($dir);
    }
    
    /**
     * Get list of installed plugins
     */
    public function get_installed_plugins() {
        if (!function_exists('get_plugins')) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }
        
        $all_plugins = get_plugins();
        $plugins = [];
        
        foreach ($all_plugins as $plugin_file => $plugin_data) {
            $plugin_slug = dirname($plugin_file);
            if ($plugin_slug === '.') {
                $plugin_slug = basename($plugin_file, '.php');
            }
            
            $plugins[] = [
                'slug' => $plugin_slug,
                'name' => $plugin_data['Name'],
                'version' => $plugin_data['Version'],
                'active' => is_plugin_active($plugin_file)
            ];
        }
        
        return $plugins;
    }
    
    /**
     * Get list of installed themes
     */
    public function get_installed_themes() {
        $all_themes = wp_get_themes();
        $themes = [];
        
        foreach ($all_themes as $theme_slug => $theme_data) {
            $themes[] = [
                'slug' => $theme_slug,
                'name' => $theme_data->get('Name'),
                'version' => $theme_data->get('Version'),
                'active' => (get_stylesheet() === $theme_slug)
            ];
        }
        
        return $themes;
    }
    
    /**
     * Get list of database tables
     */
    public function get_database_tables() {
        global $wpdb;
        
        $tables = $wpdb->get_col("SHOW TABLES");
        $table_info = [];
        
        foreach ($tables as $table) {
            $size = $wpdb->get_var("SELECT ROUND((DATA_LENGTH + INDEX_LENGTH) / 1024 / 1024, 2) FROM information_schema.TABLES WHERE TABLE_SCHEMA = '{$wpdb->dbname}' AND TABLE_NAME = '{$table}'");
            
            $table_info[] = [
                'name' => $table,
                'size' => $size . ' MB',
                'is_core' => (strpos($table, $wpdb->prefix) === 0)
            ];
        }
        
        return $table_info;
    }
}
