<?php
namespace AssertivlogixBackup;

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

/**
 * Handles license key validation and premium feature access.
 */
class License {
    
    const OPTION_LICENSE_KEY = 'assertivlogix_backup_license_key';
    const OPTION_LICENSE_STATUS = 'assertivlogix_backup_license_status';
    const OPTION_LICENSE_EXPIRES = 'assertivlogix_backup_license_expires';
    const OPTION_LICENSE_DATA = 'assertivlogix_backup_license_data';
    
    // License API endpoint (you'll need to replace this with your actual license server URL)
    const LICENSE_API_URL = 'https://plugins.assertivlogix.com/api/v1/validate';
    
    
    /**
     * Check if license is valid and active.
     * 
     * @return bool
     */
    public static function is_valid() {
        $status = get_option(self::OPTION_LICENSE_STATUS, 'invalid');
        $expires = get_option(self::OPTION_LICENSE_EXPIRES, 0);
        $key = get_option(self::OPTION_LICENSE_KEY, '');
        
        // Check if license key exists
        if (empty($key)) {
            return false;
        }
        
        // Check if license is active and not expired
        if ($status === 'valid' && ($expires === 0 || $expires > time())) {
            return true;
        }
        
        // If expired, mark as invalid
        if ($status === 'valid' && $expires > 0 && $expires <= time()) {
            update_option(self::OPTION_LICENSE_STATUS, 'expired');
            return false;
        }
        
        return false;
    }
    
    /**
     * Get license key.
     * 
     * @return string
     */
    public static function get_key() {
        return get_option(self::OPTION_LICENSE_KEY, '');
    }
    
    /**
     * Validate license key with remote server.
     * 
     * @param string $license_key
     * @return array|WP_Error
     */
    public static function validate($license_key) {
        if (empty($license_key)) {
            return new \WP_Error('empty_key', __('License key cannot be empty.', 'assertivlogix-backup-and-migration'));
        }
        
        $site_url = home_url();
        
        // Prepare JSON body data
        $body_data = [
            'license_key' => sanitize_text_field($license_key),
            'site_url' => $site_url,
            'product_id' => 7,
        ];
        // Make the API request
        // Use curl if available for better localhost support
        $use_curl = function_exists('curl_init');
        
        $args = [
            'method' => 'POST',
            'headers' => [
                'Content-Type' => 'application/json',
                'Accept' => 'application/json',
            ],
            'body' => json_encode($body_data),
            'timeout' => 20,
            'sslverify' => false, // Disable SSL verification for localhost
            'blocking' => true,
            'redirection' => 5,
            'httpversion' => '1.1',
        ];
        
        // For localhost, try to bypass WordPress HTTP filters
        if (strpos(self::LICENSE_API_URL, '127.0.0.1') !== false || strpos(self::LICENSE_API_URL, 'localhost') !== false) {
            // Add user agent to help with localhost requests
            $args['user-agent'] = 'WordPress/' . get_bloginfo('version') . '; ' . home_url();
            
            // Try to use stream context for localhost
            if (!$use_curl) {
                $args['stream'] = true;
            }
        }
        
        $response = wp_remote_post(self::LICENSE_API_URL, $args);
        
        if (is_wp_error($response)) {
            $error_message = $response->get_error_message();
            $error_code = $response->get_error_code();
            
            // Provide more specific error messages
            if (strpos($error_message, 'resolve') !== false || strpos($error_message, 'getaddrinfo') !== false) {
                return new \WP_Error('connection_failed', __('Could not connect to license server. Please check if the API URL is correct and the server is running.', 'assertivlogix-backup-and-migration'));
            } elseif (strpos($error_message, 'timeout') !== false) {
                return new \WP_Error('timeout', __('Request timed out. Please try again.', 'assertivlogix-backup-and-migration'));
            } else {
                return new \WP_Error('request_failed', sprintf(__('Request failed: %s', 'assertivlogix-backup-and-migration'), $error_message));
            }
        }
        
        $response_code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);
        
        
        // Handle non-200 status codes
        if ($response_code !== 200 && $response_code !== 201) {
            $error_message = __('API returned status code %d.', 'assertivlogix-backup-and-migration');
            if (!empty($body)) {
                $error_data = json_decode($body, true);
                if (is_array($error_data) && isset($error_data['message'])) {
                    $error_message = $error_data['message'];
                } else {
                    $error_message .= ' ' . $body;
                }
            }
            update_option(self::OPTION_LICENSE_STATUS, 'invalid');
            return new \WP_Error('api_error', sprintf($error_message, $response_code));
        }
        
        // Parse JSON response
        $data = json_decode($body, true);
        
        if (json_last_error() !== JSON_ERROR_NONE) {
            update_option(self::OPTION_LICENSE_STATUS, 'invalid');
            return new \WP_Error('invalid_response', sprintf(__('Invalid JSON response from license server. Error: %s', 'assertivlogix-backup-and-migration'), json_last_error_msg()));
        }
        
        // Check for various success response formats
        $is_valid = false;
        $success_message = '';
        
        // Pattern 1: success === true (most common)
        if (isset($data['success']) && $data['success'] === true) {
            $is_valid = true;
            $success_message = isset($data['message']) ? $data['message'] : __('License key validated successfully!', 'assertivlogix-backup-and-migration');
            
        }
        // Pattern 2: status === 'valid' or 'active'
        elseif (isset($data['status']) && ($data['status'] === 'valid' || $data['status'] === 'active')) {
            $is_valid = true;
            $success_message = isset($data['message']) ? $data['message'] : __('License key validated successfully!', 'assertivlogix-backup-and-migration');
        }
        // Pattern 3: data.status === 'valid' or 'active' (nested structure)
        elseif (isset($data['data']['status']) && ($data['data']['status'] === 'valid' || $data['data']['status'] === 'active')) {
            $is_valid = true;
            $success_message = isset($data['message']) ? $data['message'] : __('License key validated successfully!', 'assertivlogix-backup-and-migration');
        }
        // Pattern 4: valid === true
        elseif (isset($data['valid']) && $data['valid'] === true) {
            $is_valid = true;
            $success_message = isset($data['message']) ? $data['message'] : __('License key validated successfully!', 'assertivlogix-backup-and-migration');
        }
        
        if ($is_valid) {
            // Handle expiration date in various formats
            $expires = 0;
            
            // Check expires_at first (your API format)
            if (isset($data['data']['expires_at'])) {
                $expires = strtotime($data['data']['expires_at']);
                // If strtotime fails, try alternative format
                if ($expires === false || $expires < 0) {
                    $expires = strtotime(str_replace(' ', 'T', $data['data']['expires_at']));
                }
            }
            // Check expires in nested data
            elseif (isset($data['data']['expires'])) {
                if (is_numeric($data['data']['expires'])) {
                    $expires = intval($data['data']['expires']);
                } else {
                    $expires = strtotime($data['data']['expires']);
                }
            }
            // Check expires at root level
            elseif (isset($data['expires'])) {
                if (is_numeric($data['expires'])) {
                    $expires = intval($data['expires']);
                } else {
                    $expires = strtotime($data['expires']);
                }
            }
            // Check expires_at at root level
            elseif (isset($data['expires_at'])) {
                $expires = strtotime($data['expires_at']);
            }
            
            // Save license data (include nested data structure)
            update_option(self::OPTION_LICENSE_KEY, sanitize_text_field($license_key));
            update_option(self::OPTION_LICENSE_STATUS, 'valid');
            update_option(self::OPTION_LICENSE_EXPIRES, $expires);
            update_option(self::OPTION_LICENSE_DATA, $data);
            
            // Verify the save worked
            $saved_status = get_option(self::OPTION_LICENSE_STATUS);
            $saved_key = get_option(self::OPTION_LICENSE_KEY);
            $saved_expires = get_option(self::OPTION_LICENSE_EXPIRES);
            
            return [
                'status' => 'valid',
                'message' => $success_message,
                'expires' => $expires > 0 ? date('Y-m-d H:i:s', $expires) : null,
                'data' => $data,
            ];
        } else {
            // License is invalid
            update_option(self::OPTION_LICENSE_STATUS, 'invalid');
            $error_message = __('Invalid license key.', 'assertivlogix-backup-and-migration');
            
            if (isset($data['message'])) {
                $error_message = $data['message'];
            } elseif (isset($data['error'])) {
                $error_message = is_array($data['error']) ? implode(', ', $data['error']) : $data['error'];
            } elseif (isset($data['data']['message'])) {
                $error_message = $data['data']['message'];
            }
            
            return new \WP_Error('invalid_license', $error_message);
        }
    }
    
    /**
     * Deactivate license.
     * 
     * @return bool|WP_Error
     */
    public static function deactivate() {
        $license_key = self::get_key();
        
        if (empty($license_key)) {
            return new \WP_Error('no_key', __('No license key to deactivate.', 'assertivlogix-backup-and-migration'));
        }
        
        $site_url = home_url();
        
        $body_data = [
            'license_key' => $license_key,
            'site_url' => $site_url,
        ];
        
        $response = wp_remote_post(self::LICENSE_API_URL . '/deactivate', [
            'headers' => [
                'Content-Type' => 'application/json',
                'Accept' => 'application/json',
            ],
            'body' => json_encode($body_data),
            'timeout' => 15,
            'sslverify' => false, // Disable SSL verification for localhost
        ]);
        
        // Clear license data regardless of response
        delete_option(self::OPTION_LICENSE_KEY);
        delete_option(self::OPTION_LICENSE_STATUS);
        delete_option(self::OPTION_LICENSE_EXPIRES);
        delete_option(self::OPTION_LICENSE_DATA);
        
        return true;
    }
    
    /**
     * Check if a specific premium feature is available.
     * 
     * @param string $feature Feature name (e.g., 'cloud_storage', 'site_to_site', 'multisite', 'advanced_scheduling')
     * @return bool
     */
    public static function has_feature($feature) {
        if (!self::is_valid()) {
            return false;
        }
        
        $license_data = get_option(self::OPTION_LICENSE_DATA, []);
        
        // Check if feature is included in license
        if (isset($license_data['features']) && is_array($license_data['features'])) {
            return in_array($feature, $license_data['features'], true);
        }
        
        // If no specific features list, assume all features are available for valid licenses
        return true;
    }
    
    /**
     * Get license information.
     * 
     * @return array
     */
    public static function get_info() {
        return [
            'key' => self::get_key(),
            'status' => get_option(self::OPTION_LICENSE_STATUS, 'invalid'),
            'expires' => get_option(self::OPTION_LICENSE_EXPIRES, 0),
            'is_valid' => self::is_valid(),
            'data' => get_option(self::OPTION_LICENSE_DATA, []),
        ];
    }
}

