<?php
namespace AssertivlogixBackup;

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

/**
 * Core backup logic.
 */
class Backup {

    /** @var string Directory where backups are stored */
    private $backup_dir;

    public function __construct() {
        $upload_dir      = wp_upload_dir();
        // Normalize path to use forward slashes for consistency
        $basedir = str_replace('\\', '/', $upload_dir['basedir']);
        $this->backup_dir = trailingslashit($basedir) . 'assertivlogix-backups/';

        // Ensure the backup directory exists.
        if (!file_exists($this->backup_dir)) {
            wp_mkdir_p($this->backup_dir);
            // Add index.php and .htaccess to protect directory
            file_put_contents($this->backup_dir . 'index.php', '<?php // Silence is golden.');
            file_put_contents($this->backup_dir . '.htaccess', 'deny from all');
        }
    }

    /**
     * Check if ZipArchive extension is available.
     *
     * @return bool True if ZipArchive is available, false otherwise.
     */
    public static function is_zip_available() {
        return class_exists('ZipArchive') && extension_loaded('zip');
    }

    /**
     * Run a full backup (files + DB) and return the archive path.
     *
     * @return string|WP_Error Path to the zip file on success, WP_Error on failure.
     */
    public function run() {
        // Check if ZipArchive is available
        if (!self::is_zip_available()) {
            return new \WP_Error(
                'zip_not_available',
                __('PHP ZipArchive extension is not installed or enabled. Please contact your hosting provider to enable the PHP zip extension.', 'assertivlogix-backup-and-migration'),
                [
                    'code' => 'zip_not_available',
                    'message' => __('PHP ZipArchive extension is required for backups. Please enable the zip extension in your PHP configuration.', 'assertivlogix-backup-and-migration'),
                    'help_url' => 'https://www.php.net/manual/en/zip.installation.php'
                ]
            );
        }

        // Increase limits for backup process
        @set_time_limit(0);
        @ini_set('memory_limit', '512M');

        $timestamp   = current_time('Ymd_His');
        $zip_filename = "assertivlogix-backup-{$timestamp}.zip";
        $zip_path    = $this->backup_dir . $zip_filename;

        $zip = new \ZipArchive();
        if (true !== $zip->open($zip_path, \ZipArchive::CREATE)) {
            return new \WP_Error('zip_open_failed', __('Unable to create backup archive.', 'assertivlogix-backup-and-migration'));
        }

        // 1. Add wp-content files (recursive).
        $content_dir = WP_CONTENT_DIR;
        $this->add_folder_to_zip($zip, $content_dir, 'wp-content/');

        // 2. Add DB dump.
        $db_dump = $this->dump_database();
        if (is_wp_error($db_dump)) {
            $zip->close();
            if (file_exists($zip_path)) unlink($zip_path);
            return $db_dump;
        }
        $zip->addFromString('database.sql', $db_dump);

        // 3. Add manifest file for restore info
        $manifest = [
            'version' => ASSERTIVLOGIX_BM_VERSION,
            'date' => $timestamp,
            'site_url' => site_url(),
            'abspath' => ABSPATH
        ];
        $zip->addFromString('manifest.json', json_encode($manifest, JSON_PRETTY_PRINT));

        if (!$zip->close()) {
            return new \WP_Error('zip_close_failed', __('Failed to save zip file. Check disk space or permissions.', 'assertivlogix-backup-and-migration'));
        }
        
        // Double check file exists
        if (!file_exists($zip_path)) {
             return new \WP_Error('file_creation_failed', __('Backup file was not created.', 'assertivlogix-backup-and-migration'));
        }
        
        // Trigger action for premium features (cloud storage upload, etc.)
        do_action('assertivlogix_backup_completed', $zip_filename);

        return $zip_filename;
    }

    /**
     * Recursively add a folder to a ZipArchive.
     *
     * @param \ZipArchive $zip   The zip object.
     * @param string      $folder Absolute path of folder to add.
     * @param string      $base   Relative path inside the zip.
     */
    private function add_folder_to_zip(\ZipArchive $zip, $folder, $base) {
        if (!is_dir($folder)) return;

        $files = new \RecursiveIteratorIterator(
            new \RecursiveDirectoryIterator($folder, \RecursiveDirectoryIterator::SKIP_DOTS),
            \RecursiveIteratorIterator::SELF_FIRST
        );

        foreach ($files as $file) {
            // Skip our own backup directory to avoid infinite loops
            if (strpos($file->getPathname(), 'assertivlogix-backups') !== false) {
                continue;
            }

            $localPath = $base . str_replace($folder, '', $file->getPathname());
            // Normalize slashes
            $localPath = str_replace('\\', '/', $localPath);

            if ($file->isDir()) {
                $zip->addEmptyDir($localPath);
            } else {
                $zip->addFile($file->getPathname(), $localPath);
            }
        }
    }

    /**
     * Dump the WordPress database to a string.
     *
     * @return string|WP_Error SQL dump or WP_Error on failure.
     */
    private function dump_database() {
        global $wpdb;

        $tables = $wpdb->get_col('SHOW TABLES LIKE "' . $wpdb->prefix . '%"');

        if (empty($tables)) {
            return new \WP_Error('no_tables', __('No tables found to export.', 'assertivlogix-backup-and-migration'));
        }

        $dump = "-- Assertivlogix Backup Database Dump\n";
        $dump .= "-- Generated: " . date('Y-m-d H:i:s') . "\n\n";
        $dump .= "SET FOREIGN_KEY_CHECKS=0;\n";
        $dump .= "SET SQL_MODE = \"NO_AUTO_VALUE_ON_ZERO\";\n\n";

        foreach ($tables as $table) {
            // Table structure.
            $create = $wpdb->get_row("SHOW CREATE TABLE `$table`", ARRAY_N);
            if (isset($create[1])) {
                $dump .= "DROP TABLE IF EXISTS `$table`;\n";
                $dump .= $create[1] . ";\n\n";
            }

            // Table data.
            // Process in chunks to avoid memory issues
            $count = $wpdb->get_var("SELECT COUNT(*) FROM `$table`");
            $limit = 1000;
            $offset = 0;

            while ($offset < $count) {
                $rows = $wpdb->get_results("SELECT * FROM `$table` LIMIT $limit OFFSET $offset", ARRAY_A);
                if ($rows) {
                    foreach ($rows as $row) {
                        $values = array_map(function($value) use ($wpdb) {
                            if ($value === null) return 'NULL';
                            return "'" . esc_sql($value) . "'";
                        }, array_values($row));
                        $dump .= "INSERT INTO `$table` VALUES (" . implode(', ', $values) . ");\n";
                    }
                }
                $offset += $limit;
            }
            $dump .= "\n";
        }
        
        $dump .= "SET FOREIGN_KEY_CHECKS=1;\n";

        return $dump;
    }

    /**
     * Get list of existing backups.
     */
    public function get_backups() {
        if (!file_exists($this->backup_dir)) return [];
        
        // Use scandir instead of glob for better compatibility
        $files = scandir($this->backup_dir);
        $backups = [];
        
        if ($files === false) return [];

        foreach ($files as $file) {
            if ($file === '.' || $file === '..') continue;
            if (pathinfo($file, PATHINFO_EXTENSION) !== 'zip') continue;
            
            $full_path = $this->backup_dir . $file;
            
            $backups[] = [
                'filename' => $file,
                'path' => $full_path,
                'size' => size_format(filesize($full_path)),
                'date' => date('Y-m-d H:i:s', filemtime($full_path))
            ];
        }
        
        // Sort by date desc
        usort($backups, function($a, $b) {
            return filemtime($b['path']) - filemtime($a['path']);
        });
        
        return $backups;
    }

    /**
     * Delete a specific backup file.
     * 
     * @param string $filename
     * @return bool|WP_Error
     */
    public function delete($filename) {
        $file_path = $this->backup_dir . $filename;
        
        // Security check: ensure file is in backup dir and is a zip
        if (!file_exists($file_path)) {
             return new \WP_Error('file_not_found', __('File not found.', 'assertivlogix-backup-and-migration'));
        }
        
        if (strpos(realpath($file_path), realpath($this->backup_dir)) !== 0) {
             return new \WP_Error('invalid_path', __('Invalid file path.', 'assertivlogix-backup-and-migration'));
        }

        if (unlink($file_path)) {
            return true;
        } else {
            return new \WP_Error('delete_failed', __('Could not delete file.', 'assertivlogix-backup-and-migration'));
        }
    }
}
