jQuery(document).ready(function ($) {

    // Toast Notification Helper
    function showToast(message, type = 'success') {
        let $toast = $('#assertivlogix-backup-toast');
        if ($toast.length === 0) {
            $('body').append('<div id="assertivlogix-backup-toast" class="assertivlogix-backup-toast"></div>');
            $toast = $('#assertivlogix-backup-toast');
        }

        // Reset classes and content
        $toast.removeClass('success error show').text(message).addClass(type);

        // Trigger reflow to restart animation if needed
        void $toast[0].offsetWidth;

        $toast.addClass('show');

        // Hide after 3 seconds
        setTimeout(function () {
            $toast.removeClass('show');
        }, 3000);
    }

    // File Upload for Migration
    let selectedFile = null;

    // Click to upload
    $('#upload-box').on('click', function () {
        $('#backup-file-upload').click();
    });

    // File selection
    $('#backup-file-upload').on('change', function (e) {
        const file = e.target.files[0];
        if (file) {
            handleFileSelect(file);
        }
    });

    // Drag and drop
    $('#upload-box').on('dragover', function (e) {
        e.preventDefault();
        e.stopPropagation();
        $(this).addClass('dragover');
    });

    $('#upload-box').on('dragleave', function (e) {
        e.preventDefault();
        e.stopPropagation();
        $(this).removeClass('dragover');
    });

    $('#upload-box').on('drop', function (e) {
        e.preventDefault();
        e.stopPropagation();
        $(this).removeClass('dragover');

        const files = e.originalEvent.dataTransfer.files;
        if (files.length > 0) {
            handleFileSelect(files[0]);
        }
    });

    // Handle file selection
    function handleFileSelect(file) {
        // Validate file type
        if (!file.name.endsWith('.zip')) {
            showToast('Please select a ZIP file.', 'error');
            return;
        }

        // Validate file size (512MB)
        const maxSize = 512 * 1024 * 1024;
        if (file.size > maxSize) {
            showToast('File size exceeds 512MB limit.', 'error');
            return;
        }

        selectedFile = file;

        // Show file info
        $('#upload-box').hide();
        $('#file-info').show();
        $('#file-name').text(file.name);
        $('#file-size').text(formatFileSize(file.size));
        $('#btn-upload-migrate').prop('disabled', false);
    }

    // Remove file
    $('#remove-file').on('click', function () {
        selectedFile = null;
        $('#backup-file-upload').val('');
        $('#file-info').hide();
        $('#upload-box').show();
        $('#btn-upload-migrate').prop('disabled', true);
    });

    // Format file size
    function formatFileSize(bytes) {
        if (bytes === 0) return '0 Bytes';
        const k = 1024;
        const sizes = ['Bytes', 'KB', 'MB', 'GB'];
        const i = Math.floor(Math.log(bytes) / Math.log(k));
        return Math.round(bytes / Math.pow(k, i) * 100) / 100 + ' ' + sizes[i];
    }

    // Upload and migrate
    $('#assertivlogix-upload-backup-form').on('submit', function (e) {
        e.preventDefault();

        if (!selectedFile) {
            showToast('Please select a file first.', 'error');
            return;
        }

        const $btn = $('#btn-upload-migrate');
        $btn.prop('disabled', true).html('<span class="dashicons dashicons-update spin"></span> Uploading...');

        const formData = new FormData();
        formData.append('action', 'assertivlogix_backup_upload_migrate');
        formData.append('backup_file', selectedFile);
        formData.append('_wpnonce', assertivlogixBackup.nonce);

        $.ajax({
            url: assertivlogixBackup.ajaxUrl,
            type: 'POST',
            data: formData,
            processData: false,
            contentType: false,
            success: function (res) {
                if (res.success) {
                    showToast(res.data.message, 'success');
                    // Reload page after 3 seconds
                    setTimeout(function () {
                        location.reload();
                    }, 3000);
                } else {
                    showToast('Error: ' + res.data, 'error');
                    $btn.prop('disabled', false).html('<span class="dashicons dashicons-upload"></span> Upload & Migrate');
                }
            },
            error: function () {
                showToast('Server error occurred.', 'error');
                $btn.prop('disabled', false).html('<span class="dashicons dashicons-upload"></span> Upload & Migrate');
            }
        });
    });

    // Run backup now.
    // Unbind first to prevent duplicate bindings if script is loaded multiple times
    $('#btn-backup-now').off('click').on('click', function (e) {
        e.preventDefault();
        const $btn = $(this);

        if ($btn.prop('disabled')) return;

        $btn.prop('disabled', true).text('Running...');

        $.post(assertivlogixBackup.ajaxUrl, {
            action: 'assertivlogix_backup_run',
            _wpnonce: assertivlogixBackup.nonce
        })
            .done(function (res) {
                if (res.success) {
                    showToast(res.data && res.data.message ? res.data.message : assertivlogixBackup.strings.backupSuccess, 'success');
                    loadBackupList(); // Refresh list
                } else {
                    // Handle error message - could be string or object
                    let errorMsg = 'Backup failed.';
                    if (typeof res.data === 'string') {
                        errorMsg = res.data;
                    } else if (res.data && res.data.message) {
                        errorMsg = res.data.message;
                    } else if (res.data && typeof res.data === 'object') {
                        errorMsg = JSON.stringify(res.data);
                    }
                    
                    // Show error with longer duration for ZipArchive errors
                    const isZipError = errorMsg.toLowerCase().includes('zip') || errorMsg.toLowerCase().includes('ziparchive');
                    showToast(errorMsg, 'error');
                    
                    // If ZipArchive error, show alert with more details
                    if (isZipError) {
                        setTimeout(function() {
                            alert('PHP ZipArchive Extension Error\n\n' + errorMsg + '\n\nPlease contact your hosting provider to enable the PHP zip extension.');
                        }, 500);
                    }
                }
            })
            .fail(function (xhr, status, error) {
                let errorMsg = 'Server error occurred.';
                if (xhr.responseJSON && xhr.responseJSON.data) {
                    if (typeof xhr.responseJSON.data === 'string') {
                        errorMsg = xhr.responseJSON.data;
                    } else if (xhr.responseJSON.data.message) {
                        errorMsg = xhr.responseJSON.data.message;
                    }
                }
                showToast(errorMsg, 'error');
            })
            .always(function () {
                $btn.prop('disabled', false).text('Run Backup');
            });
    });

    // Save schedule.
    $('#btn-save-schedule').off('click').on('click', function (e) {
        e.preventDefault();
        const interval = $('#backup-interval').val();
        const $btn = $(this);

        if ($btn.prop('disabled')) return;

        $btn.prop('disabled', true);

        $.post(assertivlogixBackup.ajaxUrl, {
            action: 'assertivlogix_backup_save_schedule',
            schedule: interval,
            _wpnonce: assertivlogixBackup.nonce
        })
            .done(function (res) {
                if (res.success) {
                    showToast('Schedule saved successfully.', 'success');
                } else {
                    showToast('Error: ' + res.data, 'error');
                }
            })
            .always(function () {
                $btn.prop('disabled', false);
            });
    });

    // Load backup list
    function loadBackupList() {
        const $list = $('#backup-list');
        // Don't clear immediately to avoid flashing if already loaded, but here we just replace.

        $.post(assertivlogixBackup.ajaxUrl, {
            action: 'assertivlogix_backup_list',
            _wpnonce: assertivlogixBackup.nonce
        })
            .done(function (res) {
                if (res.success) {
                    $list.html(res.data);
                } else {
                    $list.html('<p>Error loading backups.</p>');
                }
            });
    }

    // Initial load if element exists
    if ($('#backup-list').length) {
        loadBackupList();
    }

    // Restore action (delegated)
    $(document).off('click', '.btn-restore').on('click', '.btn-restore', function (e) {
        e.preventDefault();
        const filename = $(this).data('filename');
        if (!confirm(assertivlogixBackup.strings.confirmRestore)) {
            return;
        }

        const $btn = $(this);
        $btn.prop('disabled', true).text('Restoring...');

        $.post(assertivlogixBackup.ajaxUrl, {
            action: 'assertivlogix_backup_restore',
            filename: filename,
            _wpnonce: assertivlogixBackup.nonce
        })
            .done(function (res) {
                if (res.success) {
                    showToast(assertivlogixBackup.strings.restoreSuccess, 'success');
                    setTimeout(function () { location.reload(); }, 1500); // Reload after delay
                } else {
                    showToast('Restore failed: ' + res.data, 'error');
                    $btn.prop('disabled', false).text('Restore');
                }
            })
            .fail(function () {
                showToast('Server error during restore.', 'error');
                $btn.prop('disabled', false).text('Restore');
            });
    });

    // Delete action (delegated)
    $(document).off('click', '.btn-delete').on('click', '.btn-delete', function (e) {
        e.preventDefault();
        const filename = $(this).data('filename');
        if (!confirm('Are you sure you want to delete this backup?')) {
            return;
        }

        const $btn = $(this);
        $btn.prop('disabled', true).text('Deleting...');

        $.post(assertivlogixBackup.ajaxUrl, {
            action: 'assertivlogix_backup_delete',
            filename: filename,
            _wpnonce: assertivlogixBackup.nonce
        })
            .done(function (res) {
                if (res.success) {
                    showToast('Backup deleted successfully.', 'success');
                    // Remove row
                    $btn.closest('tr').fadeOut(function () {
                        $(this).remove();
                        if ($('#backup-list tbody tr').length === 0) {
                            loadBackupList(); // Reload to show empty message
                        }
                    });
                } else {
                    showToast('Delete failed: ' + res.data, 'error');
                    $btn.prop('disabled', false).text('Delete');
                }
            })
            .fail(function () {
                showToast('Server error during delete.', 'error');
                $btn.prop('disabled', false).text('Delete');
            });
    });

    // Save Settings
    // API Key Management
    $(document).on('click', '#btn-copy-api-key', function(e) {
        e.preventDefault();
        const $input = $('#api-key-display');
        $input.select();
        $input[0].setSelectionRange(0, 99999); // For mobile devices
        
        try {
            document.execCommand('copy');
            const $btn = $(this);
            const originalHtml = $btn.html();
            $btn.html('<span class="dashicons dashicons-yes-alt"></span> ' + assertivlogixBackup.strings.copied || 'Copied!');
            $btn.css('background', '#10b981').css('color', '#fff');
            showToast('API key copied to clipboard!', 'success');
            
            setTimeout(function() {
                $btn.html(originalHtml).css('background', '').css('color', '');
            }, 2000);
        } catch (err) {
            showToast('Failed to copy. Please select and copy manually.', 'error');
        }
    });
    
    // Regenerate API Key
    $(document).on('click', '#btn-regenerate-api-key', function(e) {
        e.preventDefault();
        const $btn = $(this);
        
        if (!confirm('Are you sure you want to regenerate the API key? The old key will no longer work. Any sites currently linked using the old key will need to be re-linked.')) {
            return;
        }
        
        $btn.prop('disabled', true).html('<span class="dashicons dashicons-update" style="animation: spin 1s linear infinite;"></span> Regenerating...');
        
        $.post(assertivlogixBackup.ajaxUrl, {
            action: 'assertivlogix_backup_regenerate_api_key',
            _wpnonce: assertivlogixBackup.nonce
        })
        .done(function(res) {
            if (res.success) {
                $('#api-key-display').val(res.data.api_key);
                showToast(res.data.message || 'API key regenerated successfully!', 'success');
            } else {
                const errorMsg = res.data && res.data.message ? res.data.message : 'Failed to regenerate API key';
                showToast('Error: ' + errorMsg, 'error');
            }
        })
        .fail(function() {
            showToast('Server error occurred.', 'error');
        })
        .always(function() {
            $btn.prop('disabled', false).html('<span class="dashicons dashicons-update"></span> Regenerate');
        });
    });
    
    $('#btn-save-settings').off('click').on('click', function (e) {
        e.preventDefault();
        const $btn = $(this);

        if ($btn.prop('disabled')) return;

        const autoUpdate = $('#setting-auto-backup').is(':checked');
        const anonymise = $('#setting-anonymise').is(':checked');
        const encrypt = $('#setting-encrypt').is(':checked');

        $btn.prop('disabled', true).text('Saving...');

        $.post(assertivlogixBackup.ajaxUrl, {
            action: 'assertivlogix_backup_save_settings',
            auto_update: autoUpdate,
            anonymise: anonymise,
            encrypt: encrypt,
            _wpnonce: assertivlogixBackup.nonce
        })
            .done(function (res) {
                if (res.success) {
                    showToast('Settings saved successfully.', 'success');
                } else {
                    showToast('Error saving settings: ' + res.data, 'error');
                }
            })
            .fail(function () {
                showToast('Server error occurred.', 'error');
            })
            .always(function () {
                $btn.prop('disabled', false).text('Save Changes');
            });
    });

    // Migration Features

    // Search and Replace
    $('#btn-search-replace').off('click').on('click', function (e) {
        e.preventDefault();
        const $btn = $(this);
        const search = $('#search-text').val();
        const replace = $('#replace-text').val();

        if (!search || !replace) {
            showToast('Please enter both search and replace text.', 'error');
            return;
        }

        if (!confirm('This will modify your database. Are you sure you want to continue?')) {
            return;
        }

        $btn.prop('disabled', true).text('Processing...');

        $.post(assertivlogixBackup.ajaxUrl, {
            action: 'assertivlogix_backup_search_replace',
            search: search,
            replace: replace,
            _wpnonce: assertivlogixBackup.nonce
        })
            .done(function (res) {
                if (res.success) {
                    showToast(res.data.message, 'success');
                } else {
                    showToast('Error: ' + res.data, 'error');
                }
            })
            .fail(function () {
                showToast('Server error occurred.', 'error');
            })
            .always(function () {
                $btn.prop('disabled', false).text('Run Search & Replace');
            });
    });

    // Migration Cleanup
    $('#btn-cleanup').off('click').on('click', function (e) {
        e.preventDefault();
        const $btn = $(this);

        $btn.prop('disabled', true).text('Running...');

        $.post(assertivlogixBackup.ajaxUrl, {
            action: 'assertivlogix_backup_cleanup',
            _wpnonce: assertivlogixBackup.nonce
        })
            .done(function (res) {
                if (res.success) {
                    showToast(res.data.message, 'success');
                } else {
                    showToast('Error: ' + res.data, 'error');
                }
            })
            .fail(function () {
                showToast('Server error occurred.', 'error');
            })
            .always(function () {
                $btn.prop('disabled', false).text('Run Cleanup Tasks');
            });
    });

    // Selective Backup - Store selections
    let selectedPlugins = [];
    let selectedThemes = [];
    let selectedTables = [];

    // Use event delegation for buttons that might be loaded dynamically
    // Select Plugins
    $(document).off('click', '#btn-select-plugins').on('click', '#btn-select-plugins', function (e) {
        e.preventDefault();
        const $btn = $(this);
        $btn.prop('disabled', true).text('Loading...');

        $.post(assertivlogixBackup.ajaxUrl, {
            action: 'assertivlogix_backup_get_plugins',
            _wpnonce: assertivlogixBackup.nonce
        })
            .done(function (res) {
                if (res.success && res.data && Array.isArray(res.data) && res.data.length > 0) {
                    console.log('Plugins loaded:', res.data.length);
                    showSelectionModal('Plugins', res.data, selectedPlugins, function (selected) {
                        selectedPlugins = selected;
                        $('#selected-plugins-count').text(selected.length + ' selected');
                    });
                } else {
                    const errorMsg = res.data && res.data.message ? res.data.message : (res.data || 'No plugins found');
                    console.error('Failed to load plugins:', errorMsg);
                    showToast('Failed to load plugins: ' + errorMsg, 'error');
                }
            })
            .fail(function (xhr, status, error) {
                console.error('AJAX error loading plugins:', status, error);
                showToast('Server error occurred while loading plugins.', 'error');
            })
            .always(function () {
                $btn.prop('disabled', false).text('Select Plugins');
            });
    });

    // Select Themes
    $(document).off('click', '#btn-select-themes').on('click', '#btn-select-themes', function (e) {
        e.preventDefault();
        const $btn = $(this);
        $btn.prop('disabled', true).text('Loading...');

        $.post(assertivlogixBackup.ajaxUrl, {
            action: 'assertivlogix_backup_get_themes',
            _wpnonce: assertivlogixBackup.nonce
        })
            .done(function (res) {
                if (res.success && res.data && Array.isArray(res.data) && res.data.length > 0) {
                    console.log('Themes loaded:', res.data.length);
                    showSelectionModal('Themes', res.data, selectedThemes, function (selected) {
                        selectedThemes = selected;
                        $('#selected-themes-count').text(selected.length + ' selected');
                    });
                } else {
                    const errorMsg = res.data && res.data.message ? res.data.message : (res.data || 'No themes found');
                    console.error('Failed to load themes:', errorMsg);
                    showToast('Failed to load themes: ' + errorMsg, 'error');
                }
            })
            .fail(function (xhr, status, error) {
                console.error('AJAX error loading themes:', status, error);
                showToast('Server error occurred while loading themes.', 'error');
            })
            .always(function () {
                $btn.prop('disabled', false).text('Select Themes');
            });
    });

    // Select Tables
    $(document).off('click', '#btn-select-tables').on('click', '#btn-select-tables', function (e) {
        e.preventDefault();
        const $btn = $(this);
        $btn.prop('disabled', true).text('Loading...');

        $.post(assertivlogixBackup.ajaxUrl, {
            action: 'assertivlogix_backup_get_tables',
            _wpnonce: assertivlogixBackup.nonce
        })
            .done(function (res) {
                if (res.success && res.data && Array.isArray(res.data) && res.data.length > 0) {
                    console.log('Database tables loaded:', res.data.length);
                    showSelectionModal('Database Tables', res.data, selectedTables, function (selected) {
                        selectedTables = selected;
                        $('#selected-tables-count').text(selected.length + ' selected');
                    });
                } else {
                    const errorMsg = res.data && res.data.message ? res.data.message : (res.data || 'No tables found');
                    console.error('Failed to load database tables:', errorMsg);
                    showToast('Failed to load database tables: ' + errorMsg, 'error');
                }
            })
            .fail(function (xhr, status, error) {
                console.error('AJAX error loading database tables:', status, error);
                showToast('Server error occurred while loading database tables.', 'error');
            })
            .always(function () {
                $btn.prop('disabled', false).text('Select Tables');
            });
    });

    // Create Selective Backup
    $('#btn-selective-backup').off('click').on('click', function (e) {
        e.preventDefault();
        const $btn = $(this);

        const includeUploads = $('#include-uploads').is(':checked');
        
        if (selectedPlugins.length === 0 && selectedThemes.length === 0 && selectedTables.length === 0 && !includeUploads) {
            showToast('Please select at least one item to backup.', 'error');
            return;
        }

        $btn.prop('disabled', true).html('<span class="dashicons dashicons-update" style="animation: spin 1s linear infinite;"></span> Creating Backup...');

        $.post(assertivlogixBackup.ajaxUrl, {
            action: 'assertivlogix_backup_selective',
            plugins: JSON.stringify(selectedPlugins),
            themes: JSON.stringify(selectedThemes),
            tables: JSON.stringify(selectedTables),
            uploads: includeUploads ? 'true' : 'false',
            _wpnonce: assertivlogixBackup.nonce
        })
            .done(function (res) {
                if (res.success) {
                    showToast('Selective backup created: ' + res.data.filename, 'success');
                    // Reset selections
                    selectedPlugins = [];
                    selectedThemes = [];
                    selectedTables = [];
                    $('#selected-plugins-count, #selected-themes-count, #selected-tables-count').text('0 selected');
                    $('#include-uploads').prop('checked', true);
                } else {
                    showToast('Error: ' + (res.data.message || res.data), 'error');
                }
            })
            .fail(function (xhr, status, error) {
                console.error('Selective backup error:', status, error);
                showToast('Server error occurred. Please check console for details.', 'error');
            })
            .always(function () {
                $btn.prop('disabled', false).html('<span class="dashicons dashicons-filter"></span> Create Selective Backup');
            });
    });

    // Helper: Show selection modal
    function showSelectionModal(title, items, currentSelection, callback) {
        // Remove any existing modal first
        $('.assertivlogix-modal-overlay').remove();
        $('body').removeClass('modal-open');
        
        // Validate items
        if (!items || !Array.isArray(items) || items.length === 0) {
            showToast('No items available to select.', 'error');
            return;
        }
        
        // Create modal HTML
        let html = '<div class="assertivlogix-modal-overlay" id="assertivlogix-selection-modal">';
        html += '<div class="assertivlogix-modal assertivlogix-selection-modal-content">';
        html += '<div class="assertivlogix-modal-header">';
        html += '<h3>Select ' + title + '</h3>';
        html += '<button type="button" class="assertivlogix-modal-close" aria-label="Close">&times;</button>';
        html += '</div>';
        html += '<div class="assertivlogix-modal-body">';
        html += '<div class="assertivlogix-selection-list">';

        items.forEach(function (item) {
            // Handle both slug (plugins/themes) and name (tables)
            const itemValue = item.slug || item.name;
            const itemName = item.name || itemValue;
            const isChecked = currentSelection.includes(itemValue) ? 'checked' : '';
            html += '<label class="assertivlogix-selection-item">';
            html += '<input type="checkbox" value="' + escapeHtml(itemValue) + '" ' + isChecked + '>';
            html += '<span class="item-name">' + escapeHtml(itemName) + '</span>';
            if (item.version) html += '<span class="item-version">v' + escapeHtml(item.version) + '</span>';
            if (item.size) html += '<span class="item-size">' + escapeHtml(item.size) + '</span>';
            if (item.active !== undefined) {
                const statusText = item.active ? 'Active' : 'Inactive';
                const statusClass = item.active ? 'active' : 'inactive';
                html += '<span class="item-status" data-status="' + statusClass + '">' + statusText + '</span>';
            }
            html += '</label>';
        });

        html += '</div>';
        html += '</div>';
        html += '<div class="assertivlogix-modal-footer">';
        html += '<button type="button" class="button button-primary assertivlogix-modal-save">Save Selection</button>';
        html += '<button type="button" class="button assertivlogix-modal-cancel">Cancel</button>';
        html += '</div>';
        html += '</div>';
        html += '</div>';

        // Add to page
        const $modalOverlay = $(html);
        $('body').append($modalOverlay);
        
        // Lock body scroll
        $('body').addClass('modal-open');
        
        // Force display
        $modalOverlay.css({
            'display': 'flex',
            'visibility': 'visible',
            'opacity': '1'
        });

        // Handle close button
        $modalOverlay.find('.assertivlogix-modal-close').on('click', function (e) {
            e.preventDefault();
            e.stopPropagation();
            closeSelectionModal($modalOverlay);
        });
        
        // Handle cancel button
        $modalOverlay.find('.assertivlogix-modal-cancel').on('click', function (e) {
            e.preventDefault();
            e.stopPropagation();
            closeSelectionModal($modalOverlay);
        });
        
        // Close on overlay click (but not on modal content)
        $modalOverlay.on('click', function (e) {
            if (e.target === this || $(e.target).hasClass('assertivlogix-modal-overlay')) {
                closeSelectionModal($modalOverlay);
            }
        });
        
        // Prevent modal content clicks from closing
        $modalOverlay.find('.assertivlogix-modal').on('click', function (e) {
            e.stopPropagation();
        });

        // Handle save button
        $modalOverlay.find('.assertivlogix-modal-save').on('click', function (e) {
            e.preventDefault();
            e.stopPropagation();
            const selected = [];
            $modalOverlay.find('.assertivlogix-selection-item input:checked').each(function () {
                selected.push($(this).val());
            });
            callback(selected);
            closeSelectionModal($modalOverlay);
        });
        
        // Close on Escape key
        const escapeHandler = function(e) {
            if (e.key === 'Escape' || e.keyCode === 27) {
                closeSelectionModal($modalOverlay);
                $(document).off('keydown.selectionModal', escapeHandler);
            }
        };
        $(document).on('keydown.selectionModal', escapeHandler);
    }
    
    // Helper: Close selection modal
    function closeSelectionModal($modalOverlay) {
        if ($modalOverlay && $modalOverlay.length) {
            $modalOverlay.fadeOut(200, function() {
                $(this).remove();
                $('body').removeClass('modal-open');
            });
        } else {
            $('.assertivlogix-modal-overlay#assertivlogix-selection-modal').remove();
            $('body').removeClass('modal-open');
        }
    }
    
    // Helper: Escape HTML
    function escapeHtml(text) {
        const map = {
            '&': '&amp;',
            '<': '&lt;',
            '>': '&gt;',
            '"': '&quot;',
            "'": '&#039;'
        };
        return String(text).replace(/[&<>"']/g, function(m) { return map[m]; });
    }

    // License Management
    
    // Validate License
    $('#btn-validate-license').off('click').on('click', function(e) {
        e.preventDefault();
        const $btn = $(this);
        const licenseKey = $('#license-key').val().trim();
        
        if (!licenseKey) {
            showToast('Please enter a license key.', 'error');
            return;
        }
        
        $btn.prop('disabled', true).text('Validating...');
        
        // Log request details for debugging
        if (assertivlogixBackup.debug) {
            console.log('=== License Validation Request ===');
            console.log('Action:', 'assertivlogix_backup_validate_license');
            console.log('License Key:', licenseKey.substring(0, 10) + '...'); // Only log partial key
            console.log('AJAX URL:', assertivlogixBackup.ajaxUrl);
            console.log('Nonce:', assertivlogixBackup.nonce ? 'present' : 'missing');
        }
        
        $.ajax({
            url: assertivlogixBackup.ajaxUrl,
            type: 'POST',
            data: {
                action: 'assertivlogix_backup_validate_license',
                license_key: licenseKey,
                _wpnonce: assertivlogixBackup.nonce
            },
            dataType: 'json',
            timeout: 20000, // 20 second timeout
        })
        .done(function(res) {
            if (assertivlogixBackup.debug) {
                console.log('=== License Validation Response ===');
                console.log('Full Response:', res);
                console.log('Response Success:', res && res.success);
                console.log('Response Data:', res && res.data);
            }
            
            if (res && res.success) {
                const message = (res.data && res.data.message) ? res.data.message : 'License activated successfully!';
                showToast(message, 'success');
                
                // Clear any cached license data
                if (assertivlogixBackup.license) {
                    assertivlogixBackup.license.is_valid = true;
                    assertivlogixBackup.license.status = 'valid';
                }
                
                // Reload page after a short delay to show updated status
                setTimeout(function() {
                    // Force reload without cache
                    window.location.href = window.location.href.split('&')[0].split('?')[0] + '?page=assertivlogix-backup&tab=settings&_=' + Date.now();
                }, 1500);
            } else {
                const errorMsg = (res && res.data && res.data.message) ? res.data.message : 
                                (res && typeof res.data === 'string' ? res.data : 'Failed to activate license');
                if (assertivlogixBackup.debug) {
                    console.error('License validation failed:', res);
                }
                showToast('Error: ' + errorMsg, 'error');
                $btn.prop('disabled', false).text('Activate License');
            }
        })
        .fail(function(xhr, status, error) {
            if (assertivlogixBackup.debug) {
                console.error('=== License Validation AJAX Error ===');
                console.error('Status:', status);
                console.error('Error:', error);
                console.error('Response Text:', xhr.responseText);
                console.error('Status Code:', xhr.status);
                console.error('Full XHR Object:', xhr);
            }
            
            let errorMsg = 'Server error occurred.';
            
            if (xhr.status === 0) {
                errorMsg = 'Network error: Could not connect to server. Please check your connection.';
            } else if (xhr.status === 404) {
                errorMsg = 'AJAX endpoint not found. Please refresh the page.';
            } else if (xhr.status === 500) {
                errorMsg = 'Server error. Please check WordPress debug log.';
            } else if (xhr.responseJSON && xhr.responseJSON.data && xhr.responseJSON.data.message) {
                errorMsg = xhr.responseJSON.data.message;
            } else if (xhr.responseText) {
                try {
                    const response = JSON.parse(xhr.responseText);
                    if (response.data && response.data.message) {
                        errorMsg = response.data.message;
                    } else if (response.data) {
                        errorMsg = typeof response.data === 'string' ? response.data : JSON.stringify(response.data);
                    }
                } catch(e) {
                    errorMsg = 'Error: ' + (xhr.responseText.substring(0, 100) || error);
                }
            }
            
            showToast('Error: ' + errorMsg, 'error');
            $btn.prop('disabled', false).text('Activate License');
        });
    });
    
    // Deactivate License
    $('#btn-deactivate-license').off('click').on('click', function(e) {
        e.preventDefault();
        const $btn = $(this);
        
        if (!confirm('Are you sure you want to deactivate your license? Premium features will be disabled.')) {
            return;
        }
        
        $btn.prop('disabled', true).text('Deactivating...');
        
        $.post(assertivlogixBackup.ajaxUrl, {
            action: 'assertivlogix_backup_deactivate_license',
            _wpnonce: assertivlogixBackup.nonce
        })
        .done(function(res) {
            if (res.success) {
                showToast(res.data.message, 'success');
                setTimeout(function() {
                    location.reload();
                }, 1500);
            } else {
                showToast('Error: ' + res.data, 'error');
                $btn.prop('disabled', false).text('Deactivate License');
            }
        })
        .fail(function() {
            showToast('Server error occurred.', 'error');
            $btn.prop('disabled', false).text('Deactivate License');
        });
    });
    
    // Save Retention Policy (Premium)
    $('#btn-save-retention').off('click').on('click', function(e) {
        e.preventDefault();
        const $btn = $(this);
        
        const policy = {
            keep_daily: parseInt($('#retention-daily').val()) || 7,
            keep_weekly: parseInt($('#retention-weekly').val()) || 4,
            keep_monthly: parseInt($('#retention-monthly').val()) || 6,
            max_backups: parseInt($('#retention-max').val()) || 50
        };
        
        $btn.prop('disabled', true).text('Saving...');
        
        $.post(assertivlogixBackup.ajaxUrl, {
            action: 'assertivlogix_backup_save_retention',
            keep_daily: policy.keep_daily,
            keep_weekly: policy.keep_weekly,
            keep_monthly: policy.keep_monthly,
            max_backups: policy.max_backups,
            _wpnonce: assertivlogixBackup.nonce
        })
        .done(function(res) {
            if (res.success) {
                showToast(res.data.message, 'success');
            } else {
                showToast('Error: ' + res.data, 'error');
            }
        })
        .fail(function() {
            showToast('Server error occurred.', 'error');
        })
        .always(function() {
            $btn.prop('disabled', false).text('Save Retention Policy');
        });
    });
    
    // Direct Site-to-Site Migration (Premium)
    // Direct Site-to-Site Migration - Link Site
    $(document).on('click', '#btn-link-site', function(e) {
        e.preventDefault();
        const $btn = $(this);
        const remoteUrl = $('#linked-site-url').val().trim();
        const apiKey = $('#linked-site-api-key').val().trim();
        
        if (!remoteUrl || !apiKey) {
            showToast('Please enter both remote site URL and API key.', 'error');
            return;
        }
        
        // Validate URL format
        try {
            new URL(remoteUrl);
        } catch (e) {
            showToast('Please enter a valid URL (e.g., https://example.com).', 'error');
            return;
        }
        
        $btn.prop('disabled', true).html('<span class="dashicons dashicons-update" style="animation: spin 1s linear infinite;"></span> Linking...');
        
        $.post(assertivlogixBackup.ajaxUrl, {
            action: 'assertivlogix_backup_link_site',
            remote_url: remoteUrl,
            api_key: apiKey,
            _wpnonce: assertivlogixBackup.nonce
        })
        .done(function(res) {
            if (res.success) {
                const message = res.data && res.data.message ? res.data.message : 'Site linked successfully!';
                showToast(message, 'success');
                // Optionally show linked site info
                if (res.data && res.data.linked_url) {
                    $('#linked-site-url').val(res.data.linked_url);
                }
            } else {
                const errorMsg = res.data && res.data.message ? res.data.message : (typeof res.data === 'string' ? res.data : 'Failed to link site');
                showToast('Error: ' + errorMsg, 'error');
            }
        })
        .fail(function(xhr, status, error) {
            console.error('Link site error:', status, error, xhr.responseText);
            showToast('Server error occurred. Please check console for details.', 'error');
        })
        .always(function() {
            $btn.prop('disabled', false).html('Link Site');
        });
    });
    
    // Direct Site-to-Site Migration - Migrate to Linked Site
    $(document).on('click', '#btn-migrate-to-linked', function(e) {
        e.preventDefault();
        const $btn = $(this);
        
        // Check if site is linked
        const remoteUrl = $('#linked-site-url').val().trim();
        if (!remoteUrl) {
            showToast('Please link a site first before migrating.', 'error');
            return;
        }
        
        if (!confirm('This will create a backup of your current site and migrate it to the linked site.\n\nThis process may take several minutes. Continue?')) {
            return;
        }
        
        $btn.prop('disabled', true).html('<span class="dashicons dashicons-update" style="animation: spin 1s linear infinite;"></span> Migrating...');
        
        $.post(assertivlogixBackup.ajaxUrl, {
            action: 'assertivlogix_backup_migrate_to_linked',
            _wpnonce: assertivlogixBackup.nonce
        })
        .done(function(res) {
            if (res.success) {
                const message = res.data && res.data.message ? res.data.message : 'Migration completed successfully!';
                showToast(message, 'success');
                // Optionally reload page after success
                setTimeout(function() {
                    if (confirm('Migration completed! Would you like to reload the page?')) {
                        location.reload();
                    }
                }, 2000);
            } else {
                const errorMsg = res.data && res.data.message ? res.data.message : (typeof res.data === 'string' ? res.data : 'Migration failed');
                showToast('Error: ' + errorMsg, 'error');
            }
        })
        .fail(function(xhr, status, error) {
            console.error('Migration error:', status, error, xhr.responseText);
            showToast('Server error occurred. Please check console for details.', 'error');
        })
        .always(function() {
            $btn.prop('disabled', false).html('Migrate to Linked Site');
        });
    });
    
    // Multisite Migration (Premium)
    $('#btn-migrate-multisite').off('click').on('click', function(e) {
        e.preventDefault();
        const $btn = $(this);
        const backupFile = $('#multisite-backup-file').val();
        const siteDomain = $('#multisite-domain').val().trim();
        
        if (!backupFile || !siteDomain) {
            showToast('Please select a backup file and enter site domain.', 'error');
            return;
        }
        
        if (!confirm('This will create a new site in the multisite network. Continue?')) {
            return;
        }
        
        $btn.prop('disabled', true).text('Migrating...');
        
        $.post(assertivlogixBackup.ajaxUrl, {
            action: 'assertivlogix_backup_migrate_to_multisite',
            backup_file: backupFile,
            site_domain: siteDomain,
            site_path: '/',
            _wpnonce: assertivlogixBackup.nonce
        })
        .done(function(res) {
            if (res.success) {
                showToast(res.data.message, 'success');
            } else {
                showToast('Error: ' + res.data, 'error');
            }
        })
        .fail(function() {
            showToast('Server error occurred.', 'error');
        })
        .always(function() {
            $btn.prop('disabled', false).text('Migrate to Multisite');
        });
    });
    
    // Cloud Storage Connection (Premium)
    
    // Verify modal exists on page load
    if ($('#assertivlogix-storage-modal').length === 0) {
        console.warn('Storage modal not found in DOM. It may not be rendered yet.');
    }
    
    // Open connection modal
    $(document).on('click', '[data-action="connect"]', function(e) {
        e.preventDefault();
        const $btn = $(this);
        const provider = $btn.data('provider');
        
        if (!provider) {
            showToast('Error: Provider not specified.', 'error');
            return;
        }
        
        $('#storage-provider').val(provider);
        
        // Hide all forms first
        $('#storage-form-google-drive, #storage-form-dropbox, #storage-form-onedrive, #storage-form-s3').hide();
        
        // Show appropriate form based on provider
        if (provider === 's3') {
            $('#storage-form-s3').show();
            $('#storage-modal-title').text('Connect Amazon S3');
        } else if (provider === 'google_drive') {
            $('#storage-form-google-drive').show();
            $('#storage-modal-title').text('Connect Google Drive');
        } else if (provider === 'dropbox') {
            $('#storage-form-dropbox').show();
            $('#storage-modal-title').text('Connect Dropbox');
        } else if (provider === 'onedrive') {
            $('#storage-form-onedrive').show();
            $('#storage-modal-title').text('Connect OneDrive');
        }
        
        // Clear form
        const $form = $('#storage-connection-form');
        if ($form.length) {
            $form[0].reset();
        }
        
        // Show modal - BULLETPROOF APPROACH
        let $modal = $('#assertivlogix-storage-modal');
        
        if ($modal.length === 0) {
            console.error('Modal element not found!');
            showToast('Error: Modal not found. Please refresh the page.', 'error');
            return;
        }
        
        // Move modal to body if it's not already there
        if ($modal.parent().is('body') === false) {
            $modal.detach().appendTo('body');
        }
        
        const modalEl = $modal[0];
        const vw = window.innerWidth || document.documentElement.clientWidth;
        const vh = window.innerHeight || document.documentElement.clientHeight;
        
        // COMPLETE STYLE RESET AND LOCK - Use viewport units
        const lockStyles = function() {
            // Lock overlay with viewport units (more stable than pixels)
            modalEl.style.cssText = `
                position: fixed !important;
                top: 0 !important;
                left: 0 !important;
                right: 0 !important;
                bottom: 0 !important;
                width: 100vw !important;
                height: 100vh !important;
                min-width: 100vw !important;
                min-height: 100vh !important;
                max-width: 100vw !important;
                max-height: 100vh !important;
                display: flex !important;
                visibility: visible !important;
                opacity: 1 !important;
                z-index: 99999999 !important;
                margin: 0 !important;
                padding: 0 !important;
                border: 0 !important;
                transform: none !important;
                transition: opacity 0.3s ease !important;
                animation: none !important;
                overflow: hidden !important;
                box-sizing: border-box !important;
                background: rgba(15, 23, 42, 0.95) !important;
            `;
            
            // Lock inner modal with viewport units
            const $innerModal = $modal.find('.assertivlogix-modal, .assertivlogix-storage-modal');
            if ($innerModal.length) {
                const innerEl = $innerModal[0];
                innerEl.style.cssText = `
                    width: 100vw !important;
                    height: 100vh !important;
                    min-width: 100vw !important;
                    min-height: 100vh !important;
                    max-width: 100vw !important;
                    max-height: 100vh !important;
                    margin: 0 !important;
                    padding: 0 !important;
                    transform: none !important;
                    transition: none !important;
                    animation: slideInModal 0.4s cubic-bezier(0.4, 0, 0.2, 1) !important;
                    box-sizing: border-box !important;
                    position: relative !important;
                `;
            }
        };
        
        // Apply styles immediately
        lockStyles();
        $modal.addClass('modal-show');
        
        // Use MutationObserver to prevent ANY style changes
        const observer = new MutationObserver(function(mutations) {
            mutations.forEach(function(mutation) {
                if (mutation.type === 'attributes' && (mutation.attributeName === 'style' || mutation.attributeName === 'class')) {
                    // Re-lock styles if they change
                    setTimeout(lockStyles, 0);
                }
            });
        });
        
        observer.observe(modalEl, {
            attributes: true,
            attributeFilter: ['style', 'class'],
            subtree: true,
            childList: false,
            characterData: false
        });
        
        // Store observer for cleanup
        $modal.data('style-observer', observer);
        
        // Prevent body scroll
        $('body').addClass('modal-open');
        document.body.style.cssText = 'overflow: hidden !important; position: fixed !important; width: 100% !important; height: 100% !important;';
        
        // Lock on resize - use viewport units
        const resizeHandler = function() {
            modalEl.style.setProperty('width', '100vw', 'important');
            modalEl.style.setProperty('height', '100vh', 'important');
            modalEl.style.setProperty('min-width', '100vw', 'important');
            modalEl.style.setProperty('min-height', '100vh', 'important');
            modalEl.style.setProperty('max-width', '100vw', 'important');
            modalEl.style.setProperty('max-height', '100vh', 'important');
            
            const $innerModal = $modal.find('.assertivlogix-modal, .assertivlogix-storage-modal');
            if ($innerModal.length) {
                $innerModal[0].style.setProperty('width', '100vw', 'important');
                $innerModal[0].style.setProperty('height', '100vh', 'important');
                $innerModal[0].style.setProperty('min-width', '100vw', 'important');
                $innerModal[0].style.setProperty('min-height', '100vh', 'important');
                $innerModal[0].style.setProperty('max-width', '100vw', 'important');
                $innerModal[0].style.setProperty('max-height', '100vh', 'important');
            }
        };
        
        $(window).on('resize', resizeHandler);
        $modal.data('resize-handler', resizeHandler);
        
        // Scroll to top
        $modal.scrollTop(0);
        
        // Focus on first input
        setTimeout(function() {
            const firstInput = $modal.find('input[type="text"], input[type="password"]').first();
            if (firstInput.length) {
                firstInput.focus();
            }
        }, 200);
    });
    
    // Close storage modal only (not selection modal)
    $(document).on('click', '#assertivlogix-storage-modal .assertivlogix-modal-close, #assertivlogix-storage-modal .assertivlogix-modal-cancel', function(e) {
        e.preventDefault();
        e.stopPropagation();
        closeStorageModal();
    });
    
    $(document).on('click', '#assertivlogix-storage-modal', function(e) {
        if (e.target === this) {
            closeStorageModal();
        }
    });
    
    // Close modal function
    function closeStorageModal() {
        const $modal = $('#assertivlogix-storage-modal');
        
        // Stop MutationObserver
        const observer = $modal.data('style-observer');
        if (observer) {
            observer.disconnect();
            $modal.removeData('style-observer');
        }
        
        // Remove resize handler
        const resizeHandler = $modal.data('resize-handler');
        if (resizeHandler) {
            $(window).off('resize', resizeHandler);
            $modal.removeData('resize-handler');
        }
        
        // Remove show class
        $modal.removeClass('modal-show');
        
        // Hide modal completely
        const modalEl = $modal[0];
        modalEl.style.cssText = 'display: none !important; visibility: hidden !important; opacity: 0 !important;';
        
        // Restore body scroll
        $('body').removeClass('modal-open');
        document.body.style.cssText = '';
    }
    
    // Close on Escape key
    $(document).on('keydown', function(e) {
        if (e.key === 'Escape' && $('#assertivlogix-storage-modal').is(':visible')) {
            closeStorageModal();
        }
    });
    
    // Save storage connection
    $('#btn-save-storage-connection').off('click').on('click', function(e) {
        e.preventDefault();
        const $btn = $(this);
        const provider = $('#storage-provider').val();
        
        if (!provider) {
            showToast('Invalid provider.', 'error');
            return;
        }
        
        // Collect credentials
        let credentials = {};
        
        if (provider === 's3') {
            credentials = {
                access_key: $('#storage-access-key').val().trim(),
                secret_key: $('#storage-secret-key').val().trim(),
                bucket: $('#storage-bucket').val().trim(),
                region: $('#storage-region').val() || 'us-east-1',
                folder: $('#storage-folder').val().trim()
            };
            
            if (!credentials.access_key || !credentials.secret_key || !credentials.bucket) {
                showToast('Please fill in all required fields.', 'error');
                return;
            }
        } else {
            // Get access token from the correct input field based on provider
            let tokenInputId = '#storage-access-token';
            if (provider === 'dropbox') {
                tokenInputId = '#storage-access-token-dropbox';
            } else if (provider === 'onedrive') {
                tokenInputId = '#storage-access-token-onedrive';
            }
            
            credentials = {
                access_token: $(tokenInputId).val().trim()
            };
            
            if (!credentials.access_token) {
                showToast('Please enter an access token.', 'error');
                return;
            }
        }
        
        $btn.prop('disabled', true).text('Connecting...');
        
        $.post(assertivlogixBackup.ajaxUrl, {
            action: 'assertivlogix_backup_connect_storage',
            provider: provider,
            access_token: credentials.access_token || '',
            access_key: credentials.access_key || '',
            secret_key: credentials.secret_key || '',
            bucket: credentials.bucket || '',
            region: credentials.region || '',
            folder: credentials.folder || '',
            _wpnonce: assertivlogixBackup.nonce
        })
        .done(function(res) {
            if (res.success) {
                showToast(res.data.message || 'Storage connected successfully!', 'success');
                closeStorageModal();
                setTimeout(function() {
                    location.reload();
                }, 1500);
            } else {
                const errorMsg = (res.data && res.data.message) ? res.data.message : 
                                (typeof res.data === 'string' ? res.data : 'Failed to connect storage');
                showToast('Error: ' + errorMsg, 'error');
                $btn.prop('disabled', false).text('Connect');
            }
        })
        .fail(function(xhr) {
            let errorMsg = 'Server error occurred.';
            if (xhr.responseJSON && xhr.responseJSON.data && xhr.responseJSON.data.message) {
                errorMsg = xhr.responseJSON.data.message;
            }
            showToast('Error: ' + errorMsg, 'error');
            $btn.prop('disabled', false).text('Connect');
        });
    });
    
    // Test storage connection
    $(document).on('click', '[data-action="test"]', function(e) {
        e.preventDefault();
        const $btn = $(this);
        const provider = $btn.data('provider');
        
        $btn.prop('disabled', true).text('Testing...');
        
        $.post(assertivlogixBackup.ajaxUrl, {
            action: 'assertivlogix_backup_test_storage',
            provider: provider,
            _wpnonce: assertivlogixBackup.nonce
        })
        .done(function(res) {
            if (res.success) {
                showToast(res.data.message || 'Connection test successful!', 'success');
            } else {
                const errorMsg = (res.data && res.data.message) ? res.data.message : 
                                (typeof res.data === 'string' ? res.data : 'Connection test failed');
                showToast('Error: ' + errorMsg, 'error');
            }
        })
        .fail(function() {
            showToast('Server error occurred.', 'error');
        })
        .always(function() {
            $btn.prop('disabled', false).text('Test Connection');
        });
    });
    
    // Disconnect storage
    $(document).on('click', '[data-action="disconnect"]', function(e) {
        e.preventDefault();
        const $btn = $(this);
        const provider = $btn.data('provider');
        
        if (!confirm('Are you sure you want to disconnect this storage provider? Backups will no longer be uploaded automatically.')) {
            return;
        }
        
        $btn.prop('disabled', true).text('Disconnecting...');
        
        $.post(assertivlogixBackup.ajaxUrl, {
            action: 'assertivlogix_backup_disconnect_storage',
            provider: provider,
            _wpnonce: assertivlogixBackup.nonce
        })
        .done(function(res) {
            if (res.success) {
                showToast(res.data.message || 'Storage disconnected successfully.', 'success');
                setTimeout(function() {
                    location.reload();
                }, 1500);
            } else {
                const errorMsg = (res.data && res.data.message) ? res.data.message : 
                                (typeof res.data === 'string' ? res.data : 'Failed to disconnect');
                showToast('Error: ' + errorMsg, 'error');
                $btn.prop('disabled', false).text('Disconnect');
            }
        })
        .fail(function() {
            showToast('Server error occurred.', 'error');
            $btn.prop('disabled', false).text('Disconnect');
        });
    });
    
    // Save Cloud Storage Settings
    $(document).on('click', '#btn-save-cloud-settings', function(e) {
        e.preventDefault();
        const $btn = $(this);
        
        // Collect settings
        const upload_only_to = [];
        $('.cloud-provider-checkbox:checked').each(function() {
            upload_only_to.push($(this).val());
        });
        
        const settings = {
            action: 'assertivlogix_backup_save_cloud_settings',
            auto_upload: $('#cloud-auto-upload').is(':checked') ? '1' : '0',
            instant_backup: $('#cloud-instant-backup').is(':checked') ? '1' : '0',
            upload_after_scheduled: $('#cloud-upload-after-scheduled').is(':checked') ? '1' : '0',
            upload_after_manual: $('#cloud-upload-after-manual').is(':checked') ? '1' : '0',
            delete_local_after_upload: $('#cloud-delete-local-after-upload').is(':checked') ? '1' : '0',
            retention_days: parseInt($('#cloud-retention-days').val()) || 30,
            upload_only_to: upload_only_to,
            notify_on_failure: $('#cloud-notify-on-failure').is(':checked') ? '1' : '0',
            compress_before_upload: $('#cloud-compress-before-upload').is(':checked') ? '1' : '0',
            upload_timeout: parseInt($('#cloud-upload-timeout').val()) || 300,
            _wpnonce: assertivlogixBackup.nonce
        };
        
        $btn.prop('disabled', true).html('<span class="dashicons dashicons-update" style="animation: spin 1s linear infinite;"></span> Saving...');
        
        $.post(assertivlogixBackup.ajaxUrl, settings)
        .done(function(res) {
            if (res.success) {
                showToast(res.data.message || 'Cloud settings saved successfully!', 'success');
            } else {
                showToast('Error: ' + (res.data && res.data.message ? res.data.message : (typeof res.data === 'string' ? res.data : 'Failed to save settings')), 'error');
            }
        })
        .fail(function(xhr, status, error) {
            console.error('Save cloud settings error:', error);
            showToast('Server error occurred: ' + error, 'error');
        })
        .always(function() {
            $btn.prop('disabled', false).html('<span class="dashicons dashicons-yes-alt"></span> Save Cloud Settings');
        });
    });
    
    // Test Cloud Upload
    $(document).on('click', '#btn-test-cloud-upload', function(e) {
        e.preventDefault();
        const $btn = $(this);
        
        if (!confirm('This will create a test backup and upload it to all connected cloud storage providers. Continue?')) {
            return;
        }
        
        $btn.prop('disabled', true).html('<span class="dashicons dashicons-update" style="animation: spin 1s linear infinite;"></span> Testing...');
        
        $.post(assertivlogixBackup.ajaxUrl, {
            action: 'assertivlogix_backup_test_cloud_upload',
            _wpnonce: assertivlogixBackup.nonce
        })
        .done(function(res) {
            if (res.success) {
                showToast(res.data.message || 'Cloud upload test successful!', 'success');
            } else {
                showToast('Error: ' + (res.data.message || res.data), 'error');
            }
        })
        .fail(function() {
            showToast('Server error occurred.', 'error');
        })
        .always(function() {
            $btn.prop('disabled', false).html('<span class="dashicons dashicons-admin-generic"></span> Test Cloud Upload');
        });
    });
});
