<?php
/**
 * Plugin Name: Assertivlogix Backup & Migration
 * Plugin URI: https://plugins.assertivlogix.com/plugin/assertivlogix-backup-pro
 * Description: Comprehensive backup, restore and migration solution for WordPress sites.
 * Version: 1.0.0
 * Author: Assertivlogix
 * Author URI: https://www.assertivlogix.com
 * License: GPL2
 * Text Domain: assertivlogix-backup-and-migration
 */

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

// Define plugin constants.
define('ASSERTIVLOGIX_BM_VERSION', '1.0.0');
define('ASSERTIVLOGIX_BM_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('ASSERTIVLOGIX_BM_PLUGIN_URL', plugin_dir_url(__FILE__));

// Simple autoloader for our classes
spl_autoload_register(function ($class) {
    if (strpos($class, 'AssertivlogixBackup\\') !== 0) {
        return;
    }
    
    // Map class names to file names
    $class_map = [
        'AssertivlogixBackup\\Activator' => 'Activator.php',
        'AssertivlogixBackup\\Deactivator' => 'Deactivator.php',
        'AssertivlogixBackup\\Backup' => 'Backup.php',
        'AssertivlogixBackup\\Restore' => 'Restore.php',
        'AssertivlogixBackup\\Scheduler' => 'Scheduler.php',
        'AssertivlogixBackup\\Migration' => 'Migration.php',
        'AssertivlogixBackup\\Admin' => 'class-assertivlogix-backup-admin.php',
        'AssertivlogixBackup\\CLI' => 'class-assertivlogix-backup-cli.php',
        'AssertivlogixBackup\\License' => 'License.php',
    ];
    
    // Handle Premium namespace classes
    if (strpos($class, 'AssertivlogixBackup\\Premium\\') === 0) {
        $class_name = str_replace('AssertivlogixBackup\\Premium\\', '', $class);
        $file = ASSERTIVLOGIX_BM_PLUGIN_DIR . 'includes/Premium/' . $class_name . '.php';
        if (file_exists($file)) {
            require_once $file;
        }
        return;
    }
    
    if (isset($class_map[$class])) {
        $file = ASSERTIVLOGIX_BM_PLUGIN_DIR . 'includes/' . $class_map[$class];
        if (file_exists($file)) {
            require_once $file;
        }
    }
});

// Initialize plugin.
function assertivlogix_bm_init() {
    // Register activation / deactivation hooks.
    register_activation_hook(__FILE__, ['AssertivlogixBackup\\Activator', 'activate']);
    register_deactivation_hook(__FILE__, ['AssertivlogixBackup\\Deactivator', 'deactivate']);

    // Initialise admin UI.
    if (is_admin()) {
        $admin = new AssertivlogixBackup\Admin();
        $admin->init();
    }

    // Initialise scheduled backups (use AdvancedScheduler if license is valid)
    if (class_exists('AssertivlogixBackup\\License') && \AssertivlogixBackup\License::has_feature('advanced_scheduling')) {
        $scheduler = new \AssertivlogixBackup\Premium\AdvancedScheduler();
    } else {
        $scheduler = new AssertivlogixBackup\Scheduler();
    }
    $scheduler->init();
    
    // Initialize premium features if license is valid
    if (class_exists('AssertivlogixBackup\\License') && \AssertivlogixBackup\License::is_valid()) {
        // Direct Site-to-Site Migration
        if (\AssertivlogixBackup\License::has_feature('site_to_site')) {
            $site_to_site = new \AssertivlogixBackup\Premium\DirectSiteToSite();
            $site_to_site->init();
        }
        
        // Multisite Support
        if (\AssertivlogixBackup\License::has_feature('multisite')) {
            $multisite = new \AssertivlogixBackup\Premium\MultisiteSupport();
            $multisite->init();
        }
        
        // Cloud Storage
        if (\AssertivlogixBackup\License::has_feature('cloud_storage')) {
            $cloud_storage = new \AssertivlogixBackup\Premium\CloudStorage();
            $cloud_storage->init();
        }
    }
    
    // Initialize WP-CLI commands if available
    if (defined('WP_CLI') && WP_CLI) {
        $cli = new AssertivlogixBackup\CLI();
    }
}
add_action('plugins_loaded', 'assertivlogix_bm_init');
