<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class AI_Growth_Suite_REST_API {

	public function __construct() {
		add_action( 'rest_api_init', array( $this, 'register_routes' ) );
	}

	public function register_routes() {
		$namespace = 'ai-growth-suite/v1';

		// Generate Product Description
		register_rest_route( $namespace, '/generate-description', array(
			'methods' => 'POST',
			'callback' => array( $this, 'generate_description' ),
			'permission_callback' => array( $this, 'check_admin_permission' ),
		) );

		// Generate Upsell Suggestions
		register_rest_route( $namespace, '/generate-upsells', array(
			'methods' => 'POST',
			'callback' => array( $this, 'generate_upsells' ),
			'permission_callback' => array( $this, 'check_admin_permission' ),
		) );

		// Get All Product IDs
		register_rest_route( $namespace, '/get-all-products', array(
			'methods' => 'GET',
			'callback' => array( $this, 'get_all_products' ),
			'permission_callback' => array( $this, 'check_admin_permission' ),
		) );

		// Delete Recommendation
		register_rest_route( $namespace, '/delete-recommendation', array(
			'methods' => 'POST',
			'callback' => array( $this, 'delete_recommendation' ),
			'permission_callback' => array( $this, 'check_admin_permission' ),
		) );

		// Fetch Analytics
		register_rest_route( $namespace, '/analytics', array(
			'methods' => 'GET',
			'callback' => array( $this, 'get_analytics' ),
			'permission_callback' => array( $this, 'check_admin_permission' ),
		) );

		// Create Dynamic Pricing Rule
		register_rest_route( $namespace, '/create-rule', array(
			'methods' => 'POST',
			'callback' => array( $this, 'create_rule' ),
			'permission_callback' => array( $this, 'check_admin_permission' ),
		) );

		// Delete Dynamic Pricing Rule
		register_rest_route( $namespace, '/delete-rule', array(
			'methods' => 'POST',
			'callback' => array( $this, 'delete_rule' ),
			'permission_callback' => array( $this, 'check_admin_permission' ),
		) );

		// Save Marketing Settings
		register_rest_route( $namespace, '/save-marketing-settings', array(
			'methods' => 'POST',
			'callback' => array( $this, 'save_marketing_settings' ),
			'permission_callback' => array( $this, 'check_admin_permission' ),
		) );

		// Get Email Template
		register_rest_route( $namespace, '/get-email-template', array(
			'methods' => 'GET',
			'callback' => array( $this, 'get_email_template' ),
			'permission_callback' => array( $this, 'check_admin_permission' ),
		) );

		// Save Email Template
		register_rest_route( $namespace, '/save-email-template', array(
			'methods' => 'POST',
			'callback' => array( $this, 'save_email_template' ),
			'permission_callback' => array( $this, 'check_admin_permission' ),
		) );

		// Create Email Flow
		register_rest_route( $namespace, '/create-flow', array(
			'methods' => 'POST',
			'callback' => array( $this, 'create_flow' ),
			'permission_callback' => array( $this, 'check_admin_permission' ),
		) );

		// Delete Email Flow
		register_rest_route( $namespace, '/delete-flow', array(
			'methods' => 'POST',
			'callback' => array( $this, 'delete_flow' ),
			'permission_callback' => array( $this, 'check_admin_permission' ),
		) );

		// Toggle Email Flow Status
		register_rest_route( $namespace, '/toggle-flow-status', array(
			'methods' => 'POST',
			'callback' => array( $this, 'toggle_flow_status' ),
			'permission_callback' => array( $this, 'check_admin_permission' ),
		) );

		// Send Recovery Email
		register_rest_route( $namespace, '/send-recovery-email', array(
			'methods' => 'POST',
			'callback' => array( $this, 'send_recovery_email' ),
			'permission_callback' => array( $this, 'check_admin_permission' ),
		) );

		// Create A/B Test
		register_rest_route( $namespace, '/create-ab-test', array(
			'methods' => 'POST',
			'callback' => array( $this, 'create_ab_test' ),
			'permission_callback' => array( $this, 'check_admin_permission' ),
		) );

		// Stop A/B Test
		register_rest_route( $namespace, '/stop-ab-test', array(
			'methods' => 'POST',
			'callback' => array( $this, 'stop_ab_test' ),
			'permission_callback' => array( $this, 'check_admin_permission' ),
		) );

		// Get A/B Test Report
		register_rest_route( $namespace, '/get-ab-test-report', array(
			'methods' => 'GET',
			'callback' => array( $this, 'get_ab_test_report' ),
			'permission_callback' => array( $this, 'check_admin_permission' ),
			'args' => array(
				'test_id' => array(
					'required' => true,
					'type' => 'integer',
					'sanitize_callback' => 'absint',
				),
			),
		) );
	}

	public function check_admin_permission() {
		return current_user_can( 'manage_options' );
	}

	// ... (Other methods) ...

	public function get_email_template( $request ) {
		$flow = sanitize_text_field( $request->get_param( 'flow' ) );
		$option_name = 'ai_growth_suite_template_' . $flow;
		$template = get_option( $option_name, array() );

		// Defaults with proper HTML layouts
		if ( empty( $template ) ) {
			if ( $flow === 'welcome' ) {
				$template = array(
					'subject' => 'Welcome to {store_name}! 🎉',
					'body' => '<!DOCTYPE html>
<html>
<head>
	<meta charset="UTF-8">
	<meta name="viewport" content="width=device-width, initial-scale=1.0">
	<title>Welcome Email</title>
</head>
<body style="margin: 0; padding: 0; font-family: Arial, sans-serif; background-color: #f7f7f7;">
	<table width="100%" cellpadding="0" cellspacing="0" style="background-color: #f7f7f7; padding: 20px 0;">
		<tr>
			<td align="center">
				<table width="600" cellpadding="0" cellspacing="0" style="background-color: #ffffff; border-radius: 8px; overflow: hidden; box-shadow: 0 2px 8px rgba(0,0,0,0.1);">
					<!-- Header -->
					<tr>
						<td style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); padding: 40px 30px; text-align: center;">
							<h1 style="color: #ffffff; margin: 0; font-size: 28px;">Welcome to {store_name}! 🎉</h1>
						</td>
					</tr>
					<!-- Body -->
					<tr>
						<td style="padding: 40px 30px;">
							<p style="font-size: 16px; line-height: 24px; color: #333333; margin: 0 0 20px 0;">Hi {name},</p>
							<p style="font-size: 16px; line-height: 24px; color: #333333; margin: 0 0 20px 0;">Welcome to {store_name}! We\'re thrilled to have you join our community.</p>
							
							<h2 style="font-size: 20px; color: #333333; margin: 30px 0 15px 0;">As a valued member, you now have access to:</h2>
							<ul style="font-size: 16px; line-height: 28px; color: #555555; margin: 0 0 20px 0; padding-left: 20px;">
								<li>Exclusive deals and promotions</li>
								<li>Early access to new products</li>
								<li>Special birthday discounts</li>
								<li>Priority customer support</li>
							</ul>
							
							<div style="background-color: #f0f9ff; border-left: 4px solid #667eea; padding: 20px; margin: 30px 0;">
								<p style="font-size: 16px; line-height: 24px; color: #333333; margin: 0;"><strong>🎁 Special Welcome Gift:</strong><br>Use code <strong style="color: #667eea; font-size: 18px;">WELCOME10</strong> for 10% off your first purchase!</p>
							</div>
							
							<div style="text-align: center; margin: 30px 0;">
								<a href="{store_url}" style="display: inline-block; background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: #ffffff; text-decoration: none; padding: 15px 40px; border-radius: 5px; font-size: 16px; font-weight: bold;">Shop Now</a>
							</div>
							
							<p style="font-size: 14px; line-height: 22px; color: #666666; margin: 30px 0 0 0;">If you have any questions, we\'re here to help!</p>
						</td>
					</tr>
					<!-- Footer -->
					<tr>
						<td style="background-color: #f7f7f7; padding: 30px; text-align: center; border-top: 1px solid #e0e0e0;">
							<p style="font-size: 14px; color: #999999; margin: 0 0 10px 0;">Best regards,<br>The {store_name} Team</p>
							<p style="font-size: 12px; color: #999999; margin: 10px 0 0 0;">You\'re receiving this email because you registered at {store_name}.<br><a href="{unsubscribe_url}" style="color: #667eea; text-decoration: none;">Unsubscribe</a></p>
						</td>
					</tr>
				</table>
			</td>
		</tr>
	</table>
</body>
</html>'
				);
			} elseif ( $flow === 'post_purchase' ) {
				$template = array(
					'subject' => 'Thank you for your order #{order_number}!',
					'body' => '<!DOCTYPE html>
<html>
<head>
	<meta charset="UTF-8">
	<meta name="viewport" content="width=device-width, initial-scale=1.0">
	<title>Order Confirmation</title>
</head>
<body style="margin: 0; padding: 0; font-family: Arial, sans-serif; background-color: #f7f7f7;">
	<table width="100%" cellpadding="0" cellspacing="0" style="background-color: #f7f7f7; padding: 20px 0;">
		<tr>
			<td align="center">
				<table width="600" cellpadding="0" cellspacing="0" style="background-color: #ffffff; border-radius: 8px; overflow: hidden; box-shadow: 0 2px 8px rgba(0,0,0,0.1);">
					<!-- Header -->
					<tr>
						<td style="background: linear-gradient(135deg, #11998e 0%, #38ef7d 100%); padding: 40px 30px; text-align: center;">
							<h1 style="color: #ffffff; margin: 0; font-size: 28px;">Thank You for Your Order! ✓</h1>
						</td>
					</tr>
					<!-- Body -->
					<tr>
						<td style="padding: 40px 30px;">
							<p style="font-size: 16px; line-height: 24px; color: #333333; margin: 0 0 20px 0;">Hi {name},</p>
							<p style="font-size: 16px; line-height: 24px; color: #333333; margin: 0 0 30px 0;">Thank you for your recent purchase! We truly appreciate your business.</p>
							
							<h2 style="font-size: 20px; color: #333333; margin: 0 0 15px 0; padding-bottom: 10px; border-bottom: 2px solid #11998e;">Order Summary</h2>
							<table width="100%" cellpadding="8" cellspacing="0" style="margin: 0 0 30px 0;">
								<tr>
									<td style="font-size: 14px; color: #666666; padding: 8px 0;">Order Number:</td>
									<td style="font-size: 14px; color: #333333; font-weight: bold; text-align: right; padding: 8px 0;">#{order_number}</td>
								</tr>
								<tr>
									<td style="font-size: 14px; color: #666666; padding: 8px 0;">Order Date:</td>
									<td style="font-size: 14px; color: #333333; text-align: right; padding: 8px 0;">{order_date}</td>
								</tr>
								<tr>
									<td style="font-size: 14px; color: #666666; padding: 8px 0; border-top: 1px solid #e0e0e0;">Order Total:</td>
									<td style="font-size: 18px; color: #11998e; font-weight: bold; text-align: right; padding: 8px 0; border-top: 1px solid #e0e0e0;">{order_total}</td>
								</tr>
							</table>
							
							<h3 style="font-size: 18px; color: #333333; margin: 30px 0 15px 0;">Items Ordered</h3>
							<div style="background-color: #f7f7f7; padding: 20px; border-radius: 5px; margin: 0 0 30px 0;">
								{order_items}
							</div>
							
							<h3 style="font-size: 18px; color: #333333; margin: 30px 0 15px 0;">Shipping Information</h3>
							<div style="background-color: #f7f7f7; padding: 20px; border-radius: 5px; margin: 0 0 20px 0;">
								<p style="font-size: 14px; line-height: 22px; color: #555555; margin: 0;">{shipping_address}</p>
							</div>
							<p style="font-size: 14px; color: #666666; margin: 0 0 5px 0;"><strong>Estimated Delivery:</strong> {estimated_delivery}</p>
							<p style="font-size: 14px; color: #666666; margin: 0 0 30px 0;"><a href="{tracking_url}" style="color: #11998e; text-decoration: none;">Track your order →</a></p>
							
							<div style="background-color: #f0f9ff; border-left: 4px solid #11998e; padding: 20px; margin: 30px 0;">
								<h3 style="font-size: 16px; color: #333333; margin: 0 0 10px 0;">Recommended For You</h3>
								<p style="font-size: 14px; line-height: 22px; color: #555555; margin: 0;">Based on your purchase, you might also like:</p>
								<div style="margin-top: 15px;">{recommended_products}</div>
							</div>
							
							<div style="text-align: center; margin: 30px 0;">
								<a href="{order_url}" style="display: inline-block; background: linear-gradient(135deg, #11998e 0%, #38ef7d 100%); color: #ffffff; text-decoration: none; padding: 15px 40px; border-radius: 5px; font-size: 16px; font-weight: bold;">View Order Details</a>
							</div>
							
							<p style="font-size: 14px; line-height: 22px; color: #666666; margin: 30px 0 0 0;">Need help? Contact us at <a href="mailto:{support_email}" style="color: #11998e; text-decoration: none;">{support_email}</a></p>
						</td>
					</tr>
					<!-- Footer -->
					<tr>
						<td style="background-color: #f7f7f7; padding: 30px; text-align: center; border-top: 1px solid #e0e0e0;">
							<p style="font-size: 14px; color: #999999; margin: 0 0 10px 0;">Thank you for choosing {store_name}!<br>The {store_name} Team</p>
							<p style="font-size: 12px; color: #999999; margin: 10px 0 0 0;"><a href="{unsubscribe_url}" style="color: #11998e; text-decoration: none;">Unsubscribe</a></p>
						</td>
					</tr>
				</table>
			</td>
		</tr>
	</table>
</body>
</html>'
				);
			} elseif ( $flow === 'win_back' ) {
				$template = array(
					'subject' => 'We miss you, {name}! Come back for 20% off 🎁',
					'body' => '<!DOCTYPE html>
<html>
<head>
	<meta charset="UTF-8">
	<meta name="viewport" content="width=device-width, initial-scale=1.0">
	<title>We Miss You</title>
</head>
<body style="margin: 0; padding: 0; font-family: Arial, sans-serif; background-color: #f7f7f7;">
	<table width="100%" cellpadding="0" cellspacing="0" style="background-color: #f7f7f7; padding: 20px 0;">
		<tr>
			<td align="center">
				<table width="600" cellpadding="0" cellspacing="0" style="background-color: #ffffff; border-radius: 8px; overflow: hidden; box-shadow: 0 2px 8px rgba(0,0,0,0.1);">
					<!-- Header -->
					<tr>
						<td style="background: linear-gradient(135deg, #f093fb 0%, #f5576c 100%); padding: 40px 30px; text-align: center;">
							<h1 style="color: #ffffff; margin: 0; font-size: 28px;">We Miss You! 💝</h1>
						</td>
					</tr>
					<!-- Body -->
					<tr>
						<td style="padding: 40px 30px;">
							<p style="font-size: 16px; line-height: 24px; color: #333333; margin: 0 0 20px 0;">Hi {name},</p>
							<p style="font-size: 16px; line-height: 24px; color: #333333; margin: 0 0 30px 0;">We noticed you haven\'t visited {store_name} in a while, and we miss you!</p>
							
							<h2 style="font-size: 20px; color: #333333; margin: 0 0 15px 0;">A lot has changed since your last visit:</h2>
							<ul style="font-size: 16px; line-height: 28px; color: #555555; margin: 0 0 30px 0; padding-left: 20px;">
								<li>New products you\'ll love</li>
								<li>Improved shopping experience</li>
								<li>Faster checkout process</li>
							</ul>
							
							<div style="background: linear-gradient(135deg, #f093fb 0%, #f5576c 100%); padding: 30px; border-radius: 8px; text-align: center; margin: 30px 0;">
								<h2 style="color: #ffffff; margin: 0 0 15px 0; font-size: 24px;">Special Comeback Offer</h2>
								<p style="color: #ffffff; font-size: 16px; margin: 0 0 20px 0;">We\'d love to welcome you back with</p>
								<div style="background-color: rgba(255,255,255,0.2); padding: 20px; border-radius: 5px; margin: 0 0 20px 0;">
									<p style="color: #ffffff; font-size: 36px; font-weight: bold; margin: 0;">20% OFF</p>
									<p style="color: #ffffff; font-size: 14px; margin: 10px 0 0 0;">your next purchase!</p>
								</div>
								<p style="color: #ffffff; font-size: 18px; margin: 0;"><strong>Use code:</strong> <span style="background-color: rgba(255,255,255,0.3); padding: 8px 20px; border-radius: 5px; display: inline-block; margin-top: 10px;">COMEBACK20</span></p>
								<p style="color: #ffffff; font-size: 14px; margin: 15px 0 0 0;">Valid until: {offer_expiry}</p>
							</div>
							
							<div style="text-align: center; margin: 30px 0;">
								<a href="{store_url}" style="display: inline-block; background: linear-gradient(135deg, #f093fb 0%, #f5576c 100%); color: #ffffff; text-decoration: none; padding: 15px 40px; border-radius: 5px; font-size: 16px; font-weight: bold;">Shop Now</a>
							</div>
							
							<div style="background-color: #f7f7f7; padding: 20px; border-radius: 5px; margin: 30px 0;">
								<h3 style="font-size: 16px; color: #333333; margin: 0 0 10px 0;">Here are some items we think you\'ll love:</h3>
								<div>{recommended_products}</div>
							</div>
							
							<p style="font-size: 14px; line-height: 22px; color: #666666; margin: 30px 0 0 0;">Questions? We\'re here to help at <a href="mailto:{support_email}" style="color: #f5576c; text-decoration: none;">{support_email}</a></p>
						</td>
					</tr>
					<!-- Footer -->
					<tr>
						<td style="background-color: #f7f7f7; padding: 30px; text-align: center; border-top: 1px solid #e0e0e0;">
							<p style="font-size: 14px; color: #999999; margin: 0 0 10px 0;">Hope to see you soon!<br>The {store_name} Team</p>
							<p style="font-size: 12px; color: #999999; margin: 10px 0 0 0;"><a href="{unsubscribe_url}" style="color: #f5576c; text-decoration: none;">Unsubscribe</a></p>
						</td>
					</tr>
				</table>
			</td>
		</tr>
	</table>
</body>
</html>'
				);
			} else {
				// Generic template for custom flows
				$template = array(
					'subject' => '{store_name} - Important Update',
					'body' => 'Hi {name},

[Your email content goes here]

Available tokens you can use:
• {name} - Customer\'s name
• {email} - Customer\'s email
• {store_name} - Your store name
• {store_url} - Your store URL
• {order_number} - Order number
• {order_date} - Order date
• {order_total} - Order total amount
• {order_items} - List of ordered items
• {order_url} - Link to view order
• {product_name} - Product name
• {product_url} - Product link
• {shipping_address} - Shipping address
• {tracking_url} - Shipment tracking link
• {support_email} - Support email
• {unsubscribe_url} - Unsubscribe link

Best regards,
The {store_name} Team

---
Unsubscribe: {unsubscribe_url}'
				);
			}
		}

		return rest_ensure_response( $template );
	}

	public function save_email_template( $request ) {
		$flow = sanitize_text_field( $request->get_param( 'flow' ) );
		$subject = sanitize_text_field( $request->get_param( 'subject' ) );
		$body = sanitize_textarea_field( $request->get_param( 'body' ) );
		$auto_send = $request->get_param( 'auto_send' ) !== 'false' && $request->get_param( 'auto_send' ) !== false;
		$send_timing = sanitize_text_field( $request->get_param( 'send_timing' ) );
		$delay_amount = intval( $request->get_param( 'delay_amount' ) );
		$specific_time = sanitize_text_field( $request->get_param( 'specific_time' ) );

		$option_name = 'ai_growth_suite_template_' . $flow;
		update_option( $option_name, array( 
			'subject' => $subject, 
			'body' => $body,
			'auto_send' => $auto_send,
			'send_timing' => $send_timing,
			'delay_amount' => $delay_amount,
			'specific_time' => $specific_time
		) );

		return rest_ensure_response( array( 'success' => true ) );
	}

	public function create_flow( $request ) {
		if ( ! function_exists( 'ai_growth_suite_table_exists' ) || ! ai_growth_suite_table_exists( 'email_flows' ) ) {
			return new WP_Error( 'table_missing', 'Database table not ready. Please deactivate and reactivate the plugin.', array( 'status' => 500 ) );
		}
		global $wpdb;
		$table_name = $wpdb->prefix . 'email_flows';

		$name = sanitize_text_field( $request->get_param( 'name' ) );
		$trigger = sanitize_text_field( $request->get_param( 'trigger' ) );
		$tone = sanitize_text_field( $request->get_param( 'tone' ) );

		// Create slug from name
		$slug = sanitize_title( $name );

		$inserted = $wpdb->insert( 
			$table_name, 
			array( 
				'flow_name' => $name,
				'flow_slug' => $slug,
				'trigger_type' => $trigger,
				'ai_tone' => $tone,
				'is_active' => 1,
				'created_at' => current_time( 'mysql' )
			), 
			array( '%s', '%s', '%s', '%s', '%d', '%s' ) 
		);

		if ( $inserted ) {
			return rest_ensure_response( array( 'success' => true, 'id' => $wpdb->insert_id, 'slug' => $slug ) );
		}
		return new WP_Error( 'create_failed', 'Failed to create flow.' );
	}

	public function delete_flow( $request ) {
		if ( ! function_exists( 'ai_growth_suite_table_exists' ) || ! ai_growth_suite_table_exists( 'email_flows' ) ) {
			return new WP_Error( 'table_missing', 'Database table not ready.', array( 'status' => 500 ) );
		}
		global $wpdb;
		$flow_id = $request->get_param( 'flow_id' );
		$table_name = $wpdb->prefix . 'email_flows';

		$deleted = $wpdb->delete( $table_name, array( 'id' => $flow_id ) );
		
		if ( $deleted ) {
			return rest_ensure_response( array( 'success' => true ) );
		}
		return new WP_Error( 'delete_failed', 'Failed to delete flow.' );
	}

	public function toggle_flow_status( $request ) {
		if ( ! function_exists( 'ai_growth_suite_table_exists' ) || ! ai_growth_suite_table_exists( 'email_flows' ) ) {
			return new WP_Error( 'table_missing', 'Database table not ready.', array( 'status' => 500 ) );
		}
		global $wpdb;
		$flow_id = $request->get_param( 'flow_id' );
		$is_active = $request->get_param( 'is_active' ) ? 1 : 0;
		$table_name = $wpdb->prefix . 'email_flows';

		$updated = $wpdb->update( 
			$table_name, 
			array( 'is_active' => $is_active ), 
			array( 'id' => $flow_id ),
			array( '%d' ),
			array( '%d' )
		);
		
		if ( $updated !== false ) {
			return rest_ensure_response( array( 'success' => true ) );
		}
		return new WP_Error( 'update_failed', 'Failed to update flow status.' );
	}

	public function get_all_products() {
		$product_ids = wc_get_products( array( 'limit' => -1, 'return' => 'ids' ) );
		$products = array();
		
		foreach ( $product_ids as $product_id ) {
			$product = wc_get_product( $product_id );
			if ( $product ) {
				$products[] = array(
					'id' => $product_id,
					'name' => $product->get_name()
				);
			}
		}
		
		return rest_ensure_response( $products );
	}

	public function delete_recommendation( $request ) {
		if ( ! function_exists( 'ai_growth_suite_table_exists' ) || ! ai_growth_suite_table_exists( 'ai_recommendations' ) ) {
			return new WP_Error( 'table_missing', 'Database table not ready.', array( 'status' => 500 ) );
		}
		global $wpdb;
		$product_id = $request->get_param( 'product_id' );
		$table_name = $wpdb->prefix . 'ai_recommendations';

		$deleted = $wpdb->delete( $table_name, array( 'product_id' => $product_id ) );
		
		if ( $deleted ) {
			return rest_ensure_response( array( 'success' => true ) );
		}
		return new WP_Error( 'delete_failed', 'Failed to delete recommendation.' );
	}

	public function create_rule( $request ) {
		if ( ! function_exists( 'ai_growth_suite_table_exists' ) || ! ai_growth_suite_table_exists( 'dynamic_pricing_rules' ) ) {
			return new WP_Error( 'table_missing', 'Database table not ready. Please deactivate and reactivate the plugin.', array( 'status' => 500 ) );
		}
		global $wpdb;
		$table_name = $wpdb->prefix . 'dynamic_pricing_rules';

		$id = intval( $request->get_param( 'id' ) );
		$name = sanitize_text_field( $request->get_param( 'name' ) );
		$condition_type = sanitize_text_field( $request->get_param( 'condition_type' ) );
		$action_type = sanitize_text_field( $request->get_param( 'action_type' ) );
		$amount = floatval( $request->get_param( 'amount' ) );

		// Construct JSONs
		$conditions = array( 'type' => $condition_type );
		$actions = array( 'type' => $action_type, 'amount' => $amount );

		$data = array( 
			'rule_name' => $name,
			'rule_type' => $condition_type,
			'conditions' => json_encode( $conditions ),
			'actions' => json_encode( $actions ),
			'is_active' => 1
		);
		$format = array( '%s', '%s', '%s', '%s', '%d' );

		if ( $id > 0 ) {
			// Update
			$updated = $wpdb->update( $table_name, $data, array( 'id' => $id ), $format, array( '%d' ) );
			if ( $updated !== false ) {
				return rest_ensure_response( array( 'success' => true, 'id' => $id, 'action' => 'updated' ) );
			}
		} else {
			// Insert
			$inserted = $wpdb->insert( $table_name, $data, $format );
			if ( $inserted ) {
				return rest_ensure_response( array( 'success' => true, 'id' => $wpdb->insert_id, 'action' => 'created' ) );
			}
		}

		return new WP_Error( 'save_failed', 'Failed to save rule.' );
	}

	public function delete_rule( $request ) {
		if ( ! function_exists( 'ai_growth_suite_table_exists' ) || ! ai_growth_suite_table_exists( 'dynamic_pricing_rules' ) ) {
			return new WP_Error( 'table_missing', 'Database table not ready.', array( 'status' => 500 ) );
		}
		global $wpdb;
		$rule_id = $request->get_param( 'rule_id' );
		$table_name = $wpdb->prefix . 'dynamic_pricing_rules';

		$deleted = $wpdb->delete( $table_name, array( 'id' => $rule_id ) );
		
		if ( $deleted ) {
			return rest_ensure_response( array( 'success' => true ) );
		}
		return new WP_Error( 'delete_failed', 'Failed to delete rule.' );
	}

	public function save_marketing_settings( $request ) {
		$tone = sanitize_text_field( $request->get_param( 'tone' ) );
		update_option( 'ai_growth_suite_brand_tone', $tone );
		return rest_ensure_response( array( 'success' => true ) );
	}

	public function generate_description( $request ) {
		$product_id = $request->get_param( 'product_id' );
		if ( ! $product_id ) {
			return new WP_Error( 'missing_param', 'Product ID is required', array( 'status' => 400 ) );
		}

		$ai_engine = new AI_Growth_Suite_AI_Engine();
		$result = $ai_engine->generate_product_description( $product_id );

		if ( is_wp_error( $result ) ) {
			return $result;
		}
		if ( ! $result || ! is_array( $result ) ) {
			return new WP_Error( 'generate_failed', 'Failed to generate description.', array( 'status' => 500 ) );
		}

		$product = wc_get_product( $product_id );
		if ( $product ) {
			$product->set_description( isset( $result['description'] ) ? $result['description'] : '' );
			$product->set_short_description( isset( $result['short_description'] ) ? $result['short_description'] : '' );
			$product->save();
			update_post_meta( $product_id, '_ai_optimized', true );
			update_post_meta( $product_id, '_ai_last_optimized', current_time( 'mysql' ) );
		}

		return rest_ensure_response( $result );
	}

	public function generate_upsells( $request ) {
		$product_id = $request->get_param( 'product_id' );
		if ( ! $product_id ) {
			return new WP_Error( 'missing_param', 'Product ID is required', array( 'status' => 400 ) );
		}

		$rec_engine = new AI_Growth_Suite_Recommendation_Engine();
		$result = $rec_engine->generate_and_save_recommendations( $product_id );

		return rest_ensure_response( $result );
	}

	public function get_analytics( $request ) {
		// Mock data for now
		$data = array(
			'views' => 100,
			'conversions' => 5,
			'revenue' => 500.00
		);
		return rest_ensure_response( $data );
	}

	public function send_recovery_email( $request ) {
		if ( ! function_exists( 'ai_growth_suite_table_exists' ) || ! ai_growth_suite_table_exists( 'abandoned_carts' ) ) {
			return new WP_Error( 'table_missing', 'Abandoned carts table not ready.', array( 'status' => 500 ) );
		}
		global $wpdb;
		$cart_id = intval( $request->get_param( 'cart_id' ) );
		$table_name = $wpdb->prefix . 'abandoned_carts';

		// Get cart data
		$cart = $wpdb->get_row( $wpdb->prepare( "SELECT * FROM $table_name WHERE id = %d", $cart_id ) );
		
		if ( ! $cart ) {
			return new WP_Error( 'cart_not_found', 'Cart not found.' );
		}
		
		// Get email template
		$template = get_option( 'ai_growth_suite_template_abandoned_cart', array() );
		$subject = isset( $template['subject'] ) ? $template['subject'] : 'Complete your purchase at {store_name}';
		$body = isset( $template['body'] ) ? $template['body'] : 'Hi {name}, You left items in your cart. Complete your purchase now!';
		
		// Replace tokens
		$store_name = get_bloginfo( 'name' );
		$subject = str_replace( '{store_name}', $store_name, $subject );
		$body = str_replace( '{store_name}', $store_name, $body );
		$body = str_replace( '{email}', $cart->user_email, $body );
		$body = str_replace( '{cart_total}', '$' . number_format( $cart->cart_total, 2 ), $body );
		
		// Send email
		$headers = array( 'Content-Type: text/html; charset=UTF-8' );
		$sent = wp_mail( $cart->user_email, $subject, $body, $headers );
		
		if ( $sent ) {
			$wpdb->update(
				$table_name,
				array( 'status' => 'emailed' ),
				array( 'id' => $cart_id ),
				array( '%s' ),
				array( '%d' )
			);
			if ( function_exists( 'ai_growth_suite_table_exists' ) && ai_growth_suite_table_exists( 'ai_email_log' ) ) {
				$log_table = $wpdb->prefix . 'ai_email_log';
				$wpdb->insert(
					$log_table,
					array(
						'email_to' => $cart->user_email,
						'email_subject' => $subject,
						'email_type' => 'abandoned_cart',
						'sent_at' => current_time( 'mysql' ),
						'status' => 'sent',
					),
					array( '%s', '%s', '%s', '%s', '%s' )
				);
			}
			return rest_ensure_response( array( 'success' => true ) );
		}
		
		return new WP_Error( 'email_failed', 'Failed to send recovery email.' );
	}

	public function create_ab_test( $request ) {
		if ( ! function_exists( 'ai_growth_suite_table_exists' ) || ! ai_growth_suite_table_exists( 'ab_tests' ) ) {
			return new WP_Error( 'table_missing', 'Database table not ready. Please deactivate and reactivate the plugin.', array( 'status' => 500 ) );
		}
		global $wpdb;
		$table_name = $wpdb->prefix . 'ab_tests';

		// Support JSON body (client sends contentType: application/json) or form params
		$params = $request->get_json_params();
		if ( is_array( $params ) && ( isset( $params['product_id'] ) || isset( $params['test_type'] ) ) ) {
			$product_id   = isset( $params['product_id'] ) ? intval( $params['product_id'] ) : 0;
			$test_type    = isset( $params['test_type'] ) ? sanitize_text_field( $params['test_type'] ) : '';
			$variant_a    = isset( $params['variant_a'] ) ? sanitize_textarea_field( $params['variant_a'] ) : '';
			$variant_b    = isset( $params['variant_b'] ) ? sanitize_textarea_field( $params['variant_b'] ) : '';
			$traffic_split = isset( $params['traffic_split'] ) ? intval( $params['traffic_split'] ) : 50;
		} else {
			$product_id   = intval( $request->get_param( 'product_id' ) );
			$test_type    = sanitize_text_field( $request->get_param( 'test_type' ) );
			$variant_a    = sanitize_textarea_field( $request->get_param( 'variant_a' ) );
			$variant_b    = sanitize_textarea_field( $request->get_param( 'variant_b' ) );
			$traffic_split = intval( $request->get_param( 'traffic_split' ) );
		}

		if ( $traffic_split < 10 || $traffic_split > 90 ) {
			$traffic_split = 50;
		}

		if ( ! $product_id || ! $test_type || ! $variant_b ) {
			return new WP_Error( 'missing_params', 'Required parameters missing (product, test type, and Variant B).' );
		}
		
		// If variant_a is empty, get it from the product
		if ( empty( $variant_a ) ) {
			$product = wc_get_product( $product_id );
			if ( $product ) {
				if ( $test_type === 'title' ) {
					$variant_a = $product->get_name();
				} elseif ( $test_type === 'description' ) {
					$variant_a = $product->get_description();
				} elseif ( $test_type === 'price' ) {
					$variant_a = $product->get_price();
				}
			}
		}
		
		// Prepare variant data as JSON
		$variant_a_data = json_encode( array(
			'type' => $test_type,
			'content' => $variant_a
		) );
		
		$variant_b_data = json_encode( array(
			'type' => $test_type,
			'content' => $variant_b
		) );
		
		$inserted = $wpdb->insert(
			$table_name,
			array(
				'product_id' => $product_id,
				'test_type' => $test_type,
				'variant_a' => $variant_a_data,
				'variant_b' => $variant_b_data,
				'traffic_split' => $traffic_split,
				'start_date' => current_time( 'mysql' ),
				'status' => 'running'
			),
			array( '%d', '%s', '%s', '%s', '%d', '%s', '%s' )
		);
		
		if ( $inserted ) {
			return rest_ensure_response( array( 
				'success' => true, 
				'id' => $wpdb->insert_id,
				'message' => 'A/B test created successfully'
			) );
		}
		
		return new WP_Error( 'create_failed', 'Failed to create A/B test.' );
	}

	/**
	 * Stop an A/B test (set status to stopped, end_date to now).
	 *
	 * @param WP_REST_Request $request
	 * @return WP_REST_Response|WP_Error
	 */
	public function stop_ab_test( $request ) {
		if ( ! function_exists( 'ai_growth_suite_table_exists' ) || ! ai_growth_suite_table_exists( 'ab_tests' ) ) {
			return new WP_Error( 'table_missing', 'Database table not ready.', array( 'status' => 500 ) );
		}
		global $wpdb;
		$table_name = $wpdb->prefix . 'ab_tests';

		$params = $request->get_json_params();
		if ( is_array( $params ) && isset( $params['test_id'] ) ) {
			$test_id = intval( $params['test_id'] );
		} else {
			$test_id = intval( $request->get_param( 'test_id' ) );
		}

		if ( ! $test_id ) {
			return new WP_Error( 'missing_params', 'Test ID is required.' );
		}

		$updated = $wpdb->update(
			$table_name,
			array( 'status' => 'stopped', 'end_date' => current_time( 'mysql' ) ),
			array( 'id' => $test_id ),
			array( '%s', '%s' ),
			array( '%d' )
		);

		if ( $updated !== false ) {
			return rest_ensure_response( array( 'success' => true, 'message' => 'Test stopped.' ) );
		}
		return new WP_Error( 'update_failed', 'Failed to stop test.' );
	}

	/**
	 * Get A/B test report data from wp_ai_events (ab_view, ab_conversion).
	 *
	 * @param WP_REST_Request $request
	 * @return WP_REST_Response|WP_Error
	 */
	public function get_ab_test_report( $request ) {
		if ( ! function_exists( 'ai_growth_suite_table_exists' ) ) {
			return new WP_Error( 'table_missing', 'Database not ready.', array( 'status' => 500 ) );
		}
		if ( ! ai_growth_suite_table_exists( 'ab_tests' ) || ! ai_growth_suite_table_exists( 'ai_events' ) ) {
			return new WP_Error( 'table_missing', 'Database tables not ready.', array( 'status' => 500 ) );
		}

		$test_id = (int) $request->get_param( 'test_id' );
		if ( ! $test_id ) {
			return new WP_Error( 'missing_params', 'Test ID is required.', array( 'status' => 400 ) );
		}

		global $wpdb;
		$tests_table = $wpdb->prefix . 'ab_tests';
		$events_table = $wpdb->prefix . 'ai_events';

		$test = $wpdb->get_row( $wpdb->prepare( "SELECT id, product_id, test_type, status FROM $tests_table WHERE id = %d", $test_id ) );
		if ( ! $test ) {
			return new WP_Error( 'not_found', 'Test not found.', array( 'status' => 404 ) );
		}

		$product_name = '';
		if ( function_exists( 'wc_get_product' ) ) {
			$product = wc_get_product( (int) $test->product_id );
			$product_name = $product ? $product->get_name() : '';
		}

		$views_a = 0;
		$views_b = 0;
		$conversions_a = 0;
		$conversions_b = 0;

		$tid = $test_id;
		$like_comma = '%"test_id":' . $tid . ',%';
		$like_brace = '%"test_id":' . $tid . '}%';

		$events = $wpdb->get_results( $wpdb->prepare(
			"SELECT event_type, event_data FROM $events_table WHERE event_type IN ('ab_view','ab_conversion') AND (event_data LIKE %s OR event_data LIKE %s)",
			$like_comma,
			$like_brace
		) );

		if ( is_array( $events ) ) {
			foreach ( $events as $row ) {
				$data = json_decode( $row->event_data, true );
				if ( ! is_array( $data ) || (int) ( isset( $data['test_id'] ) ? $data['test_id'] : 0 ) !== (int) $test_id ) {
					continue;
				}
				$v = isset( $data['variant'] ) && $data['variant'] === 'B' ? 'B' : 'A';
				if ( $row->event_type === 'ab_view' ) {
					if ( $v === 'A' ) {
						$views_a++;
					} else {
						$views_b++;
					}
				} elseif ( $row->event_type === 'ab_conversion' ) {
					if ( $v === 'A' ) {
						$conversions_a++;
					} else {
						$conversions_b++;
					}
				}
			}
		}

		$rate_a = $views_a > 0 ? round( ( $conversions_a / $views_a ) * 100, 2 ) : 0;
		$rate_b = $views_b > 0 ? round( ( $conversions_b / $views_b ) * 100, 2 ) : 0;

		$total_views = $views_a + $views_b;
		if ( $total_views === 0 ) {
			$summary = 'No data yet. Start driving traffic to your product page to see results.';
		} elseif ( $rate_a <= 0 && $rate_b > 0 ) {
			$summary = 'Variant B has conversions while A has none. Consider making Variant B the default.';
		} elseif ( $rate_a > 0 && $rate_b <= 0 ) {
			$summary = 'Variant A is performing better (Variant B has no conversions yet).';
		} elseif ( $rate_b > $rate_a ) {
			$pct = $rate_a > 0 ? round( ( ( $rate_b - $rate_a ) / $rate_a ) * 100, 1 ) : 100;
			$summary = sprintf( 'Variant B is performing %s%% better than Variant A. Consider making Variant B the default.', $pct );
		} elseif ( $rate_a > $rate_b ) {
			$pct = $rate_b > 0 ? round( ( ( $rate_a - $rate_b ) / $rate_b ) * 100, 1 ) : 100;
			$summary = sprintf( 'Variant A is performing %s%% better than Variant B.', $pct );
		} else {
			$summary = 'No conclusive difference between variants. Consider running the test longer.';
		}

		return rest_ensure_response( array(
			'views_a'       => $views_a,
			'views_b'       => $views_b,
			'conversions_a' => $conversions_a,
			'conversions_b' => $conversions_b,
			'rate_a'        => $rate_a,
			'rate_b'        => $rate_b,
			'summary'       => $summary,
			'product_name'  => $product_name,
			'test_type'     => $test->test_type,
			'status'        => $test->status,
		) );
	}
}
