<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class AI_Growth_Suite_Recommendation_Engine {

	public function __construct() {
		add_action( 'woocommerce_after_single_product_summary', array( $this, 'render_recommendations' ), 15 );
		add_action( 'ai_growth_suite_cron_event', array( $this, 'regenerate_recommendations' ) );
		add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_styles' ) );
	}

	public function enqueue_styles() {
		if ( is_product() ) {
			wp_enqueue_style( 'ai-growth-suite-frontend', AI_GROWTH_SUITE_URL . 'assets/css/frontend.css', array(), AI_GROWTH_SUITE_VERSION );
		}
	}

	/**
	 * Render Recommendations on Product Page
	 */
	public function render_recommendations() {
		global $product;
		if ( ! $product ) {
			return;
		}

		$recommendations = $this->get_recommendations( $product->get_id() );

		if ( ! empty( $recommendations ) ) {
			// Load template
			include AI_GROWTH_SUITE_PATH . 'templates/recommendations.php';
		}
	}

	/**
	 * Get Recommendations from DB or Generate
	 *
	 * @param int $product_id
	 * @return array
	 */
	public function get_recommendations( $product_id ) {
		if ( ! function_exists( 'ai_growth_suite_table_exists' ) || ! ai_growth_suite_table_exists( 'ai_recommendations' ) ) {
			return $this->generate_and_save_recommendations( $product_id );
		}
		global $wpdb;
		$table_name = $wpdb->prefix . 'ai_recommendations';

		$row = $wpdb->get_row( $wpdb->prepare( "SELECT * FROM $table_name WHERE product_id = %d", $product_id ) );

		if ( $row ) {
			$dec = json_decode( $row->recommendations, true );
			return is_array( $dec ) ? $dec : array();
		}

		return $this->generate_and_save_recommendations( $product_id );
	}

	/**
	 * Generate and Save Recommendations
	 *
	 * @param int $product_id
	 * @return array
	 */
	public function generate_and_save_recommendations( $product_id ) {
		$ai_engine = new AI_Growth_Suite_AI_Engine();
		$suggestions = $ai_engine->generate_recommendations( $product_id );
		if ( ! is_array( $suggestions ) ) {
			$suggestions = array();
		}

		if ( function_exists( 'ai_growth_suite_table_exists' ) && ai_growth_suite_table_exists( 'ai_recommendations' ) ) {
			global $wpdb;
			$table_name = $wpdb->prefix . 'ai_recommendations';
			$data = array( 'recommendations' => json_encode( $suggestions ), 'generated_at' => current_time( 'mysql' ) );
			$exists = $wpdb->get_var( $wpdb->prepare( "SELECT id FROM $table_name WHERE product_id = %d", $product_id ) );
			if ( $exists ) {
				$wpdb->update( $table_name, $data, array( 'product_id' => $product_id ), array( '%s', '%s' ), array( '%d' ) );
			} else {
				$wpdb->insert( $table_name, array_merge( array( 'product_id' => $product_id ), $data ), array( '%d', '%s', '%s' ) );
			}
		}

		return $suggestions;
	}

	/**
	 * Cron job to regenerate recommendations
	 */
	public function regenerate_recommendations() {
		// Logic to loop through products and regenerate
		// For performance, maybe do 10 at a time
		$products = wc_get_products( array( 'limit' => 10, 'orderby' => 'rand' ) );
		foreach ( $products as $product ) {
			$this->generate_and_save_recommendations( $product->get_id() );
		}
	}
}
