<?php
/**
 * License management for AI Growth Suite (paid plugin).
 * Validates and verifies license via remote API, similar to Assertivlogix Central Pro.
 *
 * @package AI_Growth_Suite
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class AI_Growth_Suite_License_Manager {

	/**
	 * License API URL (filterable for custom endpoints).
	 *
	 * @var string
	 */
	private $api_url = 'https://plugins.assertivlogix.com/api/v1/validate';

	/**
	 * Product ID for this plugin in the license server.
	 *
	 * @var int
	 */
	private $product_id = 11;

	public function __construct() {
		$this->api_url = apply_filters( 'ai_growth_suite_license_api_url', $this->api_url );
		add_action( 'admin_init', array( $this, 'verify_license' ), 5 );
		add_action( 'wp_ajax_ai_growth_suite_activate_license', array( $this, 'ajax_activate_license' ) );
		add_action( 'wp_ajax_ai_growth_suite_deactivate_license', array( $this, 'ajax_deactivate_license' ) );
	}

	/**
	 * Validate license key via remote API.
	 *
	 * @param string $license_key License key to validate.
	 * @return array { success: bool, message: string, plan_type?: string, expires?: int }
	 */
	public function validate_license( $license_key ) {
		if ( empty( trim( (string) $license_key ) ) ) {
			return array( 'success' => false, 'message' => __( 'License key is required.', 'ai-growth-suite' ) );
		}

		$license_key = sanitize_text_field( trim( $license_key ) );

		// Allow a development/demo key for local or testing (no API call).
		$dev_keys = array( 'DEVMODE', 'dev', 'demo', 'localhost' );
		if ( defined( 'AI_GROWTH_SUITE_DEV_LICENSE' ) && AI_GROWTH_SUITE_DEV_LICENSE ) {
			$dev_keys[] = AI_GROWTH_SUITE_DEV_LICENSE;
		}
		if ( in_array( strtoupper( $license_key ), array_map( 'strtoupper', $dev_keys ), true ) ) {
			$hash = hash( 'sha256', $license_key . site_url() );
			update_option( 'ai_growth_suite_license_key_hash', $hash );
			update_option( 'ai_growth_suite_license_status', 'valid' );
			update_option( 'ai_growth_suite_license_plan', 'yearly' );
			update_option( 'ai_growth_suite_license_activated', time() );
			update_option( 'ai_growth_suite_license_expires', 0 );
			update_option( 'ai_growth_suite_license_site_url', site_url() );
			delete_transient( 'ai_growth_suite_license_check' );
			return array( 'success' => true, 'message' => __( 'License activated (development mode).', 'ai-growth-suite' ), 'plan_type' => 'yearly' );
		}

		// Cache: avoid spamming API for same key+site.
		$cache_key = 'ai_growth_suite_license_' . md5( $license_key . site_url() );
		$cached = get_transient( $cache_key );
		if ( false !== $cached && is_array( $cached ) ) {
			return $cached;
		}

		$body = array(
			'license_key' => $license_key,
			'site_url'    => site_url(),
			'product_id'  => $this->product_id,
		);
		
		$response = wp_remote_post(
			$this->api_url,
			array(
				'body'      => $body,
				'timeout'   => 15,
				'sslverify' => true,
			)
		);
		if ( is_wp_error( $response ) ) {
			return array(
				'success' => false,
				'message' => sprintf(
					/* translators: %s: error message */
					__( 'License validation failed: %s', 'ai-growth-suite' ),
					$response->get_error_message()
				),
			);
		}

		$code = wp_remote_retrieve_response_code( $response );
		$body = wp_remote_retrieve_body( $response );
		$data = json_decode( $body, true );

		if ( 200 !== $code || ! is_array( $data ) ) {
			$err = array( 'success' => false, 'message' => __( $data['message'], 'ai-growth-suite' ) );
			set_transient( $cache_key, $err, 5 * MINUTE_IN_SECONDS );
			return $err;
		}

		if ( empty( $data['success'] ) ) {
			$msg = isset( $data['message'] ) ? $data['message'] : __( 'License validation failed.', 'ai-growth-suite' );
			$err = array( 'success' => false, 'message' => $msg );
			set_transient( $cache_key, $err, 5 * MINUTE_IN_SECONDS );
			return $err;
		}

		$plan_type = isset( $data['plan_type'] ) ? sanitize_text_field( $data['plan_type'] ) : 'monthly';
		if ( ! in_array( $plan_type, array( 'monthly', 'yearly' ), true ) ) {
			$plan_type = 'monthly';
		}

		$hash = hash( 'sha256', $license_key . site_url() );
		update_option( 'ai_growth_suite_license_key_hash', $hash );
		update_option( 'ai_growth_suite_license_status', 'valid' );
		update_option( 'ai_growth_suite_license_plan', $plan_type );
		update_option( 'ai_growth_suite_license_activated', time() );
		update_option( 'ai_growth_suite_license_expires', isset( $data['expires'] ) ? intval( $data['expires'] ) : 0 );
		update_option( 'ai_growth_suite_license_site_url', site_url() );
		delete_transient( 'ai_growth_suite_license_check' );

		$ok = array(
			'success'   => true,
			'message'   => sprintf(
				/* translators: %s: plan type */
				__( 'License activated successfully (%s plan).', 'ai-growth-suite' ),
				ucfirst( $plan_type )
			),
			'plan_type' => $plan_type,
		);
		set_transient( $cache_key, $ok, HOUR_IN_SECONDS );
		do_action( 'ai_growth_suite_license_activated', $plan_type );
		return $ok;
	}

	/**
	 * Verify license is still valid (periodic check).
	 *
	 * @return bool
	 */
	public function verify_license() {
		$hash   = get_option( 'ai_growth_suite_license_key_hash' );
		$status = get_option( 'ai_growth_suite_license_status' );

		// Fix inconsistent state: status 'valid' but no stored key hash (e.g. key cleared, bad DB).
		if ( 'valid' === $status && empty( $hash ) ) {
			$this->deactivate_license();
			return false;
		}

		if ( empty( $hash ) || 'valid' !== $status ) {
			return false;
		}

		$cached = get_transient( 'ai_growth_suite_license_check' );
		if ( false !== $cached ) {
			return (bool) $cached;
		}

		$stored_url = get_option( 'ai_growth_suite_license_site_url' );
		if ( $stored_url !== site_url() ) {
			$this->deactivate_license();
			set_transient( 'ai_growth_suite_license_check', false, DAY_IN_SECONDS );
			return false;
		}

		$expires = (int) get_option( 'ai_growth_suite_license_expires', 0 );
		if ( $expires > 0 && time() > $expires ) {
			$this->deactivate_license();
			set_transient( 'ai_growth_suite_license_check', false, DAY_IN_SECONDS );
			return false;
		}

		set_transient( 'ai_growth_suite_license_check', true, DAY_IN_SECONDS );
		return true;
	}

	/**
	 * Deactivate license (clear stored data).
	 */
	public function deactivate_license() {
		delete_option( 'ai_growth_suite_license_key_hash' );
		update_option( 'ai_growth_suite_license_status', 'invalid' );
		delete_option( 'ai_growth_suite_license_plan' );
		delete_option( 'ai_growth_suite_license_expires' );
		delete_option( 'ai_growth_suite_license_key' );
		delete_transient( 'ai_growth_suite_license_check' );
		do_action( 'ai_growth_suite_license_deactivated' );
	}

	/**
	 * Get masked license key for display.
	 *
	 * @return string
	 */
	public function get_license_key_masked() {
		return self::get_masked_key();
	}

	/**
	 * Get masked license key (static helper for views).
	 *
	 * @return string
	 */
	public static function get_masked_key() {
		$hash = get_option( 'ai_growth_suite_license_key_hash' );
		if ( empty( $hash ) ) {
			return '';
		}
		return '****-****-****-' . substr( $hash, -4 );
	}

	/**
	 * AJAX: Activate license.
	 */
	public function ajax_activate_license() {
		check_ajax_referer( 'ai_growth_suite_license_nonce', 'nonce' );
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( __( 'Unauthorized.', 'ai-growth-suite' ) );
		}

		$key = isset( $_POST['license_key'] ) ? sanitize_text_field( wp_unslash( $_POST['license_key'] ) ) : '';
		$result = $this->validate_license( $key );

		if ( ! empty( $result['success'] ) ) {
			wp_send_json_success( $result['message'] );
		}
		wp_send_json_error( isset( $result['message'] ) ? $result['message'] : __( 'Invalid license key.', 'ai-growth-suite' ) );
	}

	/**
	 * AJAX: Deactivate license.
	 */
	public function ajax_deactivate_license() {
		check_ajax_referer( 'ai_growth_suite_license_nonce', 'nonce' );
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( __( 'Unauthorized.', 'ai-growth-suite' ) );
		}
		$this->deactivate_license();
		wp_send_json_success( __( 'License deactivated.', 'ai-growth-suite' ) );
	}

	/**
	 * Check if license is active (PRO features).
	 * Requires both status 'valid' and a stored key hash (real or dev activation).
	 * verify_license() runs on admin_init to revoke when expired or site URL changes.
	 *
	 * @return bool
	 */
	public static function is_pro_active() {
		return get_option( 'ai_growth_suite_license_status' ) === 'valid'
			&& ! empty( get_option( 'ai_growth_suite_license_key_hash', '' ) );
	}
}
