<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class AI_Growth_Suite_Dynamic_Pricing {

	public function __construct() {
		add_filter( 'woocommerce_product_get_price', array( $this, 'apply_dynamic_pricing' ), 10, 2 );
		add_filter( 'woocommerce_product_variation_get_price', array( $this, 'apply_dynamic_pricing' ), 10, 2 );
		
		// Admin Meta Box
		add_action( 'add_meta_boxes', array( $this, 'add_product_meta_box' ) );
		add_action( 'save_post', array( $this, 'save_product_meta_box' ) );
	}

	/**
	 * Apply Dynamic Pricing Logic
	 *
	 * @param float $price
	 * @param WC_Product $product
	 * @return float
	 */
	public function apply_dynamic_pricing( $price, $product ) {
		if ( is_admin() && ! defined( 'DOING_AJAX' ) ) {
			return $price;
		}

		// Master switch: respect Settings > Enable Dynamic Pricing Globally
		if ( ! (int) get_option( 'ai_growth_suite_dynamic_pricing_enabled', 0 ) ) {
			return $price;
		}

		// 1. Check for Product-Specific Rule
		$product_rule_active = get_post_meta( $product->get_id(), '_ai_pricing_active', true );
		if ( $product_rule_active === 'yes' ) {
			$action_type = get_post_meta( $product->get_id(), '_ai_pricing_action', true );
			$amount = floatval( get_post_meta( $product->get_id(), '_ai_pricing_amount', true ) );
			
			if ( $action_type && $amount > 0 ) {
				// Create a temporary rule object
				$rule = (object) array(
					'actions' => json_encode( array( 'type' => $action_type, 'amount' => $amount ) )
				);
				return $this->calculate_new_price( $price, $rule );
			}
		}

		// 2. Fetch active global rules
		$rules = $this->get_active_rules();

		foreach ( $rules as $rule ) {
			// Check conditions (simplified logic)
			if ( $this->check_rule_conditions( $rule, $product ) ) {
				$price = $this->calculate_new_price( $price, $rule );
			}
		}

		return $price;
	}

	/**
	 * Add Meta Box
	 */
	public function add_product_meta_box() {
		add_meta_box(
			'ai_growth_suite_pricing',
			'AI Growth Suite - Dynamic Pricing',
			array( $this, 'render_meta_box' ),
			'product',
			'side',
			'default'
		);
	}

	/**
	 * Render Meta Box
	 */
	public function render_meta_box( $post ) {
		$active = get_post_meta( $post->ID, '_ai_pricing_active', true );
		$action = get_post_meta( $post->ID, '_ai_pricing_action', true );
		$amount = get_post_meta( $post->ID, '_ai_pricing_amount', true );
		
		wp_nonce_field( 'ai_save_pricing_meta', 'ai_pricing_nonce' );
		?>
		<p>
			<label>
				<input type="checkbox" name="_ai_pricing_active" value="yes" <?php checked( $active, 'yes' ); ?>>
				Enable Custom Pricing Rule
			</label>
		</p>
		<p>
			<label>Action:</label><br>
			<select name="_ai_pricing_action" style="width: 100%;">
				<option value="percentage" <?php selected( $action, 'percentage' ); ?>>Decrease by %</option>
				<option value="increase_percentage" <?php selected( $action, 'increase_percentage' ); ?>>Increase by %</option>
				<option value="fixed" <?php selected( $action, 'fixed' ); ?>>Set Fixed Price</option>
			</select>
		</p>
		<p>
			<label>Amount:</label><br>
			<input type="number" name="_ai_pricing_amount" value="<?php echo esc_attr( $amount ); ?>" style="width: 100%;" step="0.01">
		</p>
		<?php
	}

	/**
	 * Save Meta Box
	 */
	public function save_product_meta_box( $post_id ) {
		if ( ! isset( $_POST['ai_pricing_nonce'] ) || ! wp_verify_nonce( $_POST['ai_pricing_nonce'], 'ai_save_pricing_meta' ) ) {
			return;
		}
		if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) {
			return;
		}

		$active = isset( $_POST['_ai_pricing_active'] ) ? 'yes' : 'no';
		update_post_meta( $post_id, '_ai_pricing_active', $active );
		
		if ( isset( $_POST['_ai_pricing_action'] ) ) {
			update_post_meta( $post_id, '_ai_pricing_action', sanitize_text_field( $_POST['_ai_pricing_action'] ) );
		}
		if ( isset( $_POST['_ai_pricing_amount'] ) ) {
			update_post_meta( $post_id, '_ai_pricing_amount', floatval( $_POST['_ai_pricing_amount'] ) );
		}
	}

	/**
	 * Get Active Rules from DB
	 *
	 * @return array
	 */
	private function get_active_rules() {
		if ( ! function_exists( 'ai_growth_suite_table_exists' ) || ! ai_growth_suite_table_exists( 'dynamic_pricing_rules' ) ) {
			return array();
		}
		global $wpdb;
		$table_name = $wpdb->prefix . 'dynamic_pricing_rules';
		$rules = $wpdb->get_results( "SELECT * FROM $table_name WHERE is_active = 1" );
		return is_array( $rules ) ? $rules : array();
	}

	/**
	 * Check if rule applies to product
	 *
	 * @param object $rule
	 * @param WC_Product $product
	 * @return bool
	 */
	private function check_rule_conditions( $rule, $product ) {
		// Parse conditions JSON
		$conditions = json_decode( $rule->conditions, true );
		$type = isset( $conditions['type'] ) ? $conditions['type'] : '';
		
		switch ( $type ) {
			case 'low_inventory':
				// Check if stock is low (< 5)
				$stock = $product->get_stock_quantity();
				return ( ! is_null( $stock ) && $stock < 5 );

			case 'time_based':
				// Check if it's the weekend (Sat/Sun)
				$day = date( 'N', current_time( 'timestamp' ) );
				return ( $day >= 6 );

			case 'cart_value':
				// Check if cart total > 100
				if ( ! WC()->cart ) return false;
				return ( WC()->cart->subtotal > 100 );

			case 'high_velocity':
				// Mock: Always true for demo purposes, or check sales count
				return ( $product->get_total_sales() > 50 );

			default:
				return true;
		}
	}

	/**
	 * Calculate new price
	 *
	 * @param float $price
	 * @param object $rule
	 * @return float
	 */
	private function calculate_new_price( $price, $rule ) {
		$actions = json_decode( $rule->actions, true );
		
		if ( isset( $actions['type'] ) && $actions['type'] === 'percentage' ) {
			$price = $price * ( 1 - ( $actions['amount'] / 100 ) );
		} elseif ( isset( $actions['type'] ) && $actions['type'] === 'increase_percentage' ) {
			$price = $price * ( 1 + ( $actions['amount'] / 100 ) );
		} elseif ( isset( $actions['type'] ) && $actions['type'] === 'fixed' ) {
			// "Set Fixed Price" = override with amount
			$price = isset( $actions['amount'] ) ? floatval( $actions['amount'] ) : $price;
		}

		return max( 0, $price );
	}
}
