<?php
/**
 * Abandoned Cart Tracker
 * Tracks when customers add items to cart but don't complete checkout
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class AI_Growth_Suite_Cart_Tracker {

	private $table_name;

	public function __construct() {
		global $wpdb;
		$this->table_name = $wpdb->prefix . 'abandoned_carts';
		
		// Hook into WooCommerce cart actions
		add_action( 'woocommerce_add_to_cart', array( $this, 'track_cart_update' ), 10, 6 );
		add_action( 'woocommerce_cart_item_removed', array( $this, 'track_cart_update' ), 10, 2 );
		add_action( 'woocommerce_after_cart_item_quantity_update', array( $this, 'track_cart_update' ), 10, 4 );
		
		// Track when order is completed (mark cart as recovered)
		add_action( 'woocommerce_thankyou', array( $this, 'mark_cart_recovered' ), 10, 1 );
		
		// Schedule cron to check for abandoned carts
		if ( ! wp_next_scheduled( 'ai_growth_suite_check_abandoned_carts' ) ) {
			wp_schedule_event( time(), 'hourly', 'ai_growth_suite_check_abandoned_carts' );
		}
		add_action( 'ai_growth_suite_check_abandoned_carts', array( $this, 'check_abandoned_carts' ) );
	}

	/**
	 * Track cart updates
	 */
	public function track_cart_update() {
		if ( ! WC()->cart || WC()->cart->is_empty() ) {
			return;
		}

		// Get customer email
		$user_email = $this->get_customer_email();
		if ( ! $user_email ) {
			return; // Can't track without email
		}

		// Get cart contents
		$cart_contents = array();
		$calculated_total = 0;
		
		foreach ( WC()->cart->get_cart() as $cart_item ) {
			$product = $cart_item['data'];
			$line_total = isset( $cart_item['line_total'] ) ? floatval( $cart_item['line_total'] ) : 0;
			
			$cart_contents[] = array(
				'product_id' => $cart_item['product_id'],
				'name' => $product->get_name(),
				'quantity' => $cart_item['quantity'],
				'price' => $product->get_price(),
				'line_subtotal' => $cart_item['line_subtotal'],
				'line_total' => $line_total
			);
			
			// Add to calculated total
			$calculated_total += $line_total;
		}

		// Get cart total from WooCommerce (including all fees, taxes, shipping)
		$wc_total = WC()->cart->get_cart_contents_total() + 
		            WC()->cart->get_fee_total() + 
		            WC()->cart->get_shipping_total() + 
		            WC()->cart->get_cart_contents_tax() + 
		            WC()->cart->get_fee_tax() + 
		            WC()->cart->get_shipping_tax();
		
		// Use WooCommerce total if available, otherwise use calculated total
		$cart_total = $wc_total > 0 ? $wc_total : $calculated_total;

		// Save or update cart
		$this->save_cart( $user_email, $cart_contents, $cart_total );
	}

	/**
	 * Get customer email
	 */
	private function get_customer_email() {
		// Try to get from current user
		if ( is_user_logged_in() ) {
			$user = wp_get_current_user();
			return $user->user_email;
		}

		// Try to get from session/checkout
		if ( WC()->session ) {
			$customer = WC()->session->get( 'customer' );
			if ( isset( $customer['email'] ) && ! empty( $customer['email'] ) ) {
				return $customer['email'];
			}
		}

		// Try to get from billing email
		if ( WC()->customer && WC()->customer->get_billing_email() ) {
			return WC()->customer->get_billing_email();
		}

		return null;
	}

	/**
	 * Save cart to database
	 */
	private function save_cart( $email, $contents, $total ) {
		global $wpdb;

		// Check if there's an existing abandoned cart for this email
		$existing = $wpdb->get_row( $wpdb->prepare(
			"SELECT * FROM {$this->table_name} WHERE user_email = %s AND status = 'abandoned' ORDER BY created_at DESC LIMIT 1",
			$email
		) );

		if ( $existing ) {
			// Parse existing cart items
			$existing_items = json_decode( $existing->cart_contents, true );
			$existing_product_ids = array();
			$existing_items_by_id = array();
			
			if ( isset( $existing_items['items'] ) && is_array( $existing_items['items'] ) ) {
				foreach ( $existing_items['items'] as $item ) {
					$existing_product_ids[] = $item['product_id'];
					$existing_items_by_id[$item['product_id']] = $item;
				}
			}
			
			// Get current product IDs
			$current_product_ids = array();
			foreach ( $contents as $item ) {
				$current_product_ids[] = $item['product_id'];
			}
			
			// Find removed products
			$removed_product_ids = array_diff( $existing_product_ids, $current_product_ids );
			
			// Check if any products are the same
			$has_common_products = count( array_intersect( $existing_product_ids, $current_product_ids ) ) > 0;
			
			if ( $has_common_products || !empty($removed_product_ids) ) {
				// Same cart session - update with removed items marked
				$updated_contents = $contents;
				
				// Add removed items with 'removed' flag
				foreach ( $removed_product_ids as $removed_id ) {
					$removed_item = $existing_items_by_id[$removed_id];
					$removed_item['removed'] = true;
					$updated_contents[] = $removed_item;
				}
				
				// Update existing cart (keep original total if items were removed)
				$wpdb->update(
					$this->table_name,
					array(
						'cart_contents' => json_encode( array( 'items' => $updated_contents ) ),
						'cart_total' => $total,
						'created_at' => current_time( 'mysql' )
					),
					array( 'id' => $existing->id ),
					array( '%s', '%f', '%s' ),
					array( '%d' )
				);
			} else {
				// Completely different products - create new record
				$wpdb->insert(
					$this->table_name,
					array(
						'user_email' => $email,
						'cart_contents' => json_encode( array( 'items' => $contents ) ),
						'cart_total' => $total,
						'status' => 'abandoned',
						'created_at' => current_time( 'mysql' )
					),
					array( '%s', '%s', '%f', '%s', '%s' )
				);
			}
		} else {
			// No existing cart - create new record
			$wpdb->insert(
				$this->table_name,
				array(
					'user_email' => $email,
					'cart_contents' => json_encode( array( 'items' => $contents ) ),
					'cart_total' => $total,
					'status' => 'abandoned',
					'created_at' => current_time( 'mysql' )
				),
				array( '%s', '%s', '%f', '%s', '%s' )
			);
		}
	}

	/**
	 * Mark cart as recovered when order is completed
	 */
	public function mark_cart_recovered( $order_id ) {
		if ( ! $order_id ) {
			return;
		}

		$order = wc_get_order( $order_id );
		if ( ! $order ) {
			return;
		}

		$email = $order->get_billing_email();
		if ( ! $email ) {
			return;
		}

		global $wpdb;
		$wpdb->update(
			$this->table_name,
			array( 'status' => 'recovered' ),
			array( 'user_email' => $email, 'status' => 'abandoned' ),
			array( '%s' ),
			array( '%s', '%s' )
		);
	}

	/**
	 * Check for abandoned carts (runs hourly)
	 */
	public function check_abandoned_carts() {
		global $wpdb;

		// Find carts abandoned for more than 1 hour that haven't been emailed
		$abandoned_carts = $wpdb->get_results(
			"SELECT * FROM {$this->table_name} 
			WHERE status = 'abandoned' 
			AND created_at < DATE_SUB(NOW(), INTERVAL 1 HOUR)
			LIMIT 10"
		);

		foreach ( $abandoned_carts as $cart ) {
			// Send recovery email
			$this->send_recovery_email( $cart );
			
			// Update status to emailed
			$wpdb->update(
				$this->table_name,
				array( 'status' => 'emailed' ),
				array( 'id' => $cart->id ),
				array( '%s' ),
				array( '%d' )
			);
		}
	}

	/**
	 * Send recovery email
	 */
	private function send_recovery_email( $cart ) {
		$template = get_option( 'ai_growth_suite_template_abandoned_cart', array() );
		$subject = isset( $template['subject'] ) ? $template['subject'] : 'Complete your purchase at {store_name}';
		$body = isset( $template['body'] ) ? $template['body'] : 'Hi, You left items in your cart. Complete your purchase now!';

		// Replace tokens
		$store_name = get_bloginfo( 'name' );
		$subject = str_replace( '{store_name}', $store_name, $subject );
		$body = str_replace( '{store_name}', $store_name, $body );
		$body = str_replace( '{email}', $cart->user_email, $body );
		$body = str_replace( '{cart_total}', '$' . number_format( $cart->cart_total, 2 ), $body );

		// Send email
		$headers = array( 'Content-Type: text/html; charset=UTF-8' );
		wp_mail( $cart->user_email, $subject, $body, $headers );

		// Log email
		global $wpdb;
		$log_table = $wpdb->prefix . 'ai_email_log';
		$wpdb->insert(
			$log_table,
			array(
				'email_to' => $cart->user_email,
				'email_subject' => $subject,
				'email_type' => 'abandoned_cart',
				'status' => 'sent',
				'sent_at' => current_time( 'mysql' )
			),
			array( '%s', '%s', '%s', '%s', '%s' )
		);
	}
}

// Initialize the tracker
new AI_Growth_Suite_Cart_Tracker();
