<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class AI_Growth_Suite_Admin_Menu {

	public function __construct() {
		add_action( 'admin_menu', array( $this, 'add_admin_menu' ) );
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_assets' ) );
		add_action( 'admin_init', array( $this, 'register_settings' ) );
		add_action( 'admin_notices', array( $this, 'maybe_license_notice' ) );
	}

	/**
	 * Show activation notice on plugin admin pages when license is not active.
	 */
	public function maybe_license_notice() {
		$screen = get_current_screen();
		if ( ! $screen || strpos( $screen->id, 'ai-growth-suite' ) === false ) {
			return;
		}
		if ( AI_Growth_Suite_License_Manager::is_pro_active() ) {
			return;
		}
		$url = admin_url( 'admin.php?page=ai-growth-suite-license' );
		echo '<div class="notice notice-warning"><p><strong>AI eCommerce Growth Suite</strong> – <a href="' . esc_url( $url ) . '">Activate your license</a> to unlock PRO features (A/B Testing, Analytics) and get updates.</p></div>';
	}

	public function register_settings() {
		register_setting( 'ai_growth_suite_options', 'ai_growth_suite_api_key', array(
			'type'              => 'string',
			'sanitize_callback' => 'sanitize_text_field',
		) );
		register_setting( 'ai_growth_suite_options', 'ai_growth_suite_dynamic_pricing_enabled', array(
			'type'              => 'integer',
			'sanitize_callback' => function( $value ) {
				return absint( $value ) ? 1 : 0;
			},
		) );

	}

	/**
	 * Add our option group to the whitelist so options.php can save them.
	 *
	 * @param array $whitelist
	 * @return array
	 */
	public function whitelist_options( $whitelist ) {
		$whitelist['ai_growth_suite_options'] = array(
			'ai_growth_suite_api_key',
			'ai_growth_suite_dynamic_pricing_enabled',
		);
		return $whitelist;
	}

	/**
	 * Pages allowed when license is NOT active (free). All others are PRO-only.
	 *
	 * @return array Slugs (e.g. 'product-ai', 'license').
	 */
	public static function get_free_pages() {
		return array( 'product-ai', 'upsell', 'settings', 'upgrade', 'license' );
	}

	/**
	 * Check if a page slug is allowed when license is inactive.
	 *
	 * @param string $slug Page slug without 'ai-growth-suite-' prefix.
	 * @return bool
	 */
	public static function is_free_page( $slug ) {
		return in_array( $slug, self::get_free_pages(), true );
	}

	public function add_admin_menu() {
		add_menu_page(
			'AI Growth Suite',
			'AI Growth Suite',
			'manage_options',
			'ai-growth-suite',
			array( $this, 'render_dashboard' ),
			'dashicons-chart-area',
			56
		);

		$tabs = array(
			'dashboard'       => __( 'Dashboard', 'ai-growth-suite' ),
			'license'         => __( 'License', 'ai-growth-suite' ),
			'product-ai'      => __( 'Product AI Tools', 'ai-growth-suite' ),
			'upsell'          => __( 'Upsell & Cross-Sell', 'ai-growth-suite' ),
			'dynamic-pricing' => __( 'Dynamic Pricing', 'ai-growth-suite' ),
			'marketing'       => __( 'Marketing Automation', 'ai-growth-suite' ),
			'abandoned-carts' => __( 'Abandoned Carts', 'ai-growth-suite' ),
			'ab-testing'      => __( 'A/B Testing', 'ai-growth-suite' ),
			'analytics'       => __( 'Analytics', 'ai-growth-suite' ),
			'settings'        => __( 'Settings', 'ai-growth-suite' ),
			'upgrade'         => __( 'Upgrade to PRO', 'ai-growth-suite' ),
		);

		foreach ( $tabs as $slug => $title ) {
			add_submenu_page(
				'ai-growth-suite',
				$title,
				$title,
				'manage_options',
				'ai-growth-suite-' . $slug,
				array( $this, 'render_page' )
			);
		}
	}

	public function enqueue_assets() {
		// Enqueue CSS/JS
		wp_enqueue_style( 'ai-growth-suite-admin', AI_GROWTH_SUITE_URL . 'admin/css/admin.css', array(), AI_GROWTH_SUITE_VERSION );
		wp_enqueue_script( 'ai-growth-suite-admin', AI_GROWTH_SUITE_URL . 'admin/js/admin.js', array( 'jquery' ), AI_GROWTH_SUITE_VERSION, true );
		
		wp_localize_script( 'ai-growth-suite-admin', 'aiGrowthSuite', array(
			'root' => esc_url_raw( rest_url() ),
			'nonce' => wp_create_nonce( 'wp_rest' ),
			'ajaxurl' => admin_url( 'admin-ajax.php' ),
			'currency' => function_exists( 'get_woocommerce_currency_symbol' ) ? get_woocommerce_currency_symbol() : '$'
		) );
	}

	public function render_dashboard() {
		$this->render_page( 'dashboard' );
	}

	public function render_page( $page = '' ) {
		if ( empty( $page ) ) {
			$page = isset( $_GET['page'] ) ? str_replace( 'ai-growth-suite-', '', $_GET['page'] ) : 'dashboard';
		}

		// Handle main menu slug case
		if ( $page === 'ai-growth-suite' ) {
			$page = 'dashboard';
		}

		$is_pro = class_exists( 'AI_Growth_Suite_License_Manager' ) && AI_Growth_Suite_License_Manager::is_pro_active();
		$is_pro_page_locked = ! $is_pro && ! self::is_free_page( $page );

		echo '<div class="wrap">';
		if ( $is_pro_page_locked ) {
			echo '<div class="ags-pro-gate">';
			echo '<div class="ags-pro-gate-content">';
		}
		echo '<h1>AI Growth Suite - ' . esc_html( ucfirst( str_replace( '-', ' ', $page ) ) ) . '</h1>';
		$file = AI_GROWTH_SUITE_PATH . 'admin/views/' . $page . '.php';
		if ( file_exists( $file ) ) {
			include $file;
		} else {
			echo '<p>Page content coming soon.</p>';
		}
		if ( $is_pro_page_locked ) {
			echo '</div>';
			echo '<div class="ags-pro-gate-overlay">';
			echo '<div class="ags-pro-gate-message">';
			echo '<span class="ags-pro-gate-icon dashicons dashicons-lock"></span>';
			echo '<p class="ags-pro-gate-title">' . esc_html__( 'PRO Feature', 'ai-growth-suite' ) . '</p>';
			echo '<p class="ags-pro-gate-desc">' . esc_html__( 'Activate your license to unlock this page and make changes.', 'ai-growth-suite' ) . '</p>';
			echo '<p class="ags-pro-gate-actions">';
			echo '<a href="' . esc_url( admin_url( 'admin.php?page=ai-growth-suite-license' ) ) . '" class="button button-primary">' . esc_html__( 'Activate License', 'ai-growth-suite' ) . '</a> ';
			echo '<a href="' . esc_url( admin_url( 'admin.php?page=ai-growth-suite-upgrade' ) ) . '" class="button">' . esc_html__( 'View Plans', 'ai-growth-suite' ) . '</a>';
			echo '</p></div></div>';
			echo '</div>';
		}
		echo '</div>';
	}
}
