<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class AI_Growth_Suite_Abandoned_Cart {

	public function __construct() {
		// Capture email on checkout field update
		add_action( 'woocommerce_checkout_update_order_review', array( $this, 'capture_abandoned_cart' ) );
		// Mark recovered on order complete
		add_action( 'woocommerce_new_order', array( $this, 'mark_recovered' ) );
		// Cron for sending emails
		add_action( 'ai_growth_suite_abandoned_cart_cron', array( $this, 'send_recovery_emails' ) );
	}

	/**
	 * Capture Cart Data
	 *
	 * @param string $post_data
	 */
	public function capture_abandoned_cart( $post_data ) {
		parse_str( $post_data, $checkout_data );

		if ( ! empty( $checkout_data['billing_email'] ) ) {
			$email = sanitize_email( $checkout_data['billing_email'] );
			$cart = WC()->cart->get_cart();
			$total = WC()->cart->total;

			if ( ! empty( $cart ) ) {
				$this->save_cart( $email, $cart, $total );
			}
		}
	}

	/**
	 * Save Cart to DB
	 *
	 * @param string $email
	 * @param array $cart
	 * @param float $total
	 */
	private function save_cart( $email, $cart, $total ) {
		global $wpdb;
		if ( ! function_exists( 'ai_growth_suite_table_exists' ) || ! ai_growth_suite_table_exists( 'abandoned_carts' ) ) {
			return;
		}
		$table_name = $wpdb->prefix . 'abandoned_carts';

		// Serialize cart items for storage
		$cart_items = array();
		foreach ( $cart as $key => $item ) {
			$cart_items[] = array(
				'product_id' => $item['product_id'],
				'quantity' => $item['quantity'],
				'variation_id' => $item['variation_id']
			);
		}

		// Check if exists
		$exists = $wpdb->get_var( $wpdb->prepare( "SELECT id FROM $table_name WHERE user_email = %s AND status = 'abandoned'", $email ) );

		if ( $exists ) {
			$wpdb->update( 
				$table_name, 
				array( 
					'cart_contents' => json_encode( $cart_items ), 
					'cart_total' => $total,
					'created_at' => current_time( 'mysql' ) // Update timestamp to reset timer
				), 
				array( 'id' => $exists ) 
			);
		} else {
			$wpdb->insert( 
				$table_name, 
				array( 
					'user_email' => $email, 
					'cart_contents' => json_encode( $cart_items ), 
					'cart_total' => $total,
					'created_at' => current_time( 'mysql' ),
					'status' => 'abandoned'
				) 
			);
		}
	}

	/**
	 * Mark Cart as Recovered
	 *
	 * @param int $order_id
	 */
	public function mark_recovered( $order_id ) {
		$order = wc_get_order( $order_id );
		if ( ! $order ) {
			return;
		}
		$email = $order->get_billing_email();
		if ( ! function_exists( 'ai_growth_suite_table_exists' ) || ! ai_growth_suite_table_exists( 'abandoned_carts' ) ) {
			return;
		}
		global $wpdb;
		$table_name = $wpdb->prefix . 'abandoned_carts';
		
		$wpdb->update( 
			$table_name, 
			array( 
				'status' => 'recovered', 
				'recovered_at' => current_time( 'mysql' ) 
			), 
			array( 'user_email' => $email, 'status' => 'abandoned' ) 
		);
	}

	/**
	 * Send Recovery Emails (Cron)
	 */
	public function send_recovery_emails() {
		if ( ! function_exists( 'ai_growth_suite_table_exists' ) || ! ai_growth_suite_table_exists( 'abandoned_carts' ) ) {
			return;
		}
		global $wpdb;
		$table_name = $wpdb->prefix . 'abandoned_carts';
		
		// Find carts older than 1 hour but less than 24 hours (example logic)
		// In real app, check against specific intervals for 3-step sequence
		$time_threshold = date( 'Y-m-d H:i:s', strtotime( '-1 hour' ) );
		
		$carts = $wpdb->get_results( $wpdb->prepare( "SELECT * FROM $table_name WHERE status = 'abandoned' AND created_at < %s", $time_threshold ) );

		foreach ( $carts as $cart ) {
			// Send email logic here
			// Use Email Engine
			// Update status or log sent email to avoid duplicates
		}
	}
}
