<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class AI_Growth_Suite_AB_Testing {

	public function __construct() {
		add_action( 'template_redirect', array( $this, 'assign_variant' ) );
		add_filter( 'the_title', array( $this, 'filter_title' ), 10, 2 );
		add_filter( 'the_content', array( $this, 'filter_description' ) );
		add_action( 'woocommerce_add_to_cart', array( $this, 'log_ab_conversion' ), 10, 6 );
	}

	/**
	 * Assign User to Variant A or B and log view once per request.
	 */
	public function assign_variant() {
		if ( ! is_product() ) {
			return;
		}

		$product_id = get_queried_object_id();
		if ( ! $product_id ) {
			return;
		}

		if ( ! isset( $_COOKIE['ai_ab_variant'] ) ) {
			$variant = ( rand( 0, 1 ) === 0 ) ? 'A' : 'B';
			setcookie( 'ai_ab_variant', $variant, time() + 3600 * 24 * 30, COOKIEPATH, COOKIE_DOMAIN );
			$_COOKIE['ai_ab_variant'] = $variant;
		}

		$test = $this->get_running_test( $product_id );
		if ( $test ) {
			$this->log_ab_view( (int) $test->id, $product_id, isset( $_COOKIE['ai_ab_variant'] ) ? $_COOKIE['ai_ab_variant'] : 'A' );
		}
	}

	/**
	 * Log one A/B view per request per test into ai_events.
	 */
	private function log_ab_view( $test_id, $product_id, $variant ) {
		if ( ! function_exists( 'ai_growth_suite_table_exists' ) || ! ai_growth_suite_table_exists( 'ai_events' ) ) {
			return;
		}
		static $logged = array();
		$key = $test_id . '-' . $product_id;
		if ( isset( $logged[ $key ] ) ) {
			return;
		}
		$logged[ $key ] = true;
		global $wpdb;
		$table = $wpdb->prefix . 'ai_events';
		$wpdb->insert(
			$table,
			array(
				'event_type'  => 'ab_view',
				'event_data'  => wp_json_encode( array( 'test_id' => $test_id, 'product_id' => $product_id, 'variant' => $variant ) ),
				'created_at'  => current_time( 'mysql' ),
				'user_id'     => get_current_user_id(),
			),
			array( '%s', '%s', '%s', '%d' )
		);
	}

	/**
	 * Log A/B conversion on add to cart when the product has a running test.
	 */
	public function log_ab_conversion( $cart_item_key, $product_id, $quantity, $variation_id, $variation, $cart_item_data ) {
		if ( ! function_exists( 'ai_growth_suite_table_exists' ) || ! ai_growth_suite_table_exists( 'ai_events' ) ) {
			return;
		}
		$test = $this->get_running_test( $product_id );
		if ( ! $test ) {
			return;
		}
		$variant = isset( $_COOKIE['ai_ab_variant'] ) ? sanitize_text_field( $_COOKIE['ai_ab_variant'] ) : 'A';
		if ( ! in_array( $variant, array( 'A', 'B' ), true ) ) {
			$variant = 'A';
		}
		global $wpdb;
		$table = $wpdb->prefix . 'ai_events';
		$wpdb->insert(
			$table,
			array(
				'event_type'  => 'ab_conversion',
				'event_data'  => wp_json_encode( array( 'test_id' => (int) $test->id, 'product_id' => (int) $product_id, 'variant' => $variant ) ),
				'created_at'  => current_time( 'mysql' ),
				'user_id'     => get_current_user_id(),
			),
			array( '%s', '%s', '%s', '%d' )
		);
	}

	/**
	 * Filter Product Title
	 */
	public function filter_title( $title, $id = null ) {
		if ( ! is_product() || ! in_the_loop() ) {
			return $title;
		}

		$test = $this->get_running_test( get_the_ID() );
		if ( ! $test ) {
			return $title;
		}

		$variant = isset( $_COOKIE['ai_ab_variant'] ) ? $_COOKIE['ai_ab_variant'] : 'A';
		
		if ( $variant === 'B' && $test->test_type === 'title' ) {
			$data = json_decode( $test->variant_b, true );
			if ( ! empty( $data['content'] ) ) {
				return $data['content'];
			}
		}

		return $title;
	}

	/**
	 * Filter Product Description
	 */
	public function filter_description( $content ) {
		if ( ! is_product() || ! in_the_loop() ) {
			return $content;
		}

		$test = $this->get_running_test( get_the_ID() );
		if ( ! $test ) {
			return $content;
		}

		$variant = isset( $_COOKIE['ai_ab_variant'] ) ? $_COOKIE['ai_ab_variant'] : 'A';
		
		if ( $variant === 'B' && $test->test_type === 'description' ) {
			$data = json_decode( $test->variant_b, true );
			if ( ! empty( $data['content'] ) ) {
				return $data['content'];
			}
		}

		return $content;
	}

	/**
	 * Get Running Test for Product
	 */
	private function get_running_test( $product_id ) {
		if ( ! function_exists( 'ai_growth_suite_table_exists' ) || ! ai_growth_suite_table_exists( 'ab_tests' ) ) {
			return null;
		}
		global $wpdb;
		$table_name = $wpdb->prefix . 'ab_tests';
		return $wpdb->get_row( $wpdb->prepare( "SELECT * FROM $table_name WHERE product_id = %d AND status = 'running'", $product_id ) );
	}
}
