<?php
/**
 * Plugin Name: AI eCommerce Growth Suite
 * Plugin URI: https://example.com/ai-growth-suite
 * Description: All-in-one AI-powered eCommerce growth suite for WooCommerce. Upsells, Dynamic Pricing, Abandoned Cart Recovery, and more.
 * Version: 1.0.0
 * Author: Your Name
 * Author URI: https://example.com
 * Text Domain: ai-growth-suite
 * Domain Path: /languages
 * Requires at least: 5.8
 * Requires PHP: 7.4
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

// Define Constants
define( 'AI_GROWTH_SUITE_VERSION', '1.0.0' );
define( 'AI_GROWTH_SUITE_PATH', plugin_dir_path( __FILE__ ) );
define( 'AI_GROWTH_SUITE_URL', plugin_dir_url( __FILE__ ) );
define( 'AI_GROWTH_SUITE_FILE', __FILE__ );

// Include Core Classes
require_once AI_GROWTH_SUITE_PATH . 'includes/class-ai-engine.php';
require_once AI_GROWTH_SUITE_PATH . 'includes/class-recommendation-engine.php';
require_once AI_GROWTH_SUITE_PATH . 'includes/class-dynamic-pricing.php';
require_once AI_GROWTH_SUITE_PATH . 'includes/class-email-engine.php';
require_once AI_GROWTH_SUITE_PATH . 'includes/class-abandoned-cart.php';
require_once AI_GROWTH_SUITE_PATH . 'includes/class-ab-testing.php';
require_once AI_GROWTH_SUITE_PATH . 'includes/class-analytics.php';
require_once AI_GROWTH_SUITE_PATH . 'includes/class-admin-menu.php';
require_once AI_GROWTH_SUITE_PATH . 'includes/class-rest-api.php';
require_once AI_GROWTH_SUITE_PATH . 'includes/class-license-manager.php';
require_once AI_GROWTH_SUITE_PATH . 'includes/class-cart-tracker.php';

/**
 * Main Plugin Class
 */
class AI_Growth_Suite {

	/**
	 * Instance of this class.
	 *
	 * @var object
	 */
	protected static $instance = null;

	/**
	 * Return an instance of this class.
	 *
	 * @return object A single instance of this class.
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Constructor
	 */
	private function __construct() {
		// Initialize classes
		$this->init_hooks();
		$this->init_classes();
	}

	/**
	 * Initialize Hooks
	 */
	private function init_hooks() {
		register_activation_hook( AI_GROWTH_SUITE_FILE, array( $this, 'activate' ) );
		register_deactivation_hook( AI_GROWTH_SUITE_FILE, array( $this, 'deactivate' ) );
		add_action( 'plugins_loaded', array( $this, 'load_textdomain' ) );
		add_action( 'plugins_loaded', array( $this, 'maybe_create_tables' ), 20 );
	}

	/**
	 * Initialize Core Classes
	 */
	private function init_classes() {
		// Initialize Admin Menu
		if ( is_admin() ) {
			new AI_Growth_Suite_Admin_Menu();
		}

		// Initialize REST API
		new AI_Growth_Suite_REST_API();

		// Initialize Engines
		new AI_Growth_Suite_AI_Engine();
		new AI_Growth_Suite_Recommendation_Engine();
		new AI_Growth_Suite_Dynamic_Pricing();
		new AI_Growth_Suite_Email_Engine();
		new AI_Growth_Suite_Abandoned_Cart();
		if ( class_exists( 'AI_Growth_Suite_License_Manager' ) && AI_Growth_Suite_License_Manager::is_pro_active() ) {
			new AI_Growth_Suite_AB_Testing();
			new AI_Growth_Suite_Analytics();
		}
		new AI_Growth_Suite_License_Manager();
	}

	/**
	 * Load Text Domain
	 */
	public function load_textdomain() {
		load_plugin_textdomain( 'ai-growth-suite', false, dirname( plugin_basename( __FILE__ ) ) . '/languages' );
	}

	/**
	 * Activation Hook
	 */
	public function activate() {
		// Create Database Tables
		$this->create_tables();

		// Default license status (paid plugin: requires activation)
		if ( false === get_option( 'ai_growth_suite_license_status' ) ) {
			add_option( 'ai_growth_suite_license_status', 'invalid' );
		}

		// Schedule Cron Jobs
		if ( ! wp_next_scheduled( 'ai_growth_suite_cron_event' ) ) {
			wp_schedule_event( time(), 'hourly', 'ai_growth_suite_cron_event' );
		}
		if ( ! wp_next_scheduled( 'ai_growth_suite_abandoned_cart_cron' ) ) {
			wp_schedule_event( time(), 'fifteen_minutes', 'ai_growth_suite_abandoned_cart_cron' );
		}
	}

	/**
	 * Ensure database tables exist (runs on plugins_loaded if tables are missing).
	 */
	public function maybe_create_tables() {
		global $wpdb;
		$prefix = $wpdb->prefix;
		$required = array( $prefix . 'abandoned_carts', $prefix . 'ab_tests', $prefix . 'ai_events' );
		foreach ( $required as $table ) {
			$found = $wpdb->get_var( $wpdb->prepare( 'SHOW TABLES LIKE %s', $table ) );
			if ( empty( $found ) ) {
				$this->create_tables();
				return;
			}
		}
	}

	/**
	 * Deactivation Hook
	 */
	public function deactivate() {
		wp_clear_scheduled_hook( 'ai_growth_suite_cron_event' );
		wp_clear_scheduled_hook( 'ai_growth_suite_abandoned_cart_cron' );
	}

	/**
	 * Create Database Tables
	 */
	private function create_tables() {
		global $wpdb;
		$charset_collate = $wpdb->get_charset_collate();

		require_once( ABSPATH . 'wp-admin/includes/upgrade.php' );

		// Table: wp_ai_recommendations
		$table_name = $wpdb->prefix . 'ai_recommendations';
		$sql = "CREATE TABLE $table_name (
			id mediumint(9) NOT NULL AUTO_INCREMENT,
			product_id bigint(20) NOT NULL,
			recommendations longtext NOT NULL,
			generated_at datetime DEFAULT '0000-00-00 00:00:00' NOT NULL,
			PRIMARY KEY  (id)
		) $charset_collate;";
		dbDelta( $sql );

		// Table: wp_dynamic_pricing_rules
		$table_name = $wpdb->prefix . 'dynamic_pricing_rules';
		$sql = "CREATE TABLE $table_name (
			id mediumint(9) NOT NULL AUTO_INCREMENT,
			rule_name varchar(255) NOT NULL,
			rule_type varchar(50) NOT NULL,
			conditions longtext NOT NULL,
			actions longtext NOT NULL,
			is_active boolean DEFAULT 1 NOT NULL,
			PRIMARY KEY  (id)
		) $charset_collate;";
		dbDelta( $sql );

		// Table: wp_abandoned_carts
		$table_name = $wpdb->prefix . 'abandoned_carts';
		$sql = "CREATE TABLE $table_name (
			id mediumint(9) NOT NULL AUTO_INCREMENT,
			user_email varchar(100) NOT NULL,
			cart_contents longtext NOT NULL,
			cart_total float NOT NULL,
			created_at datetime DEFAULT '0000-00-00 00:00:00' NOT NULL,
			status varchar(20) DEFAULT 'abandoned' NOT NULL,
			recovered_at datetime DEFAULT NULL,
			PRIMARY KEY  (id)
		) $charset_collate;";
		dbDelta( $sql );

		// Table: wp_ai_email_log
		$table_name = $wpdb->prefix . 'ai_email_log';
		$sql = "CREATE TABLE $table_name (
			id mediumint(9) NOT NULL AUTO_INCREMENT,
			email_to varchar(100) NOT NULL,
			email_subject varchar(255) NOT NULL,
			email_type varchar(50) NOT NULL,
			sent_at datetime DEFAULT '0000-00-00 00:00:00' NOT NULL,
			status varchar(20) NOT NULL,
			PRIMARY KEY  (id)
		) $charset_collate;";
		dbDelta( $sql );

		// Table: wp_ab_tests
		$table_name = $wpdb->prefix . 'ab_tests';
		$sql = "CREATE TABLE $table_name (
			id mediumint(9) NOT NULL AUTO_INCREMENT,
			product_id bigint(20) NOT NULL,
			test_type varchar(50) NOT NULL,
			variant_a longtext NOT NULL,
			variant_b longtext NOT NULL,
			traffic_split int DEFAULT 50 NOT NULL,
			start_date datetime DEFAULT '0000-00-00 00:00:00' NOT NULL,
			end_date datetime DEFAULT NULL,
			status varchar(20) DEFAULT 'running' NOT NULL,
			PRIMARY KEY  (id)
		) $charset_collate;";
		dbDelta( $sql );
		
		// Table: wp_ai_events (for analytics)
		$table_name = $wpdb->prefix . 'ai_events';
		$sql = "CREATE TABLE $table_name (
			id mediumint(9) NOT NULL AUTO_INCREMENT,
			event_type varchar(50) NOT NULL,
			event_data longtext NOT NULL,
			created_at datetime DEFAULT '0000-00-00 00:00:00' NOT NULL,
			user_id bigint(20) DEFAULT 0,
			PRIMARY KEY  (id)
		) $charset_collate;";
		dbDelta( $sql );

		// Table: wp_email_flows
		$table_name = $wpdb->prefix . 'email_flows';
		$sql = "CREATE TABLE $table_name (
			id mediumint(9) NOT NULL AUTO_INCREMENT,
			flow_name varchar(255) NOT NULL,
			flow_slug varchar(100) NOT NULL,
			trigger_type varchar(50) NOT NULL,
			ai_tone varchar(50) DEFAULT 'Friendly',
			is_active boolean DEFAULT 1 NOT NULL,
			created_at datetime DEFAULT '0000-00-00 00:00:00' NOT NULL,
			PRIMARY KEY  (id),
			UNIQUE KEY flow_slug (flow_slug)
		) $charset_collate;";
		dbDelta( $sql );

		// Table: wp_ai_licenses (PRO)
		$table_name = $wpdb->prefix . 'ai_licenses';
		$sql = "CREATE TABLE $table_name (
			id mediumint(9) NOT NULL AUTO_INCREMENT,
			license_key varchar(255) NOT NULL,
			status varchar(20) NOT NULL,
			activated_at datetime DEFAULT NULL,
			expires_at datetime DEFAULT NULL,
			PRIMARY KEY  (id)
		) $charset_collate;";
		dbDelta( $sql );
	}
}

/**
 * Check PHP, WordPress and WooCommerce requirements.
 *
 * @return bool
 */
function ai_growth_suite_check_requirements() {
	if ( version_compare( PHP_VERSION, '7.4', '<' ) ) {
		add_action( 'admin_notices', function () {
			echo '<div class="notice notice-error"><p><strong>AI eCommerce Growth Suite</strong> requires PHP 7.4 or higher.</p></div>';
		} );
		return false;
	}
	if ( ! class_exists( 'WooCommerce' ) ) {
		add_action( 'admin_notices', function () {
			echo '<div class="notice notice-warning"><p><strong>AI eCommerce Growth Suite</strong> requires WooCommerce. Please install and activate WooCommerce to use this plugin.</p></div>';
		} );
		return false;
	}
	return true;
}

/**
 * Initialize the plugin
 */
function ai_growth_suite_init() {
	if ( ! ai_growth_suite_check_requirements() ) {
		return;
	}
	return AI_Growth_Suite::get_instance();
}
add_action( 'plugins_loaded', 'ai_growth_suite_init', 5 );

// Add custom cron interval
add_filter( 'cron_schedules', 'ai_growth_suite_add_cron_interval' );
function ai_growth_suite_add_cron_interval( $schedules ) { 
	$schedules['fifteen_minutes'] = array(
		'interval' => 900,
		'display'  => esc_html__( 'Every Fifteen Minutes' ), 
	);
	return $schedules;
}

/**
 * Check if a plugin table exists (fallback when create_tables may have failed).
 *
 * @param string $short_name Table name without prefix (e.g. 'abandoned_carts', 'ab_tests', 'ai_events').
 * @return bool
 */
function ai_growth_suite_table_exists( $short_name ) {
	global $wpdb;
	$table = $wpdb->prefix . $short_name;
	return ( $wpdb->get_var( $wpdb->prepare( 'SHOW TABLES LIKE %s', $table ) ) === $table );
}
