jQuery(document).ready(function ($) {

    // Helper: Show Toast Notification
    function showToast(message, type = 'success') {
        var $container = $('.ai-toast-container');
        if ($container.length === 0) {
            $container = $('<div class="ai-toast-container"></div>');
            $('body').append($container);
        }

        var icon = type === 'success' ? 'dashicons-yes-alt' : 'dashicons-warning';
        var $toast = $('<div class="ai-toast ' + type + '"><span class="dashicons ' + icon + '"></span> ' + message + '</div>');

        $container.append($toast);

        // Auto remove after 4 seconds
        setTimeout(function () {
            $toast.css('animation', 'fadeOut 0.3s ease-out forwards');
            setTimeout(function () { $toast.remove(); }, 300);
        }, 4000);
    }

    // Helper: Button Loading State
    function setButtonLoading($btn, isLoading, loadingText = 'Processing...') {
        if (isLoading) {
            $btn.data('original-text', $btn.text());
            $btn.prop('disabled', true).html('<span class="ai-spinner"></span> ' + loadingText);
        } else {
            $btn.prop('disabled', false).html($btn.data('original-text'));
        }
    }

    // License Activation (License page)
    $(document).on('click', '#ai_growth_suite_activate_license', function (e) {
        e.preventDefault();
        var key = $('#ai_growth_suite_license_key').val();
        var nonce = $('#ai_growth_suite_license_nonce').val();
        var $btn = $(this);

        setButtonLoading($btn, true, 'Activating...');

        $.post(aiGrowthSuite.ajaxurl, {
            action: 'ai_growth_suite_activate_license',
            license_key: key,
            nonce: nonce
        }, function (response) {
            setButtonLoading($btn, false);
            if (response.success) {
                showToast(response.data, 'success');
                setTimeout(function () { location.reload(); }, 1500);
            } else {
                showToast(response.data, 'error');
            }
        });
    });

    // Product AI: Single Generator
    $('.ai-generate-btn').on('click', function (e) {
        e.preventDefault();
        var $btn = $(this);
        var productId = $btn.data('id');

        // Custom loading state for this button to change text to "Generating..."
        $btn.data('original-text', 'Regenerate'); // Set future text
        $btn.prop('disabled', true).html('<span class="ai-spinner"></span> Generating...');

        $.ajax({
            url: aiGrowthSuite.root + 'ai-growth-suite/v1/generate-description',
            method: 'POST',
            beforeSend: function (xhr) {
                xhr.setRequestHeader('X-WP-Nonce', aiGrowthSuite.nonce);
            },
            data: {
                product_id: productId
            },
            success: function (response) {
                $btn.prop('disabled', false).html('Regenerate');
                // Update status UI
                var $row = $btn.closest('tr');
                // Update the status column (3rd column, index 2)
                $row.find('td.ai-status').html('<span class="ags-status ags-status-ok"><span class="dashicons dashicons-yes-alt"></span> Optimized</span>');

                showToast('Content generated successfully!', 'success');
            },
            error: function (xhr) {
                $btn.prop('disabled', false).html('Retry');
                var msg = xhr.responseJSON && xhr.responseJSON.message ? xhr.responseJSON.message : 'Error generating content.';
                showToast(msg, 'error');
            }
        });
    });

    // Product AI: Generate All Descriptions
    $('.ai-growth-suite-product-ai .button-primary:contains("Generate All Descriptions")').on('click', function (e) {
        e.preventDefault();
        var $btn = $(this);
        var $buttons = $('.ai-generate-btn');
        var total = $buttons.length;
        var processed = 0;

        if (total === 0) {
            showToast('No products found to generate.', 'error');
            return;
        }

        if (!confirm('This will generate content for ' + total + ' products on this page. This may take a while. Continue?')) {
            return;
        }

        // Start Batch Loading State
        $btn.data('original-text', $btn.text());
        $btn.prop('disabled', true).html('<span class="ai-spinner"></span> Processing 0/' + total + '...');

        // Process sequentially
        function processNext(index) {
            if (index >= total) {
                $btn.prop('disabled', false).html($btn.data('original-text'));
                showToast('Batch generation completed successfully!', 'success');
                return;
            }

            var $currentBtn = $buttons.eq(index);
            var productId = $currentBtn.data('id');

            // Show mini-loader on individual button too
            $currentBtn.prop('disabled', true).html('<span class="ai-spinner"></span>');

            $.ajax({
                url: aiGrowthSuite.root + 'ai-growth-suite/v1/generate-description',
                method: 'POST',
                beforeSend: function (xhr) {
                    xhr.setRequestHeader('X-WP-Nonce', aiGrowthSuite.nonce);
                },
                data: { product_id: productId },
                success: function () {
                    $currentBtn.prop('disabled', false).html('Regenerate');
                    $currentBtn.closest('tr').find('td.ai-status').html('<span class="ags-status ags-status-ok"><span class="dashicons dashicons-yes-alt"></span> Optimized</span>');
                },
                error: function () {
                    $currentBtn.prop('disabled', false).html('Retry');
                },
                complete: function () {
                    processed++;
                    $btn.html('<span class="ai-spinner"></span> Processing ' + processed + '/' + total + '...');
                    processNext(index + 1);
                }
            });
        }

        processNext(0);
    });

    // Product AI: Select All checkbox
    $(document).on('change', '.ags-product-ai .ags-check-all', function () {
        var checked = $(this).prop('checked');
        $(this).closest('table').find('tbody input[name="product_ids[]"]').prop('checked', checked);
    });

    // Dynamic Pricing: Create Rule Toggle
    $('#add-new-rule-btn').on('click', function () {
        $('#ai-rule-id').val(''); // Clear ID for new rule
        $('#ai-rule-name').val('');
        $('#ai-rule-amount').val('');
        $('#create-rule-card').slideDown();
    });

    // ... (Upsell code) ...

    // Dynamic Pricing: Save Rule
    $('#ai-save-rule-btn').on('click', function (e) {
        e.preventDefault();
        var $btn = $(this);
        var id = $('#ai-rule-id').val();
        var name = $('#ai-rule-name').val();
        var condition = $('#ai-rule-condition').val();
        var action = $('#ai-rule-action-type').val();
        var amount = $('#ai-rule-amount').val();

        if (!name || !amount) {
            showToast('Please fill in all fields.', 'error');
            return;
        }

        setButtonLoading($btn, true, 'Saving...');

        $.ajax({
            url: aiGrowthSuite.root + 'ai-growth-suite/v1/create-rule',
            method: 'POST',
            beforeSend: function (xhr) {
                xhr.setRequestHeader('X-WP-Nonce', aiGrowthSuite.nonce);
            },
            data: {
                id: id,
                name: name,
                condition_type: condition,
                action_type: action,
                amount: amount
            },
            success: function (response) {
                setButtonLoading($btn, false);
                showToast('Rule saved successfully!', 'success');
                setTimeout(function () { location.reload(); }, 1000);
            },
            error: function () {
                setButtonLoading($btn, false);
                showToast('Error saving rule.', 'error');
            }
        });
    });

    // ... (Delete Rule code) ...

    // Upsell: Force Regenerate All
    $('.ai-regenerate-upsells-btn').on('click', function (e) {
        e.preventDefault();
        var $btn = $(this);

        if (!confirm('This will analyze ALL products in your store and generate new upsell recommendations. This process may take several minutes. Continue?')) {
            return;
        }

        // Start Loading
        setButtonLoading($btn, true, 'Fetching Products...');

        // Step 1: Get All Product IDs
        $.ajax({
            url: aiGrowthSuite.root + 'ai-growth-suite/v1/get-all-products',
            method: 'GET',
            beforeSend: function (xhr) {
                xhr.setRequestHeader('X-WP-Nonce', aiGrowthSuite.nonce);
            },
            success: function (productIds) {
                var total = productIds.length;
                var processed = 0;

                if (total === 0) {
                    showToast('No products found in store.', 'error');
                    setButtonLoading($btn, false);
                    return;
                }

                setButtonLoading($btn, true, 'Processing 0/' + total + '...');

                // Step 2: Process Each Product
                function processNextUpsell(index) {
                    if (index >= total) {
                        setButtonLoading($btn, false);
                        showToast('Upsell generation completed!', 'success');
                        setTimeout(function () { location.reload(); }, 1500);
                        return;
                    }

                    var p = productIds[index];
                    var pid = (p && typeof p === 'object' && p.id) ? p.id : p;

                    $.ajax({
                        url: aiGrowthSuite.root + 'ai-growth-suite/v1/generate-upsells',
                        method: 'POST',
                        beforeSend: function (xhr) {
                            xhr.setRequestHeader('X-WP-Nonce', aiGrowthSuite.nonce);
                        },
                        data: { product_id: pid },
                        complete: function () {
                            processed++;
                            setButtonLoading($btn, true, 'Processing ' + processed + '/' + total + '...');
                            processNextUpsell(index + 1);
                        }
                    });
                }

                processNextUpsell(0);
            },
            error: function () {
                showToast('Failed to fetch product list.', 'error');
                setButtonLoading($btn, false);
            }
        });
    });

    // Upsell: Disable Recommendation
    $('.ai-disable-upsell-btn').on('click', function (e) {
        e.preventDefault();
        if (!confirm('Are you sure you want to remove these recommendations?')) return;

        var $btn = $(this);
        var productId = $btn.data('id');
        var $row = $btn.closest('tr');

        $btn.prop('disabled', true).text('...');

        $.ajax({
            url: aiGrowthSuite.root + 'ai-growth-suite/v1/delete-recommendation',
            method: 'POST',
            beforeSend: function (xhr) {
                xhr.setRequestHeader('X-WP-Nonce', aiGrowthSuite.nonce);
            },
            data: { product_id: productId },
            success: function () {
                $row.fadeOut(function () { $(this).remove(); });
                showToast('Recommendation removed.', 'success');
            },
            error: function () {
                $btn.prop('disabled', false).text('Disable');
                showToast('Error removing recommendation.', 'error');
            }
        });
    });

    // Upsell: Edit Button (Redirect to Product Edit)
    $('.ai-edit-upsell-btn').on('click', function (e) {
        e.preventDefault();
        var productId = $(this).data('id');
        // Redirect to WooCommerce product edit page
        window.open('post.php?post=' + productId + '&action=edit', '_blank');
    });

    // Upsell: Search Functionality
    $('#ai-upsell-search-btn').on('click', function (e) {
        e.preventDefault();
        var term = $('#ai-upsell-search-input').val().toLowerCase();

        $('.ai-upsell-row').each(function () {
            var name = $(this).find('.product-name').text().toLowerCase();
            if (name.indexOf(term) > -1) {
                $(this).show();
            } else {
                $(this).hide();
            }
        });
    });

    // Upsell: Search on Enter key
    $('#ai-upsell-search-input').on('keyup', function (e) {
        if (e.key === 'Enter') {
            $('#ai-upsell-search-btn').click();
        }
    });


    // Dynamic Pricing: Delete Rule
    $('.ai-delete-rule-btn').on('click', function (e) {
        e.preventDefault();
        if (!confirm('Are you sure you want to delete this rule?')) return;

        var $btn = $(this);
        var ruleId = $btn.data('id');
        var $row = $btn.closest('tr');

        $btn.prop('disabled', true).text('...');

        $.ajax({
            url: aiGrowthSuite.root + 'ai-growth-suite/v1/delete-rule',
            method: 'POST',
            beforeSend: function (xhr) {
                xhr.setRequestHeader('X-WP-Nonce', aiGrowthSuite.nonce);
            },
            data: { rule_id: ruleId },
            success: function () {
                $row.fadeOut(function () { $(this).remove(); });
                showToast('Rule deleted.', 'success');
            },
            error: function () {
                $btn.prop('disabled', false).text('Delete');
                showToast('Error deleting rule.', 'error');
            }
        });
    });

    // Dynamic Pricing: Edit Rule (Populate Form) - single handler; populates id, name, condition, action, amount
    $('.ai-edit-rule-btn').on('click', function (e) {
        e.preventDefault();
        var $btn = $(this);
        $('#ai-rule-id').val($btn.data('id') || '');
        $('#ai-rule-name').val($btn.data('name'));
        $('#ai-rule-condition').val($btn.data('condition'));
        $('#ai-rule-action-type').val($btn.data('action'));
        $('#ai-rule-amount').val($btn.data('amount'));
        $('#create-rule-card').slideDown();
        $('html, body').animate({ scrollTop: $("#create-rule-card").offset().top - 100 }, 500);
        showToast('Rule details loaded. Modify and Save to update.', 'success');
    });

    // Marketing: Save Settings
    $('#ai-save-marketing-settings-btn').on('click', function (e) {
        e.preventDefault();
        var $btn = $(this);
        var tone = $('#ai-marketing-tone').val();

        setButtonLoading($btn, true, 'Saving...');

        $.ajax({
            url: aiGrowthSuite.root + 'ai-growth-suite/v1/save-marketing-settings',
            method: 'POST',
            beforeSend: function (xhr) {
                xhr.setRequestHeader('X-WP-Nonce', aiGrowthSuite.nonce);
            },
            data: { tone: tone },
            success: function () {
                setButtonLoading($btn, false);
                showToast('Settings saved successfully!', 'success');
            },
            error: function () {
                setButtonLoading($btn, false);
                showToast('Error saving settings.', 'error');
            }
        });
    });

    // Marketing: Edit Template Modal
    $('.ai-edit-template-btn').on('click', function (e) {
        e.preventDefault();
        var $btn = $(this);
        var flow = $btn.data('flow');

        // Reset and Show Modal
        $('#ai-template-flow').val(flow);
        $('#ai-template-subject').val('Loading...');
        $('#ai-template-body').val('Loading...');
        $('#ai-template-modal').css('display', 'flex');

        // Fetch Template Data
        $.ajax({
            url: aiGrowthSuite.root + 'ai-growth-suite/v1/get-email-template',
            method: 'GET',
            beforeSend: function (xhr) {
                xhr.setRequestHeader('X-WP-Nonce', aiGrowthSuite.nonce);
            },
            data: { flow: flow },
            success: function (response) {
                $('#ai-template-subject').val(response.subject);
                $('#ai-template-body').val(response.body);

                // Load automation settings
                $('#ai-template-auto-send').prop('checked', response.auto_send !== false);
                $('#ai-template-send-timing').val(response.send_timing || 'immediate');
                $('#ai-template-delay-amount').val(response.delay_amount || 1);
                $('#ai-template-specific-time').val(response.specific_time || '09:00');

                // Trigger timing change to show/hide fields
                $('#ai-template-send-timing').trigger('change');
                $('#ai-template-auto-send').trigger('change');
            },
            error: function () {
                showToast('Error fetching template.', 'error');
                $('#ai-template-modal').hide();
            }
        });
    });

    // Marketing: Template Timing Settings UI
    $('#ai-template-send-timing').on('change', function () {
        var timing = $(this).val();
        $('#ai-template-delay-amount-wrapper').hide();
        $('#ai-template-specific-time-wrapper').hide();

        if (timing === 'delay_minutes') {
            $('#ai-template-delay-amount-wrapper').show();
            $('#ai-template-delay-unit').text('minutes');
        } else if (timing === 'delay_hours') {
            $('#ai-template-delay-amount-wrapper').show();
            $('#ai-template-delay-unit').text('hours');
        } else if (timing === 'delay_days') {
            $('#ai-template-delay-amount-wrapper').show();
            $('#ai-template-delay-unit').text('days');
        } else if (timing === 'specific_time') {
            $('#ai-template-specific-time-wrapper').show();
        }
    });

    // Marketing: Auto-send checkbox toggle
    $('#ai-template-auto-send').on('change', function () {
        if ($(this).is(':checked')) {
            $('#ai-template-timing-settings').slideDown();
        } else {
            $('#ai-template-timing-settings').slideUp();
        }
    });

    // Marketing: Save Template
    $('#ai-save-template-btn').on('click', function (e) {
        e.preventDefault();
        var $btn = $(this);
        var flow = $('#ai-template-flow').val();
        var subject = $('#ai-template-subject').val();
        var body = $('#ai-template-body').val();
        var autoSend = $('#ai-template-auto-send').is(':checked');
        var sendTiming = $('#ai-template-send-timing').val();
        var delayAmount = $('#ai-template-delay-amount').val();
        var specificTime = $('#ai-template-specific-time').val();

        setButtonLoading($btn, true, 'Saving...');

        $.ajax({
            url: aiGrowthSuite.root + 'ai-growth-suite/v1/save-email-template',
            method: 'POST',
            beforeSend: function (xhr) {
                xhr.setRequestHeader('X-WP-Nonce', aiGrowthSuite.nonce);
            },
            data: {
                flow: flow,
                subject: subject,
                body: body,
                auto_send: autoSend,
                send_timing: sendTiming,
                delay_amount: delayAmount,
                specific_time: specificTime
            },
            success: function () {
                setButtonLoading($btn, false);
                showToast('Template and automation settings saved successfully!', 'success');
                setTimeout(function () { $('#ai-template-modal').hide(); }, 500);
            },
            error: function () {
                setButtonLoading($btn, false);
                showToast('Error saving template.', 'error');
            }
        });
    });

    // Marketing: Create Flow Modal
    $('#ai-create-flow-btn').on('click', function (e) {
        e.preventDefault();
        $('#ai-new-flow-name').val('');
        $('#ai-new-flow-trigger').val('');
        $('#ai-custom-trigger-text').val('');
        $('#ai-custom-trigger-wrapper').hide();
        $('#ai-new-flow-tone').val('Friendly');
        $('#ai-create-flow-modal').css('display', 'flex');
    });

    // Marketing: Handle Custom Trigger Selection
    $('#ai-new-flow-trigger').on('change', function () {
        if ($(this).val() === 'custom') {
            $('#ai-custom-trigger-wrapper').slideDown();
        } else {
            $('#ai-custom-trigger-wrapper').slideUp();
            $('#ai-custom-trigger-text').val('');
        }
    });

    // Marketing: Save New Flow
    $('#ai-save-flow-btn').on('click', function (e) {
        e.preventDefault();
        var $btn = $(this);
        var name = $('#ai-new-flow-name').val();
        var trigger = $('#ai-new-flow-trigger').val();
        var customTrigger = $('#ai-custom-trigger-text').val();
        var tone = $('#ai-new-flow-tone').val();

        // Use custom trigger text if "custom" is selected
        if (trigger === 'custom') {
            if (!customTrigger) {
                showToast('Please describe your custom trigger.', 'error');
                return;
            }
            trigger = customTrigger;
        }

        if (!name || !trigger) {
            showToast('Please fill in all required fields.', 'error');
            return;
        }

        setButtonLoading($btn, true, 'Creating...');

        $.ajax({
            url: aiGrowthSuite.root + 'ai-growth-suite/v1/create-flow',
            method: 'POST',
            beforeSend: function (xhr) {
                xhr.setRequestHeader('X-WP-Nonce', aiGrowthSuite.nonce);
            },
            data: {
                name: name,
                trigger: trigger,
                tone: tone
            },
            success: function () {
                setButtonLoading($btn, false);
                showToast('Flow created successfully!', 'success');
                setTimeout(function () { location.reload(); }, 1000);
            },
            error: function () {
                setButtonLoading($btn, false);
                showToast('Error creating flow.', 'error');
            }
        });
    });

    // Marketing: Delete Flow
    $('.ai-delete-flow-btn').on('click', function (e) {
        e.preventDefault();
        if (!confirm('Are you sure you want to delete this flow?')) return;

        var $btn = $(this);
        var flowId = $btn.data('id');
        var $row = $btn.closest('tr');

        $btn.prop('disabled', true).text('...');

        $.ajax({
            url: aiGrowthSuite.root + 'ai-growth-suite/v1/delete-flow',
            method: 'POST',
            beforeSend: function (xhr) {
                xhr.setRequestHeader('X-WP-Nonce', aiGrowthSuite.nonce);
            },
            data: { flow_id: flowId },
            success: function () {
                $row.fadeOut(function () { $(this).remove(); });
                showToast('Flow deleted.', 'success');
            },
            error: function () {
                $btn.prop('disabled', false).text('Delete');
                showToast('Error deleting flow.', 'error');
            }
        });
    });

    // Marketing: Toggle Flow Status
    $('.ai-toggle-flow-status').on('change', function () {
        var $checkbox = $(this);
        var flowId = $checkbox.data('id');
        var isActive = $checkbox.is(':checked');

        $.ajax({
            url: aiGrowthSuite.root + 'ai-growth-suite/v1/toggle-flow-status',
            method: 'POST',
            beforeSend: function (xhr) {
                xhr.setRequestHeader('X-WP-Nonce', aiGrowthSuite.nonce);
            },
            data: {
                flow_id: flowId,
                is_active: isActive ? 1 : 0
            },
            success: function () {
                var statusText = isActive ? 'Active' : 'Inactive';
                $checkbox.closest('td').find('span').last().text(statusText);
                showToast('Flow status updated.', 'success');
            },
            error: function () {
                // Revert checkbox on error
                $checkbox.prop('checked', !isActive);
                showToast('Error updating flow status.', 'error');
            }
        });
    });

    // Abandoned Carts: View Cart Details
    $('.ai-view-cart-btn').on('click', function () {
        var $btn = $(this);
        var cartId = $btn.data('id');
        var email = $btn.data('email');
        var total = $btn.data('total');
        var items = $btn.data('items');
        var date = $btn.data('date');
        var status = $btn.data('status');

        // Get currency symbol
        var currency = aiGrowthSuite.currency || '$';

        // Populate modal
        $('#ai-cart-email').text(email);
        $('#ai-cart-total').text(currency + parseFloat(total || 0).toFixed(2));
        $('#ai-cart-date').text(new Date(date).toLocaleString());
        $('#ai-cart-status').text(status.charAt(0).toUpperCase() + status.slice(1));

        // Parse and display cart items
        var cartItems = '';
        var calculatedTotal = 0; // Track total from active items only

        try {
            var itemsData = typeof items === 'string' ? JSON.parse(items) : items;

            // Handle different JSON structures
            var itemsArray = [];
            if (itemsData && typeof itemsData === 'object') {
                // Check if it has 'items' property (nested structure)
                if (itemsData.items && Array.isArray(itemsData.items)) {
                    itemsArray = itemsData.items;
                }
                // Check if it's already an array
                else if (Array.isArray(itemsData)) {
                    itemsArray = itemsData;
                }
                // Check if it's an object with numeric keys
                else {
                    itemsArray = Object.values(itemsData);
                }
            }

            if (itemsArray.length > 0) {
                $.each(itemsArray, function (index, item) {
                    var productName = item.name || item.product_name || 'Product';
                    var quantity = item.quantity || 1;
                    var price = parseFloat(item.price || item.line_total || item.total || 0);
                    var lineTotal = price * quantity;
                    var isRemoved = item.removed === true || item.removed === 'true';

                    // Add to total only if not removed
                    if (!isRemoved) {
                        calculatedTotal += lineTotal;
                    }

                    // Different styling for removed items
                    var rowStyle = isRemoved ? 'padding: 12px; border-bottom: 1px solid #e0e0e0; display: flex; justify-content: space-between; align-items: center; background: #fff3f3; opacity: 0.7;' : 'padding: 12px; border-bottom: 1px solid #e0e0e0; display: flex; justify-content: space-between; align-items: center;';

                    cartItems += '<div style="' + rowStyle + '">';
                    cartItems += '<div style="flex: 1;">';
                    cartItems += '<strong style="font-size: 14px; color: #333;">' + productName + '</strong>';

                    if (isRemoved) {
                        cartItems += ' <span style="background: #dc3232; color: white; padding: 2px 8px; border-radius: 3px; font-size: 11px; margin-left: 8px;">Removed from Cart</span>';
                    }

                    cartItems += '<br>';
                    cartItems += '<span style="color: #666; font-size: 13px;">Quantity: ' + quantity + ' × ' + currency + price.toFixed(2) + '</span>';
                    cartItems += '</div>';

                    if (!isRemoved) {
                        cartItems += '<div style="font-weight: bold; font-size: 15px; color: #0073aa;">' + currency + lineTotal.toFixed(2) + '</div>';
                    } else {
                        cartItems += '<div style="font-weight: bold; font-size: 15px; color: #999; text-decoration: line-through;">' + currency + lineTotal.toFixed(2) + '</div>';
                    }

                    cartItems += '</div>';
                });

                // Update cart total with calculated total from active items
                $('#ai-cart-total').text(currency + calculatedTotal.toFixed(2));
            } else {
                cartItems = '<p style="color: #999; text-align: center; padding: 20px;">No items found in cart</p>';
            }
        } catch (e) {
            console.error('Error parsing cart items:', e);
            cartItems = '<p style="color: #d63638; text-align: center; padding: 20px;">Unable to load cart items<br><small>' + e.message + '</small></p>';
        }
        $('#ai-cart-items').html(cartItems);

        // Store cart ID for send button
        $('#ai-modal-send-recovery-btn').data('id', cartId).data('email', email);

        // Show modal
        $('#ai-cart-details-modal').css('display', 'flex');
    });

    // Abandoned Carts: Send Recovery Email (from table)
    $('.ai-send-recovery-btn').on('click', function (e) {
        e.preventDefault();
        var $btn = $(this);
        var cartId = $btn.data('id');
        var email = $btn.data('email');

        if (!confirm('Send recovery email to ' + email + '?')) {
            return;
        }

        setButtonLoading($btn, true, 'Sending...');

        $.ajax({
            url: aiGrowthSuite.root + 'ai-growth-suite/v1/send-recovery-email',
            method: 'POST',
            beforeSend: function (xhr) {
                xhr.setRequestHeader('X-WP-Nonce', aiGrowthSuite.nonce);
            },
            data: { cart_id: cartId },
            success: function () {
                setButtonLoading($btn, false);
                showToast('Recovery email sent successfully!', 'success');
                $btn.closest('tr').find('td:nth-child(5)').html('<span class="dashicons dashicons-email-alt" style="color: var(--ai-primary);"></span> Emailed');
                $btn.remove();
            },
            error: function () {
                setButtonLoading($btn, false);
                showToast('Error sending recovery email.', 'error');
            }
        });
    });

    // Abandoned Carts: Send Recovery Email (from modal)
    $('#ai-modal-send-recovery-btn').on('click', function () {
        var $btn = $(this);
        var cartId = $btn.data('id');
        var email = $btn.data('email');

        if (!confirm('Send recovery email to ' + email + '?')) {
            return;
        }

        setButtonLoading($btn, true, 'Sending...');

        $.ajax({
            url: aiGrowthSuite.root + 'ai-growth-suite/v1/send-recovery-email',
            method: 'POST',
            beforeSend: function (xhr) {
                xhr.setRequestHeader('X-WP-Nonce', aiGrowthSuite.nonce);
            },
            data: { cart_id: cartId },
            success: function () {
                setButtonLoading($btn, false);
                showToast('Recovery email sent successfully!', 'success');
                $('#ai-cart-details-modal').hide();
                location.reload();
            },
            error: function () {
                setButtonLoading($btn, false);
                showToast('Error sending recovery email.', 'error');
            }
        });
    });

    // Abandoned Carts: Search
    $('#ai-cart-search').on('keyup', function () {
        var searchTerm = $(this).val().toLowerCase();
        filterCarts();
    });

    // Abandoned Carts: Filter
    $('#ai-cart-filter').on('change', function () {
        filterCarts();
    });

    // Abandoned Carts: Refresh
    $('#ai-refresh-carts-btn').on('click', function () {
        location.reload();
    });

    // Filter function for abandoned carts
    function filterCarts() {
        var searchTerm = $('#ai-cart-search').val().toLowerCase();
        var statusFilter = $('#ai-cart-filter').val();

        $('.ai-cart-row').each(function () {
            var $row = $(this);
            var email = $row.data('email').toLowerCase();
            var status = $row.data('status');

            var matchesSearch = email.includes(searchTerm);
            var matchesFilter = (statusFilter === 'all' || status === statusFilter);

            if (matchesSearch && matchesFilter) {
                $row.show();
            } else {
                $row.hide();
            }
        });
    }

    // ========================================
    // A/B Testing Functionality
    // ========================================

    // A/B Testing: Create New Test Button
    $('.ai-growth-suite-ab-testing .button-primary:contains("Create New Test")').on('click', function (e) {
        e.preventDefault();

        // Create modal HTML
        var modalHtml = `
            <div id="ai-ab-test-modal" style="display: flex; position: fixed; z-index: 999999; left: 0; top: 0; width: 100%; height: 100%; background-color: rgba(0,0,0,0.5); align-items: center; justify-content: center;">
                <div style="background: white; padding: 30px; border-radius: 8px; max-width: 600px; width: 90%; max-height: 90vh; overflow-y: auto;">
                    <h2 style="margin-top: 0;">Create New A/B Test</h2>
                    
                    <div style="margin-bottom: 20px;">
                        <label style="display: block; margin-bottom: 5px; font-weight: 600;">Select Product</label>
                        <select id="ai-ab-product-id" style="width: 100%; padding: 8px; border: 1px solid #ddd; border-radius: 4px;">
                            <option value="">Loading products...</option>
                        </select>
                    </div>
                    
                    <div style="margin-bottom: 20px;">
                        <label style="display: block; margin-bottom: 5px; font-weight: 600;">Test Type</label>
                        <select id="ai-ab-test-type" style="width: 100%; padding: 8px; border: 1px solid #ddd; border-radius: 4px;">
                            <option value="title">Product Title</option>
                            <option value="description">Product Description</option>
                            <option value="price">Pricing Strategy</option>
                            <option value="image">Product Image</option>
                        </select>
                    </div>
                    
                    <div style="margin-bottom: 20px;">
                        <label style="display: block; margin-bottom: 5px; font-weight: 600;">Traffic Split (%)</label>
                        <input type="number" id="ai-ab-traffic-split" value="50" min="10" max="90" style="width: 100%; padding: 8px; border: 1px solid #ddd; border-radius: 4px;">
                        <small style="color: #666;">Percentage of visitors who will see Variant B (rest will see Variant A)</small>
                    </div>
                    
                    <div id="ai-ab-variant-fields" style="margin-bottom: 20px;">
                        <h3>Variant A (Control)</h3>
                        <textarea id="ai-ab-variant-a" rows="4" style="width: 100%; padding: 8px; border: 1px solid #ddd; border-radius: 4px;" placeholder="Original content (will be auto-filled from product)"></textarea>
                        
                        <h3>Variant B (Test)</h3>
                        <textarea id="ai-ab-variant-b" rows="4" style="width: 100%; padding: 8px; border: 1px solid #ddd; border-radius: 4px;" placeholder="Enter alternative content to test"></textarea>
                    </div>
                    
                    <div style="display: flex; gap: 10px; justify-content: flex-end;">
                        <button id="ai-ab-cancel-btn" class="button">Cancel</button>
                        <button id="ai-ab-save-test-btn" class="button button-primary">Create Test</button>
                    </div>
                </div>
            </div>
        `;

        // Remove existing modal if any
        $('#ai-ab-test-modal').remove();

        // Append modal to body
        $('body').append(modalHtml);

        // Load products
        $.ajax({
            url: aiGrowthSuite.root + 'ai-growth-suite/v1/get-all-products',
            method: 'GET',
            beforeSend: function (xhr) {
                xhr.setRequestHeader('X-WP-Nonce', aiGrowthSuite.nonce);
            },
            success: function (products) {
                var options = '<option value="">Select a product...</option>';
                if (products && products.length > 0) {
                    $.each(products, function (index, product) {
                        // Product object should have id and name
                        options += '<option value="' + product.id + '">' + product.name + '</option>';
                    });
                } else {
                    options = '<option value="">No products found</option>';
                }
                $('#ai-ab-product-id').html(options);
            },
            error: function (xhr, status, error) {
                console.error('Error loading products:', error);
                $('#ai-ab-product-id').html('<option value="">Error loading products</option>');
            }
        });
    });

    // A/B Testing: Close Modal
    $(document).on('click', '#ai-ab-cancel-btn', function () {
        $('#ai-ab-test-modal').remove();
    });

    // A/B Testing: Save Test
    $(document).on('click', '#ai-ab-save-test-btn', function (e) {
        e.preventDefault();
        var $btn = $(this);
        var productId = $('#ai-ab-product-id').val();
        var testType = $('#ai-ab-test-type').val();
        var trafficSplit = $('#ai-ab-traffic-split').val();
        var variantA = $('#ai-ab-variant-a').val();
        var variantB = $('#ai-ab-variant-b').val();

        if (!productId) {
            showToast('Please select a product', 'error');
            return;
        }

        if (!variantB) {
            showToast('Please enter content for Variant B', 'error');
            return;
        }

        setButtonLoading($btn, true, 'Creating Test...');

        $.ajax({
            url: aiGrowthSuite.root + 'ai-growth-suite/v1/create-ab-test',
            method: 'POST',
            contentType: 'application/json',
            beforeSend: function (xhr) {
                xhr.setRequestHeader('X-WP-Nonce', aiGrowthSuite.nonce);
            },
            data: JSON.stringify({
                product_id: productId,
                test_type: testType,
                variant_a: variantA || '',
                variant_b: variantB || '',
                traffic_split: parseInt(trafficSplit, 10) || 50
            }),
            success: function (response) {
                setButtonLoading($btn, false);
                showToast('A/B Test created successfully!', 'success');
                $('#ai-ab-test-modal').remove();
                setTimeout(function () { location.reload(); }, 1000);
            },
            error: function (xhr) {
                setButtonLoading($btn, false);
                var msg = 'Error creating test.';
                if (xhr.responseJSON) {
                    if (xhr.responseJSON.message) msg = xhr.responseJSON.message;
                    else if (xhr.responseJSON.code && xhr.responseJSON.message) msg = xhr.responseJSON.message;
                }
                showToast(msg, 'error');
            }
        });
    });

    // A/B Testing: View Report Button
    $(document).on('click', '.ai-growth-suite-ab-testing .button-small[data-test-id]', function (e) {
        e.preventDefault();
        var $btn = $(this);
        var testId = $btn.data('test-id');
        var productName = $btn.data('product-name') || '';
        var testType = ($btn.data('test-type') || '').toLowerCase();
        var testTypeLabel = testType ? testType.charAt(0).toUpperCase() + testType.slice(1) + ' Test' : 'Test';
        var testName = (productName ? productName + ' — ' : '') + testTypeLabel + (testId ? ' #' + testId : '');

        // Create report modal
        var reportHtml = `
            <div id="ai-ab-report-modal" style="display: flex; position: fixed; z-index: 999999; left: 0; top: 0; width: 100%; height: 100%; background-color: rgba(0,0,0,0.5); align-items: center; justify-content: center;">
                <div style="background: white; padding: 30px; border-radius: 8px; max-width: 800px; width: 90%; max-height: 90vh; overflow-y: auto;">
                    <h2 style="margin-top: 0;">A/B Test Report: ${testName || ('Test #' + testId)}</h2>
                    
                    <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 20px; margin-bottom: 30px;">
                        <div style="background: #f0f0f1; padding: 20px; border-radius: 8px;">
                            <h3 style="margin-top: 0;">Variant A (Control)</h3>
                            <div style="margin-bottom: 10px;">
                                <strong>Views:</strong> <span id="variant-a-views">Loading...</span>
                            </div>
                            <div style="margin-bottom: 10px;">
                                <strong>Conversions:</strong> <span id="variant-a-conversions">Loading...</span>
                            </div>
                            <div style="margin-bottom: 10px;">
                                <strong>Conversion Rate:</strong> <span id="variant-a-rate">Loading...</span>
                            </div>
                        </div>
                        
                        <div style="background: #f0f0f1; padding: 20px; border-radius: 8px;">
                            <h3 style="margin-top: 0;">Variant B (Test)</h3>
                            <div style="margin-bottom: 10px;">
                                <strong>Views:</strong> <span id="variant-b-views">Loading...</span>
                            </div>
                            <div style="margin-bottom: 10px;">
                                <strong>Conversions:</strong> <span id="variant-b-conversions">Loading...</span>
                            </div>
                            <div style="margin-bottom: 10px;">
                                <strong>Conversion Rate:</strong> <span id="variant-b-rate">Loading...</span>
                            </div>
                        </div>
                    </div>
                    
                    <div style="background: #e7f5ff; padding: 20px; border-radius: 8px; margin-bottom: 20px;">
                        <h3 style="margin-top: 0;">Performance Summary</h3>
                        <p id="performance-summary">Loading...</p>
                    </div>
                    
                    <div style="display: flex; gap: 10px; justify-content: flex-end;">
                        <button id="ai-ab-stop-test-btn" class="button">Stop Test</button>
                        <button id="ai-ab-close-report-btn" class="button button-primary">Close</button>
                    </div>
                </div>
            </div>
        `;

        $('#ai-ab-report-modal').remove();
        $('body').append(reportHtml);
        $('#ai-ab-stop-test-btn').data('test-id', testId);

        $.ajax({
            url: aiGrowthSuite.root + 'ai-growth-suite/v1/get-ab-test-report',
            method: 'GET',
            data: { test_id: testId },
            beforeSend: function (xhr) { xhr.setRequestHeader('X-WP-Nonce', aiGrowthSuite.nonce); },
            success: function (r) {
                var vA = parseInt(r.views_a, 10) || 0, vB = parseInt(r.views_b, 10) || 0;
                var cA = parseInt(r.conversions_a, 10) || 0, cB = parseInt(r.conversions_b, 10) || 0;
                var rateA = parseFloat(r.rate_a) || 0, rateB = parseFloat(r.rate_b) || 0;

                $('#variant-a-views').text(vA.toLocaleString());
                $('#variant-a-conversions').text(cA.toLocaleString());
                $('#variant-a-rate').text(rateA + '%');

                $('#variant-b-views').text(vB.toLocaleString());
                $('#variant-b-conversions').text(cB.toLocaleString());
                $('#variant-b-rate').text(rateB + '%');

                $('#performance-summary').text(r.summary || 'No summary available.');

                if (r.status && r.status !== 'running') {
                    $('#ai-ab-stop-test-btn').hide();
                }
            },
            error: function (xhr) {
                var msg = (xhr.responseJSON && xhr.responseJSON.message) ? xhr.responseJSON.message : 'Could not load report.';
                $('#variant-a-views').text('0');
                $('#variant-a-conversions').text('0');
                $('#variant-a-rate').text('0%');
                $('#variant-b-views').text('0');
                $('#variant-b-conversions').text('0');
                $('#variant-b-rate').text('0%');
                $('#performance-summary').text('No data yet. Start driving traffic to your product page to see results. ' + msg);
            }
        });
    });

    // A/B Testing: Close Report Modal
    $(document).on('click', '#ai-ab-close-report-btn', function () {
        $('#ai-ab-report-modal').remove();
    });

    // A/B Testing: Stop Test
    $(document).on('click', '#ai-ab-stop-test-btn', function () {
        if (!confirm('Are you sure you want to stop this test?')) return;
        var testId = $(this).data('test-id');
        var $modal = $('#ai-ab-report-modal');
        if (!testId) {
            showToast('Test stopped', 'success');
            $modal.remove();
            setTimeout(function () { location.reload(); }, 800);
            return;
        }
        $.ajax({
            url: aiGrowthSuite.root + 'ai-growth-suite/v1/stop-ab-test',
            method: 'POST',
            contentType: 'application/json',
            beforeSend: function (xhr) { xhr.setRequestHeader('X-WP-Nonce', aiGrowthSuite.nonce); },
            data: JSON.stringify({ test_id: testId }),
            success: function () {
                showToast('Test stopped successfully', 'success');
                $modal.remove();
                setTimeout(function () { location.reload(); }, 800);
            },
            error: function (xhr) {
                var msg = (xhr.responseJSON && xhr.responseJSON.message) ? xhr.responseJSON.message : 'Failed to stop test.';
                showToast(msg, 'error');
            }
        });
    });

    // Close modals when clicking outside
    $(document).on('click', '#ai-ab-test-modal, #ai-ab-report-modal, #ai-template-modal, #ai-cart-details-modal, #ai-create-flow-modal', function (e) {
        if (e.target.id === 'ai-ab-test-modal' || e.target.id === 'ai-ab-report-modal' ||
            e.target.id === 'ai-template-modal' || e.target.id === 'ai-cart-details-modal' ||
            e.target.id === 'ai-create-flow-modal') {
            $(this).hide();
        }
    });
});
